var q = require('q');

var testOut = {
  failedCount: 0, specResults: [{
    description: 'Console output',
    assertions: [],
    duration: 0
  }]
};

/**
 * This plugin checks the browser log after each test for warnings and errors.
 * It can be configured to fail a test if either is detected.  There is also an
 * optional exclude parameter which accepts both regex and strings.  Any log
 * matching the exclude parameter will not fail the test or be logged to the
 * console.
 *
 *    exports.config = {
 *      plugins: [{
 *        path: 'node_modules/protractor/plugins/console',
 *        failOnWarning: {Boolean}                (Default - false),
 *        failOnError: {Boolean}                  (Default - true)
 *        exclude: {Array of strings and regex}   (Default - [])
 *      }]
 *    };
 */
var ConsolePlugin = function() {
};

/**
 * Gets the browser log.
 *
 * @return {webdriver.promise.Promise.<!Array.<!webdriver.logging.Entry>>}
 */
ConsolePlugin.getBrowserLog = function() {
  return browser.manage().logs().get('browser');
};

/**
 * Logs messages to the test outputl
 *
 * @param {Object} warnings The list of warnings detected by the browser log.
 * @param {Object} errors The list of errors detected by the browser log.
 */
ConsolePlugin.logMessages = function(warnings, errors) {
  var passed = (testOut.failedCount === 0);
  warnings.map(function(warning) {
    testOut.specResults[0].assertions.push(
      {passed: passed, errorMsg: warning.level.name + ': ' + warning.message}
    );
  });

  errors.map(function(error) {
    testOut.specResults[0].assertions.push(
      {passed: passed, errorMsg: error.level.name + ': ' + error.message}
    );
  });
};

/**
 * Determines if a log message is filtered out or not. This can be set at the
 * config stage using the exclude parameter.  The parameter accepts both strings
 * and regex.
 *
 * @param {string} logMessage Current log message.
 * @return {boolean} true iff the log should be included in the output
 */
ConsolePlugin.includeLog = function(logMessage) {
  return ConsolePlugin.exclude.filter(function(e) {
      return (e instanceof RegExp) ? logMessage.match(e) :
          logMessage.indexOf(e) > -1;
  }).length === 0;
};

/**
 * Parses the log and decides whether to throw an error or not.
 *
 * @param {Object} config The plugin's config block
 * @return {!webdriver.promise.Promise.<R>} A promise which resolves when the
 *    logs have been gathered
 */
ConsolePlugin.parseLog = function(config) {
  var failOnWarning = (config.failOnWarning === undefined) ? false :
      config.failOnWarning;
  var failOnError = (config.failOnError === undefined) ? true :
      config.failOnError;
  ConsolePlugin.exclude = config.exclude || [];

  return ConsolePlugin.getBrowserLog().then(function(log) {

    var warnings = log.filter(function(node) {
      return (node.level || {}).name === 'WARNING' &&
          ConsolePlugin.includeLog(node.message);
    });

    var errors = log.filter(function(node) {
      return (node.level || {}).name === 'SEVERE' &&
          ConsolePlugin.includeLog(node.message);
    });

    testOut.failedCount += failOnWarning ? warnings.length : 0;
    testOut.failedCount += failOnError ? errors.length : 0;

    ConsolePlugin.logMessages(warnings, errors);
  });

};

/**
 * Gather the console logs and output them as test results.  See the
 * documentation of the teardown function in the protractor plugin API.
 *
 * @param {Object} config The plugin's config block
 * @return {!webdriver.promise.Promise.<Object>} A promise which resolves to the
 *    test results generated by the console logs
 */
ConsolePlugin.prototype.teardown = function(config) {
  return ConsolePlugin.parseLog(config).then(function() {
    return testOut;
  });
};

var consolePlugin = new ConsolePlugin();

exports.teardown = consolePlugin.teardown.bind(consolePlugin);
