"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frame = exports.FrameManager = void 0;
const EventEmitter = require("events");
const helper_1 = require("./helper");
const Events_1 = require("./Events");
const ExecutionContext_1 = require("./ExecutionContext");
const LifecycleWatcher_1 = require("./LifecycleWatcher");
const DOMWorld_1 = require("./DOMWorld");
const NetworkManager_1 = require("./NetworkManager");
const UTILITY_WORLD_NAME = '__puppeteer_utility_world__';
class FrameManager extends EventEmitter {
    constructor(client, page, ignoreHTTPSErrors, timeoutSettings) {
        super();
        this._frames = new Map();
        this._contextIdToContext = new Map();
        this._isolatedWorlds = new Set();
        this._client = client;
        this._page = page;
        this._networkManager = new NetworkManager_1.NetworkManager(client, ignoreHTTPSErrors, this);
        this._timeoutSettings = timeoutSettings;
        this._client.on('Page.frameAttached', (event) => this._onFrameAttached(event.frameId, event.parentFrameId));
        this._client.on('Page.frameNavigated', (event) => this._onFrameNavigated(event.frame));
        this._client.on('Page.navigatedWithinDocument', (event) => this._onFrameNavigatedWithinDocument(event.frameId, event.url));
        this._client.on('Page.frameDetached', (event) => this._onFrameDetached(event.frameId));
        this._client.on('Page.frameStoppedLoading', (event) => this._onFrameStoppedLoading(event.frameId));
        this._client.on('Runtime.executionContextCreated', (event) => this._onExecutionContextCreated(event.context));
        this._client.on('Runtime.executionContextDestroyed', (event) => this._onExecutionContextDestroyed(event.executionContextId));
        this._client.on('Runtime.executionContextsCleared', () => this._onExecutionContextsCleared());
        this._client.on('Page.lifecycleEvent', (event) => this._onLifecycleEvent(event));
    }
    async initialize() {
        const result = await Promise.all([
            this._client.send('Page.enable'),
            this._client.send('Page.getFrameTree'),
        ]);
        const { frameTree } = result[1];
        this._handleFrameTree(frameTree);
        await Promise.all([
            this._client.send('Page.setLifecycleEventsEnabled', { enabled: true }),
            this._client
                .send('Runtime.enable', {})
                .then(() => this._ensureIsolatedWorld(UTILITY_WORLD_NAME)),
            this._networkManager.initialize(),
        ]);
    }
    networkManager() {
        return this._networkManager;
    }
    async navigateFrame(frame, url, options = {}) {
        assertNoLegacyNavigationOptions(options);
        const { referer = this._networkManager.extraHTTPHeaders()['referer'], waitUntil = ['load'], timeout = this._timeoutSettings.navigationTimeout(), } = options;
        const watcher = new LifecycleWatcher_1.LifecycleWatcher(this, frame, waitUntil, timeout);
        let ensureNewDocumentNavigation = false;
        let error = await Promise.race([
            navigate(this._client, url, referer, frame._id),
            watcher.timeoutOrTerminationPromise(),
        ]);
        if (!error) {
            error = await Promise.race([
                watcher.timeoutOrTerminationPromise(),
                ensureNewDocumentNavigation
                    ? watcher.newDocumentNavigationPromise()
                    : watcher.sameDocumentNavigationPromise(),
            ]);
        }
        watcher.dispose();
        if (error)
            throw error;
        return watcher.navigationResponse();
        async function navigate(client, url, referrer, frameId) {
            try {
                const response = await client.send('Page.navigate', {
                    url,
                    referrer,
                    frameId,
                });
                ensureNewDocumentNavigation = !!response.loaderId;
                return response.errorText
                    ? new Error(`${response.errorText} at ${url}`)
                    : null;
            }
            catch (error) {
                return error;
            }
        }
    }
    async waitForFrameNavigation(frame, options = {}) {
        assertNoLegacyNavigationOptions(options);
        const { waitUntil = ['load'], timeout = this._timeoutSettings.navigationTimeout(), } = options;
        const watcher = new LifecycleWatcher_1.LifecycleWatcher(this, frame, waitUntil, timeout);
        const error = await Promise.race([
            watcher.timeoutOrTerminationPromise(),
            watcher.sameDocumentNavigationPromise(),
            watcher.newDocumentNavigationPromise(),
        ]);
        watcher.dispose();
        if (error)
            throw error;
        return watcher.navigationResponse();
    }
    _onLifecycleEvent(event) {
        const frame = this._frames.get(event.frameId);
        if (!frame)
            return;
        frame._onLifecycleEvent(event.loaderId, event.name);
        this.emit(Events_1.Events.FrameManager.LifecycleEvent, frame);
    }
    _onFrameStoppedLoading(frameId) {
        const frame = this._frames.get(frameId);
        if (!frame)
            return;
        frame._onLoadingStopped();
        this.emit(Events_1.Events.FrameManager.LifecycleEvent, frame);
    }
    _handleFrameTree(frameTree) {
        if (frameTree.frame.parentId)
            this._onFrameAttached(frameTree.frame.id, frameTree.frame.parentId);
        this._onFrameNavigated(frameTree.frame);
        if (!frameTree.childFrames)
            return;
        for (const child of frameTree.childFrames)
            this._handleFrameTree(child);
    }
    page() {
        return this._page;
    }
    mainFrame() {
        return this._mainFrame;
    }
    frames() {
        return Array.from(this._frames.values());
    }
    frame(frameId) {
        return this._frames.get(frameId) || null;
    }
    _onFrameAttached(frameId, parentFrameId) {
        if (this._frames.has(frameId))
            return;
        helper_1.assert(parentFrameId);
        const parentFrame = this._frames.get(parentFrameId);
        const frame = new Frame(this, this._client, parentFrame, frameId);
        this._frames.set(frame._id, frame);
        this.emit(Events_1.Events.FrameManager.FrameAttached, frame);
    }
    _onFrameNavigated(framePayload) {
        const isMainFrame = !framePayload.parentId;
        let frame = isMainFrame
            ? this._mainFrame
            : this._frames.get(framePayload.id);
        helper_1.assert(isMainFrame || frame, 'We either navigate top level or have old version of the navigated frame');
        // Detach all child frames first.
        if (frame) {
            for (const child of frame.childFrames())
                this._removeFramesRecursively(child);
        }
        // Update or create main frame.
        if (isMainFrame) {
            if (frame) {
                // Update frame id to retain frame identity on cross-process navigation.
                this._frames.delete(frame._id);
                frame._id = framePayload.id;
            }
            else {
                // Initial main frame navigation.
                frame = new Frame(this, this._client, null, framePayload.id);
            }
            this._frames.set(framePayload.id, frame);
            this._mainFrame = frame;
        }
        // Update frame payload.
        frame._navigated(framePayload);
        this.emit(Events_1.Events.FrameManager.FrameNavigated, frame);
    }
    async _ensureIsolatedWorld(name) {
        if (this._isolatedWorlds.has(name))
            return;
        this._isolatedWorlds.add(name);
        await this._client.send('Page.addScriptToEvaluateOnNewDocument', {
            source: `//# sourceURL=${ExecutionContext_1.EVALUATION_SCRIPT_URL}`,
            worldName: name,
        }),
            await Promise.all(this.frames().map((frame) => this._client
                .send('Page.createIsolatedWorld', {
                frameId: frame._id,
                grantUniveralAccess: true,
                worldName: name,
            })
                .catch(helper_1.debugError))); // frames might be removed before we send this
    }
    _onFrameNavigatedWithinDocument(frameId, url) {
        const frame = this._frames.get(frameId);
        if (!frame)
            return;
        frame._navigatedWithinDocument(url);
        this.emit(Events_1.Events.FrameManager.FrameNavigatedWithinDocument, frame);
        this.emit(Events_1.Events.FrameManager.FrameNavigated, frame);
    }
    _onFrameDetached(frameId) {
        const frame = this._frames.get(frameId);
        if (frame)
            this._removeFramesRecursively(frame);
    }
    _onExecutionContextCreated(contextPayload) {
        const auxData = contextPayload.auxData;
        const frameId = auxData ? auxData.frameId : null;
        const frame = this._frames.get(frameId) || null;
        let world = null;
        if (frame) {
            if (contextPayload.auxData && !!contextPayload.auxData['isDefault']) {
                world = frame._mainWorld;
            }
            else if (contextPayload.name === UTILITY_WORLD_NAME &&
                !frame._secondaryWorld._hasContext()) {
                // In case of multiple sessions to the same target, there's a race between
                // connections so we might end up creating multiple isolated worlds.
                // We can use either.
                world = frame._secondaryWorld;
            }
        }
        if (contextPayload.auxData && contextPayload.auxData['type'] === 'isolated')
            this._isolatedWorlds.add(contextPayload.name);
        const context = new ExecutionContext_1.ExecutionContext(this._client, contextPayload, world);
        if (world)
            world._setContext(context);
        this._contextIdToContext.set(contextPayload.id, context);
    }
    /**
     * @param {number} executionContextId
     */
    _onExecutionContextDestroyed(executionContextId) {
        const context = this._contextIdToContext.get(executionContextId);
        if (!context)
            return;
        this._contextIdToContext.delete(executionContextId);
        if (context._world)
            context._world._setContext(null);
    }
    _onExecutionContextsCleared() {
        for (const context of this._contextIdToContext.values()) {
            if (context._world)
                context._world._setContext(null);
        }
        this._contextIdToContext.clear();
    }
    executionContextById(contextId) {
        const context = this._contextIdToContext.get(contextId);
        helper_1.assert(context, 'INTERNAL ERROR: missing context with id = ' + contextId);
        return context;
    }
    _removeFramesRecursively(frame) {
        for (const child of frame.childFrames())
            this._removeFramesRecursively(child);
        frame._detach();
        this._frames.delete(frame._id);
        this.emit(Events_1.Events.FrameManager.FrameDetached, frame);
    }
}
exports.FrameManager = FrameManager;
class Frame {
    constructor(frameManager, client, parentFrame, frameId) {
        this._url = '';
        this._detached = false;
        this._loaderId = '';
        this._lifecycleEvents = new Set();
        this._frameManager = frameManager;
        this._client = client;
        this._parentFrame = parentFrame;
        this._url = '';
        this._id = frameId;
        this._detached = false;
        this._loaderId = '';
        this._mainWorld = new DOMWorld_1.DOMWorld(frameManager, this, frameManager._timeoutSettings);
        this._secondaryWorld = new DOMWorld_1.DOMWorld(frameManager, this, frameManager._timeoutSettings);
        this._childFrames = new Set();
        if (this._parentFrame)
            this._parentFrame._childFrames.add(this);
    }
    async goto(url, options) {
        return await this._frameManager.navigateFrame(this, url, options);
    }
    async waitForNavigation(options) {
        return await this._frameManager.waitForFrameNavigation(this, options);
    }
    executionContext() {
        return this._mainWorld.executionContext();
    }
    async evaluateHandle(pageFunction, ...args) {
        return this._mainWorld.evaluateHandle(pageFunction, ...args);
    }
    async evaluate(pageFunction, ...args) {
        return this._mainWorld.evaluate(pageFunction, ...args);
    }
    async $(selector) {
        return this._mainWorld.$(selector);
    }
    async $x(expression) {
        return this._mainWorld.$x(expression);
    }
    async $eval(selector, pageFunction, ...args) {
        return this._mainWorld.$eval(selector, pageFunction, ...args);
    }
    async $$eval(selector, pageFunction, ...args) {
        return this._mainWorld.$$eval(selector, pageFunction, ...args);
    }
    async $$(selector) {
        return this._mainWorld.$$(selector);
    }
    async content() {
        return this._secondaryWorld.content();
    }
    async setContent(html, options = {}) {
        return this._secondaryWorld.setContent(html, options);
    }
    name() {
        return this._name || '';
    }
    url() {
        return this._url;
    }
    parentFrame() {
        return this._parentFrame;
    }
    childFrames() {
        return Array.from(this._childFrames);
    }
    isDetached() {
        return this._detached;
    }
    async addScriptTag(options) {
        return this._mainWorld.addScriptTag(options);
    }
    async addStyleTag(options) {
        return this._mainWorld.addStyleTag(options);
    }
    async click(selector, options) {
        return this._secondaryWorld.click(selector, options);
    }
    async focus(selector) {
        return this._secondaryWorld.focus(selector);
    }
    async hover(selector) {
        return this._secondaryWorld.hover(selector);
    }
    select(selector, ...values) {
        return this._secondaryWorld.select(selector, ...values);
    }
    async tap(selector) {
        return this._secondaryWorld.tap(selector);
    }
    async type(selector, text, options) {
        return this._mainWorld.type(selector, text, options);
    }
    waitFor(selectorOrFunctionOrTimeout, options = {}, ...args) {
        const xPathPattern = '//';
        if (helper_1.helper.isString(selectorOrFunctionOrTimeout)) {
            const string = selectorOrFunctionOrTimeout;
            if (string.startsWith(xPathPattern))
                return this.waitForXPath(string, options);
            return this.waitForSelector(string, options);
        }
        if (helper_1.helper.isNumber(selectorOrFunctionOrTimeout))
            return new Promise((fulfill) => setTimeout(fulfill, selectorOrFunctionOrTimeout));
        if (typeof selectorOrFunctionOrTimeout === 'function')
            return this.waitForFunction(selectorOrFunctionOrTimeout, options, ...args);
        return Promise.reject(new Error('Unsupported target type: ' + typeof selectorOrFunctionOrTimeout));
    }
    async waitForSelector(selector, options) {
        const handle = await this._secondaryWorld.waitForSelector(selector, options);
        if (!handle)
            return null;
        const mainExecutionContext = await this._mainWorld.executionContext();
        const result = await mainExecutionContext._adoptElementHandle(handle);
        await handle.dispose();
        return result;
    }
    async waitForXPath(xpath, options) {
        const handle = await this._secondaryWorld.waitForXPath(xpath, options);
        if (!handle)
            return null;
        const mainExecutionContext = await this._mainWorld.executionContext();
        const result = await mainExecutionContext._adoptElementHandle(handle);
        await handle.dispose();
        return result;
    }
    waitForFunction(pageFunction, options = {}, ...args) {
        return this._mainWorld.waitForFunction(pageFunction, options, ...args);
    }
    async title() {
        return this._secondaryWorld.title();
    }
    _navigated(framePayload) {
        this._name = framePayload.name;
        this._url = framePayload.url;
    }
    _navigatedWithinDocument(url) {
        this._url = url;
    }
    _onLifecycleEvent(loaderId, name) {
        if (name === 'init') {
            this._loaderId = loaderId;
            this._lifecycleEvents.clear();
        }
        this._lifecycleEvents.add(name);
    }
    _onLoadingStopped() {
        this._lifecycleEvents.add('DOMContentLoaded');
        this._lifecycleEvents.add('load');
    }
    _detach() {
        this._detached = true;
        this._mainWorld._detach();
        this._secondaryWorld._detach();
        if (this._parentFrame)
            this._parentFrame._childFrames.delete(this);
        this._parentFrame = null;
    }
}
exports.Frame = Frame;
function assertNoLegacyNavigationOptions(options) {
    helper_1.assert(options['networkIdleTimeout'] === undefined, 'ERROR: networkIdleTimeout option is no longer supported.');
    helper_1.assert(options['networkIdleInflight'] === undefined, 'ERROR: networkIdleInflight option is no longer supported.');
    helper_1.assert(options.waitUntil !== 'networkidle', 'ERROR: "networkidle" option is no longer supported. Use "networkidle2" instead');
}
