var QRCode = require('./core/qrcode')
var PngRenderer = require('./renderer/png')
var Utf8Renderer = require('./renderer/utf8')
var TerminalRenderer = require('./renderer/terminal')
var SvgRenderer = require('./renderer/svg')

function checkParams (text, opts, cb) {
  if (typeof text === 'undefined') {
    throw new Error('String required as first argument')
  }

  if (typeof cb === 'undefined') {
    cb = opts
    opts = {}
  }

  if (typeof cb !== 'function') {
    throw new Error('Callback required as last argument')
  }

  return {
    opts: opts,
    cb: cb
  }
}

function getTypeFromFilename (path) {
  return path.slice((path.lastIndexOf('.') - 1 >>> 0) + 2).toLowerCase()
}

function getRendererFromType (type) {
  switch (type) {
    case 'svg':
      return SvgRenderer

    case 'txt':
    case 'utf8':
      return Utf8Renderer

    case 'png':
    case 'image/png':
    default:
      return PngRenderer
  }
}

function getStringRendererFromType (type) {
  switch (type) {
    case 'svg':
      return SvgRenderer

    case 'terminal':
      return TerminalRenderer

    case 'utf8':
    default:
      return Utf8Renderer
  }
}

function render (renderFunc, text, params) {
  try {
    var data = QRCode.create(text, params.opts)
    return renderFunc(data, params.opts, params.cb)
  } catch (e) {
    params.cb(e)
  }
}

exports.create = QRCode.create

exports.toCanvas = require('./browser').toCanvas

exports.toString = function toString (text, opts, cb) {
  var params = checkParams(text, opts, cb)
  var renderer = getStringRendererFromType(params.opts.type)
  var string = render(renderer.render, text, params)
  if (string) params.cb(null, string)
}

exports.toDataURL = function toDataURL (text, opts, cb) {
  var params = checkParams(text, opts, cb)
  var renderer = getRendererFromType(params.opts.type)

  render(renderer.renderToDataURL, text, params)
}

exports.toFile = function toFile (path, text, opts, cb) {
  if (arguments.length < 3) {
    throw new Error('Too few arguments provided')
  }

  var params = checkParams(text, opts, cb)
  var type = params.opts.type || getTypeFromFilename(path)
  var renderer = getRendererFromType(type)
  var renderToFile = renderer.renderToFile.bind(null, path)

  render(renderToFile, text, params)
}

exports.toFileStream = function toFileStream (stream, text, opts) {
  if (arguments.length < 2) {
    throw new Error('Too few arguments provided')
  }

  var params = checkParams(text, opts, stream.emit.bind(stream, 'error'))
  var renderer = getRendererFromType('png') // Only png support for now
  var renderToFileStream = renderer.renderToFileStream.bind(null, stream)
  render(renderToFileStream, text, params)
}

/**
 * Legacy API
 */
exports.drawText = exports.toString
exports.drawSvg = function (text, opts, cb) {
  if (!opts) opts = {}
  opts.type = 'svg'

  return exports.toString(text, opts, cb)
}

exports.drawPNGStream = exports.toFileStream
exports.save = exports.toFile
exports.toDataURI = exports.toDataURL
