import { UseQueryOptions } from '@tanstack/react-query';
import { HintedString } from '../types';
/**
 * Checks whether users can access the provided resources.
 *
 * `useCanAccessResources` returns an object describing the state of the request:
 *
 * - start: { isPending: true }
 * - success: { canAccess: Object<string, boolean>, isPending: false }
 * - error: { error: [error from provider], isPending: false }
 *
 * @param {Object} params Any params you want to pass to the authProvider
 * @param {string} params.action The action to check access for
 * @param {string[]} params.resources The list of resources to check access for
 * @param {Object} params.record Optional. The record to check access for
 *
 * @returns Return the react-query result and a canAccess property which is a map of the resources and their access status { [resource: string]: boolean }
 *
 * @example
 * import { useCanAccessResources } from 'react-admin';
 *
 * const UserList = ({ record }) => {
 *     const { isPending, canAccess } = useCanAccessResources({
 *         action: 'read',
 *         resources: ['users.id', 'users.name', 'users.email'],
 *         record
 *     });
 *
 *     if (isPending) {
 *         return null;
 *     }
 *     return (
 *         <SimpleList
 *              primaryText={record => canAccess.users.name ? record.name : ''}
 *              secondaryText={record => canAccess.users.email ? record.email : ''}
 *              tertiaryText={record => canAccess.users.id ? record.id : ''}
 *          />
 *     );
 * };
 */
export declare const useCanAccessResources: <RecordType extends Record<string, any> = Record<string, any>, ErrorType extends Error = Error>(params: UseCanAccessResourcesOptions<RecordType, ErrorType>) => UseCanAccessResourcesResult<ErrorType>;
export interface UseCanAccessResourcesOptions<RecordType extends Record<string, any> = Record<string, any>, ErrorType extends Error = Error> extends Omit<UseQueryOptions<boolean, ErrorType>, 'queryKey' | 'queryFn'> {
    resources: string[];
    action: HintedString<'list' | 'create' | 'edit' | 'show' | 'delete'>;
    record?: RecordType;
}
export type UseCanAccessResourcesResult<ErrorType = Error> = UseCanAccessResourcesLoadingResult | UseCanAccessResourcesLoadingErrorResult<ErrorType> | UseCanAccessResourcesRefetchErrorResult<ErrorType> | UseCanAccessResourcesSuccessResult;
export interface UseCanAccessResourcesLoadingResult {
    canAccess: undefined;
    error: null;
    isPending: true;
}
export interface UseCanAccessResourcesLoadingErrorResult<ErrorType = Error> {
    canAccess: undefined;
    error: ErrorType;
    isPending: false;
}
export interface UseCanAccessResourcesRefetchErrorResult<ErrorType = Error> {
    canAccess: Record<string, boolean>;
    error: ErrorType;
    isPending: false;
}
export interface UseCanAccessResourcesSuccessResult {
    canAccess: Record<string, boolean>;
    error: null;
    isPending: false;
}
//# sourceMappingURL=useCanAccessResources.d.ts.map