import { SortPayload, FilterPayload } from '../../types';
/**
 * Get the reference inputs parameters (page, sort, filters) and modifiers.
 *
 * @returns {Array} A tuple [parameters, modifiers].
 * Destructure as [
 *    { page, perPage, sort, order, filter, filterValues, displayedFilters, requestSignature },
 *    { setFilters, hideFilter, showFilter, setPage, setPerPage, setSort }
 * ]
 *
 * @example
 *
 * const [referenceParams, referenceParamsActions] = useReferenceParams({
 *      resource: 'posts',
 *      filterDefaultValues: {
 *          published: true
 *      },
 *      sort: {
 *          field: 'published_at',
 *          order: 'DESC'
 *      },
 *      perPage: 25
 * });
 *
 * const {
 *      page,
 *      perPage,
 *      sort,
 *      order,
 *      filter,
 *      filterValues,
 *      displayedFilters,
 *      requestSignature
 * } = referenceParams;
 *
 * const {
 *      setFilters,
 *      hideFilter,
 *      showFilter,
 *      setPage,
 *      setPerPage,
 *      setSort,
 * } = referenceParamsActions;
 */
export declare const useReferenceParams: ({ resource, filter, sort, page, perPage, debounce, }: ReferenceParamsOptions) => [Parameters, Modifiers];
export declare const validQueryParams: string[];
/**
 * Merge list params from 2 different sources:
 *   - the params stored in the local state
 *   - the props passed to the List component (including the filter defaultValues)
 */
export declare const getQuery: ({ params, filterDefaultValues, sort, page, perPage, }: {
    params: any;
    filterDefaultValues: any;
    sort: any;
    page: any;
    perPage: any;
}) => ReferenceParams;
/**
 * Check if user has already set custom sort, page, or filters for this list
 *
 * User params come from the Redux store as the params props. By default,
 * this object is:
 *
 * { filter: {}, order: null, page: 1, perPage: null, sort: null }
 *
 * To check if the user has custom params, we must compare the params
 * to these initial values.
 *
 * @param {Object} params
 */
export declare const hasCustomParams: (params: ReferenceParams) => any;
export declare const getNumberOrDefault: (possibleNumber: string | number | undefined, defaultValue: number) => number;
export interface ReferenceParamsOptions {
    resource: string;
    page?: number;
    perPage?: number;
    sort?: SortPayload;
    filter?: FilterPayload;
    debounce?: number;
}
export interface ReferenceParams {
    sort: string;
    order: 'ASC' | 'DESC';
    page: number;
    perPage: number;
    filter: any;
    displayedFilters: any;
}
interface Parameters extends ReferenceParams {
    filterValues: object;
    displayedFilters: {
        [key: string]: boolean;
    };
    requestSignature: any[];
}
interface Modifiers {
    changeParams: (action: any) => void;
    setPage: (page: number) => void;
    setPerPage: (pageSize: number) => void;
    setSort: (sort: SortPayload) => void;
    setFilters: (filters: any, displayedFilters?: any, debounce?: boolean) => void;
    hideFilter: (filterName: string) => void;
    showFilter: (filterName: string, defaultValue: any) => void;
}
export {};
//# sourceMappingURL=useReferenceParams.d.ts.map