import { UseMutationOptions, UseMutationResult, MutateOptions } from '@tanstack/react-query';
import { RaRecord, CreateParams, Identifier, DataProvider } from '../types';
/**
 * Get a callback to call the dataProvider.create() method, the result and the loading state.
 *
 * @param {string} resource
 * @param {Params} params The create parameters { data }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 *
 * @typedef Params
 * @prop params.data The record to create, e.g. { title: 'hello, world' }
 *
 * @returns The current mutation state. Destructure as [create, { data, error, isPending }].
 *
 * The return value updates according to the request state:
 *
 * - initial: [create, { isPending: false, isIdle: true }]
 * - start:   [create, { isPending: true }]
 * - success: [create, { data: [data from response], isPending: false, isSuccess: true }]
 * - error:   [create, { error: [error from response], isPending: false, isError: true }]
 *
 * The create() function must be called with a resource and a parameter object: create(resource, { data, meta }, options)
 *
 * This hook uses react-query useMutation under the hood.
 * This means the state object contains mutate, isIdle, reset and other react-query methods.
 *
 * @see https://tanstack.com/query/v5/docs/react/reference/useMutation
 *
 * @example // set params when calling the create callback
 *
 * import { useCreate, useRecordContext } from 'react-admin';
 *
 * const LikeButton = () => {
 *     const record = useRecordContext();
 *     const like = { postId: record.id };
 *     const [create, { isPending, error }] = useCreate();
 *     const handleClick = () => {
 *         create('likes', { data: like })
 *     }
 *     if (error) { return <p>ERROR</p>; }
 *     return <button disabled={isPending} onClick={handleClick}>Like</button>;
 * };
 *
 * @example // set params when calling the hook
 *
 * import { useCreate, useRecordContext } from 'react-admin';
 *
 * const LikeButton = () => {
 *     const record = useRecordContext();
 *     const like = { postId: record.id };
 *     const [create, { isPending, error }] = useCreate('likes', { data: like });
 *     if (error) { return <p>ERROR</p>; }
 *     return <button disabled={isPending} onClick={() => create()}>Like</button>;
 * };
 *
 * @example // TypeScript
 * const [create, { data }] = useCreate<Product>('products', { data: product });
 *                    \-- data is Product
 */
export declare const useCreate: <RecordType extends Omit<RaRecord<Identifier>, "id"> = any, MutationError = unknown, ResultRecordType extends RaRecord<Identifier> = RecordType & {
    id: Identifier;
}>(resource?: string, params?: Partial<CreateParams<Partial<RecordType>>>, options?: UseCreateOptions<RecordType, MutationError, ResultRecordType>) => UseCreateResult<RecordType, boolean, MutationError, ResultRecordType>;
export interface UseCreateMutateParams<RecordType extends Omit<RaRecord, 'id'> = any> {
    resource?: string;
    data?: Partial<Omit<RecordType, 'id'>>;
    meta?: any;
}
export type UseCreateOptions<RecordType extends Omit<RaRecord, 'id'> = any, MutationError = unknown, ResultRecordType extends RaRecord = RecordType & {
    id: Identifier;
}> = Omit<UseMutationOptions<ResultRecordType, MutationError, Partial<UseCreateMutateParams<RecordType>>>, 'mutationFn'> & {
    returnPromise?: boolean;
    getMutateWithMiddlewares?: <CreateFunctionType extends DataProvider['create'] = DataProvider['create']>(mutate: CreateFunctionType) => (...Params: Parameters<CreateFunctionType>) => ReturnType<CreateFunctionType>;
};
export type CreateMutationFunction<RecordType extends Omit<RaRecord, 'id'> = any, TReturnPromise extends boolean = boolean, MutationError = unknown, ResultRecordType extends RaRecord = RecordType & {
    id: Identifier;
}> = (resource?: string, params?: Partial<CreateParams<Partial<RecordType>>>, options?: MutateOptions<ResultRecordType, MutationError, Partial<UseCreateMutateParams<RecordType>>, unknown> & {
    returnPromise?: TReturnPromise;
}) => TReturnPromise extends true ? Promise<ResultRecordType> : void;
export type UseCreateResult<RecordType extends Omit<RaRecord, 'id'> = any, TReturnPromise extends boolean = boolean, MutationError = unknown, ResultRecordType extends RaRecord = RecordType & {
    id: Identifier;
}> = [
    CreateMutationFunction<RecordType, TReturnPromise, MutationError, ResultRecordType>,
    UseMutationResult<ResultRecordType, MutationError, Partial<UseCreateMutateParams<RecordType>>, unknown> & {
        isLoading: boolean;
    }
];
//# sourceMappingURL=useCreate.d.ts.map