import { UseQueryOptions, UseQueryResult } from '@tanstack/react-query';
import { RaRecord, GetListParams, GetListResult } from '../types';
/**
 * Call the dataProvider.getList() method and return the resolved result
 * as well as the loading state.
 *
 * The return value updates according to the request state:
 *
 * - start: { isPending: true, refetch }
 * - success: { data: [data from store], total: [total from response], isPending: false, refetch }
 * - error: { error: [error from response], isPending: false, refetch }
 *
 * This hook will return the cached result when called a second time
 * with the same parameters, until the response arrives.
 *
 * @param {string} resource The resource name, e.g. 'posts'
 * @param {Params} params The getList parameters { pagination, sort, filter, meta }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 *
 * @typedef Params
 * @prop params.pagination The request pagination { page, perPage }, e.g. { page: 1, perPage: 10 }
 * @prop params.sort The request sort { field, order }, e.g. { field: 'id', order: 'DESC' }
 * @prop params.filter The request filters, e.g. { title: 'hello, world' }
 * @prop params.meta Optional meta parameters
 *
 * @returns The current request state. Destructure as { data, total, error, isPending, refetch }.
 *
 * @example
 *
 * import { useGetList } from 'react-admin';
 *
 * const LatestNews = () => {
 *     const { data, total, isPending, error } = useGetList(
 *         'posts',
 *         { pagination: { page: 1, perPage: 10 }, sort: { field: 'published_at', order: 'DESC' } }
 *     );
 *     if (isPending) { return <Loading />; }
 *     if (error) { return <p>ERROR</p>; }
 *     return <ul>{data.map(item =>
 *         <li key={item.id}>{item.title}</li>
 *     )}</ul>;
 * };
 */
export declare const useGetList: <RecordType extends RaRecord<import("../types").Identifier> = any>(resource: string, params?: Partial<GetListParams>, options?: UseGetListOptions<RecordType>) => UseGetListHookValue<RecordType>;
export type UseGetListOptions<RecordType extends RaRecord = any> = Omit<UseQueryOptions<GetListResult<RecordType>, Error>, 'queryKey' | 'queryFn'> & {
    onSuccess?: (value: GetListResult<RecordType>) => void;
    onError?: (error: Error) => void;
    onSettled?: (data?: GetListResult<RecordType>, error?: Error | null) => void;
};
export type UseGetListHookValue<RecordType extends RaRecord = any> = UseQueryResult<RecordType[], Error> & {
    total?: number;
    pageInfo?: {
        hasNextPage?: boolean;
        hasPreviousPage?: boolean;
    };
    meta?: any;
};
//# sourceMappingURL=useGetList.d.ts.map