"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.useGetList = void 0;
var react_1 = require("react");
var react_query_1 = require("@tanstack/react-query");
var useDataProvider_1 = require("./useDataProvider");
var util_1 = require("../util");
var MAX_DATA_LENGTH_TO_CACHE = 100;
/**
 * Call the dataProvider.getList() method and return the resolved result
 * as well as the loading state.
 *
 * The return value updates according to the request state:
 *
 * - start: { isPending: true, refetch }
 * - success: { data: [data from store], total: [total from response], isPending: false, refetch }
 * - error: { error: [error from response], isPending: false, refetch }
 *
 * This hook will return the cached result when called a second time
 * with the same parameters, until the response arrives.
 *
 * @param {string} resource The resource name, e.g. 'posts'
 * @param {Params} params The getList parameters { pagination, sort, filter, meta }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 *
 * @typedef Params
 * @prop params.pagination The request pagination { page, perPage }, e.g. { page: 1, perPage: 10 }
 * @prop params.sort The request sort { field, order }, e.g. { field: 'id', order: 'DESC' }
 * @prop params.filter The request filters, e.g. { title: 'hello, world' }
 * @prop params.meta Optional meta parameters
 *
 * @returns The current request state. Destructure as { data, total, error, isPending, refetch }.
 *
 * @example
 *
 * import { useGetList } from 'react-admin';
 *
 * const LatestNews = () => {
 *     const { data, total, isPending, error } = useGetList(
 *         'posts',
 *         { pagination: { page: 1, perPage: 10 }, sort: { field: 'published_at', order: 'DESC' } }
 *     );
 *     if (isPending) { return <Loading />; }
 *     if (error) { return <p>ERROR</p>; }
 *     return <ul>{data.map(item =>
 *         <li key={item.id}>{item.title}</li>
 *     )}</ul>;
 * };
 */
var useGetList = function (resource, params, options) {
    if (params === void 0) { params = {}; }
    if (options === void 0) { options = {}; }
    var _a = params.pagination, pagination = _a === void 0 ? { page: 1, perPage: 25 } : _a, _b = params.sort, sort = _b === void 0 ? { field: 'id', order: 'DESC' } : _b, _c = params.filter, filter = _c === void 0 ? {} : _c, meta = params.meta;
    var dataProvider = (0, useDataProvider_1.useDataProvider)();
    var queryClient = (0, react_query_1.useQueryClient)();
    var _d = options.onError, onError = _d === void 0 ? noop : _d, _e = options.onSuccess, onSuccess = _e === void 0 ? noop : _e, _f = options.onSettled, onSettled = _f === void 0 ? noop : _f, queryOptions = __rest(options, ["onError", "onSuccess", "onSettled"]);
    var onSuccessEvent = (0, util_1.useEvent)(onSuccess);
    var onErrorEvent = (0, util_1.useEvent)(onError);
    var onSettledEvent = (0, util_1.useEvent)(onSettled);
    var result = (0, react_query_1.useQuery)(__assign({ queryKey: [resource, 'getList', { pagination: pagination, sort: sort, filter: filter, meta: meta }], queryFn: function (queryParams) {
            return dataProvider
                .getList(resource, {
                pagination: pagination,
                sort: sort,
                filter: filter,
                meta: meta,
                signal: dataProvider.supportAbortSignal === true
                    ? queryParams.signal
                    : undefined,
            })
                .then(function (_a) {
                var data = _a.data, total = _a.total, pageInfo = _a.pageInfo, meta = _a.meta;
                return ({
                    data: data,
                    total: total,
                    pageInfo: pageInfo,
                    meta: meta,
                });
            });
        } }, queryOptions));
    var metaValue = (0, react_1.useRef)(meta);
    var resourceValue = (0, react_1.useRef)(resource);
    (0, react_1.useEffect)(function () {
        metaValue.current = meta;
    }, [meta]);
    (0, react_1.useEffect)(function () {
        resourceValue.current = resource;
    }, [resource]);
    (0, react_1.useEffect)(function () {
        var _a;
        if (result.data === undefined ||
            result.error != null ||
            result.isFetching)
            return;
        // optimistically populate the getOne cache
        if (((_a = result.data) === null || _a === void 0 ? void 0 : _a.data) &&
            result.data.data.length <= MAX_DATA_LENGTH_TO_CACHE) {
            result.data.data.forEach(function (record) {
                queryClient.setQueryData([
                    resourceValue.current,
                    'getOne',
                    { id: String(record.id), meta: metaValue.current },
                ], function (oldRecord) { return oldRecord !== null && oldRecord !== void 0 ? oldRecord : record; });
            });
        }
        onSuccessEvent(result.data);
    }, [
        onSuccessEvent,
        queryClient,
        result.data,
        result.error,
        result.isFetching,
    ]);
    (0, react_1.useEffect)(function () {
        if (result.error == null || result.isFetching)
            return;
        onErrorEvent(result.error);
    }, [onErrorEvent, result.error, result.isFetching]);
    (0, react_1.useEffect)(function () {
        if (result.status === 'pending' || result.isFetching)
            return;
        onSettledEvent(result.data, result.error);
    }, [
        onSettledEvent,
        result.data,
        result.error,
        result.status,
        result.isFetching,
    ]);
    return (0, react_1.useMemo)(function () {
        return result.data
            ? __assign(__assign({}, result), result.data) : result;
    }, [result]);
};
exports.useGetList = useGetList;
var noop = function () { return undefined; };
//# sourceMappingURL=useGetList.js.map