import { UseQueryOptions, UseQueryResult } from '@tanstack/react-query';
import { RaRecord, GetManyParams } from '../types';
/**
 * Call the dataProvider.getMany() method and return the resolved result
 * as well as the loading state.
 *
 * The return value updates according to the request state:
 *
 * - start: { isPending: true, refetch }
 * - success: { data: [data from store], isPending: false, refetch }
 * - error: { error: [error from response], isPending: false, refetch }
 *
 * This hook will return the cached result when called a second time
 * with the same parameters, until the response arrives.
 *
 * @param {string} resource The resource name, e.g. 'posts'
 * @param {Params} params The getMany parameters { ids, meta }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 *
 * @typedef Params
 * @prop params.ids The ids to get, e.g. [123, 456, 789]
 * @prop params.meta Optional meta parameters
 *
 * @returns The current request state. Destructure as { data, error, isPending, refetch }.
 *
 * @example
 *
 * import { useGetMany } from 'react-admin';
 *
 * const PostTags = ({ post }) => {
 *     const { data, isPending, error } = useGetMany(
 *         'tags',
 *         { ids: post.tags },
 *     );
 *     if (isPending) { return <Loading />; }
 *     if (error) { return <p>ERROR</p>; }
 *     return <ul>{data.map(tag =>
 *         <li key={tag.id}>{tag.name}</li>
 *     )}</ul>;
 * };
 */
export declare const useGetMany: <RecordType extends RaRecord<import("../types").Identifier> = any>(resource: string, params: Partial<GetManyParams<RecordType>>, options?: UseGetManyOptions<RecordType>) => UseGetManyHookValue<RecordType>;
export type UseGetManyOptions<RecordType extends RaRecord = any> = Omit<UseQueryOptions<RecordType[], Error>, 'queryKey' | 'queryFn'> & {
    onSuccess?: (data: RecordType[]) => void;
    onError?: (error: Error) => void;
    onSettled?: (data?: RecordType[], error?: Error | null) => void;
};
export type UseGetManyHookValue<RecordType extends RaRecord = any> = UseQueryResult<RecordType[], Error>;
//# sourceMappingURL=useGetMany.d.ts.map