import { UseQueryOptions } from '@tanstack/react-query';
import { UseGetManyHookValue } from './useGetMany';
import { Identifier, RaRecord, GetManyParams } from '../types';
/**
 * Call the dataProvider.getMany() method and return the resolved result
 * as well as the loading state.
 *
 * The return value updates according to the request state:
 *
 * - start: { isPending: true, isFetching: true, refetch }
 * - success: { data: [data from response], isPending: false, isFetching: false, refetch }
 * - error: { error: [error from response], isPending: false, isFetching: false, refetch }
 *
 * This hook will return the cached result when called a second time
 * with the same parameters, until the response arrives.
 *
 * This hook aggregates and deduplicates calls to the same resource, so for instance, if an app calls:
 *
 * useGetManyAggregate('tags', [1, 2, 3]);
 * useGetManyAggregate('tags', [3, 4]);
 *
 * during the same tick, the hook will only call the dataProvider once with the following parameters:
 *
 * dataProvider.getMany('tags', [1, 2, 3, 4])
 *
 * @param resource The resource name, e.g. 'posts'
 * @param {Params} params The getMany parameters { ids, meta }
 * @param {Object} options Options object to pass to the dataProvider.
 * @param {boolean} options.enabled Flag to conditionally run the query. If it's false, the query will not run
 * @param {Function} options.onSuccess Side effect function to be executed upon success, e.g. { onSuccess: { refresh: true } }
 * @param {Function} options.onError Side effect function to be executed upon failure, e.g. { onError: error => notify(error.message) }
 *
 * @typedef Params
 * @prop params.ids The ids to get, e.g. [123, 456, 789]
 * @prop params.meta Optional meta parameters

 * @returns The current request state. Destructure as { data, error, isPending, isFetching, refetch }.
 *
 * @example
 *
 * import { useGetManyAggregate, useRecordContext } from 'react-admin';
 *
 * const PostTags = () => {
 *     const record = useRecordContext();
 *     const { data, isPending, error } = useGetManyAggregate('tags', { ids: record.tagIds });
 *     if (isPending) { return <Loading />; }
 *     if (error) { return <p>ERROR</p>; }
 *     return (
 *          <ul>
 *              {data.map(tag => (
 *                  <li key={tag.id}>{tag.name}</li>
 *              ))}
 *          </ul>
 *      );
 * };
 */
export declare const useGetManyAggregate: <RecordType extends RaRecord<Identifier> = any>(resource: string, params: Partial<GetManyParams<RecordType>>, options?: UseGetManyAggregateOptions<RecordType>) => UseGetManyHookValue<RecordType>;
export type UseGetManyAggregateOptions<RecordType extends RaRecord> = Omit<UseQueryOptions<RecordType[]>, 'queryKey' | 'queryFn'> & {
    onSuccess?: (data: RecordType[]) => void;
    onError?: (error: Error) => void;
    onSettled?: (data?: RecordType[], error?: Error | null) => void;
};
//# sourceMappingURL=useGetManyAggregate.d.ts.map