import { UseMutationOptions, UseMutationResult, MutateOptions } from '@tanstack/react-query';
import { RaRecord, UpdateParams, MutationMode, DataProvider } from '../types';
/**
 * Get a callback to call the dataProvider.update() method, the result and the loading state.
 *
 * @param {string} resource
 * @param {Params} params The update parameters { id, data, previousData, meta }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 * May include a mutation mode (optimistic/pessimistic/undoable), e.g. { mutationMode: 'undoable' }
 *
 * @typedef Params
 * @prop params.id The resource identifier, e.g. 123
 * @prop params.data The updates to merge into the record, e.g. { views: 10 }
 * @prop params.previousData The record before the update is applied
 * @prop params.meta Optional meta data
 *
 * @returns The current mutation state. Destructure as [update, { data, error, isPending }].
 *
 * The return value updates according to the request state:
 *
 * - initial: [update, { isPending: false, isIdle: true }]
 * - start:   [update, { isPending: true }]
 * - success: [update, { data: [data from response], isPending: false, isSuccess: true }]
 * - error:   [update, { error: [error from response], isPending: false, isError: true }]
 *
 * The update() function must be called with a resource and a parameter object: update(resource, { id, data, previousData }, options)
 *
 * This hook uses react-query useMutation under the hood.
 * This means the state object contains mutate, isIdle, reset and other react-query methods.
 *
 * @see https://react-query-v3.tanstack.com/reference/useMutation
 *
 * @example // set params when calling the update callback
 *
 * import { useUpdate, useRecordContext } from 'react-admin';
 *
 * const IncreaseLikeButton = () => {
 *     const record = useRecordContext();
 *     const diff = { likes: record.likes + 1 };
 *     const [update, { isPending, error }] = useUpdate();
 *     const handleClick = () => {
 *         update('likes', { id: record.id, data: diff, previousData: record })
 *     }
 *     if (error) { return <p>ERROR</p>; }
 *     return <button disabled={isPending} onClick={handleClick}>Like</div>;
 * };
 *
 * @example // set params when calling the hook
 *
 * import { useUpdate, useRecordContext } from 'react-admin';
 *
 * const IncreaseLikeButton = () => {
 *     const record = useRecordContext();
 *     const diff = { likes: record.likes + 1 };
 *     const [update, { isPending, error }] = useUpdate('likes', { id: record.id, data: diff, previousData: record });
 *     if (error) { return <p>ERROR</p>; }
 *     return <button disabled={isPending} onClick={() => update()}>Like</button>;
 * };
 *
 * @example // TypeScript
 * const [update, { data }] = useUpdate<Product>('products', { id, data: diff, previousData: product });
 *                    \-- data is Product
 */
export declare const useUpdate: <RecordType extends RaRecord<import("../types").Identifier> = any, ErrorType = Error>(resource?: string, params?: Partial<UpdateParams<RecordType>>, options?: UseUpdateOptions<RecordType, ErrorType>) => UseUpdateResult<RecordType, boolean, ErrorType>;
export interface UseUpdateMutateParams<RecordType extends RaRecord = any> {
    resource?: string;
    id?: RecordType['id'];
    data?: Partial<RecordType>;
    previousData?: any;
    meta?: any;
}
export type UseUpdateOptions<RecordType extends RaRecord = any, ErrorType = Error> = UseMutationOptions<RecordType, ErrorType, Partial<Omit<UseUpdateMutateParams<RecordType>, 'mutationFn'>>> & {
    mutationMode?: MutationMode;
    returnPromise?: boolean;
    getMutateWithMiddlewares?: <UpdateFunctionType extends DataProvider['update'] = DataProvider['update']>(mutate: UpdateFunctionType) => (...Params: Parameters<UpdateFunctionType>) => ReturnType<UpdateFunctionType>;
};
export type UpdateMutationFunction<RecordType extends RaRecord = any, TReturnPromise extends boolean = boolean, ErrorType = Error> = (resource?: string, params?: Partial<UpdateParams<RecordType>>, options?: MutateOptions<RecordType, ErrorType, Partial<UseUpdateMutateParams<RecordType>>, unknown> & {
    mutationMode?: MutationMode;
    returnPromise?: TReturnPromise;
}) => Promise<TReturnPromise extends true ? RecordType : void>;
export type UseUpdateResult<RecordType extends RaRecord = any, TReturnPromise extends boolean = boolean, ErrorType = Error> = [
    UpdateMutationFunction<RecordType, TReturnPromise, ErrorType>,
    UseMutationResult<RecordType, ErrorType, Partial<UpdateParams<RecordType> & {
        resource?: string;
    }>, unknown> & {
        isLoading: boolean;
    }
];
//# sourceMappingURL=useUpdate.d.ts.map