"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useGetPathForRecord = void 0;
var react_1 = require("react");
var useResourceContext_1 = require("../core/useResourceContext");
var useRecordContext_1 = require("../controller/record/useRecordContext");
var auth_1 = require("../auth");
var core_1 = require("../core");
var useCreatePath_1 = require("./useCreatePath");
/**
 * Get a path for a record, based on the current resource and the link type.
 *
 * Accepted link types are 'edit', 'show', a route string, false, or a function returning one of these types.
 *
 * @example
 * // basic usage (leverages RecordContext, ResourceContext and ResourceDefinitionContext)
 * const EditLink = () => {
 *   const path = useGetPathForRecord();
 *   return path ? <Link to={path}>Edit</Link> : null;
 * };
 *
 * // controlled mode
 * const EditLink = ({ record, resource }) => {
 *    const path = useGetPathForRecord({ record, resource, link: 'edit' });
 *    return path ? <Link to={path}>Edit</Link> : null;
 * };
 *
 * // the link option can be a function
 * const EditLink = ({ record, resource }) => {
 *   const path = useGetPathForRecord({ record, resource, link: (record, resource) => record.canEdit ? 'edit' : false });
 *   return path ? <Link to={path}>Edit</Link> : null;
 * };
 *
 * // the link option can be a function returning a promise
 * const EditLink = ({ record, resource }) => {
 *   const path = useGetPathForRecord({ record, resource, link: async (record, resource) => {
 *     const canEdit = await canEditRecord(record, resource);
 *     return canEdit ? 'edit' : false;
 *   }});
 *   return path ? <Link to={path}>Edit</Link> : null;
 * };
 */
var useGetPathForRecord = function (options) {
    if (options === void 0) { options = {}; }
    var link = (options || {}).link;
    var record = (0, useRecordContext_1.useRecordContext)(options);
    var resource = (0, useResourceContext_1.useResourceContext)(options);
    if (!resource) {
        throw new Error('Cannot generate a link for a record without a resource. You must use useGetPathForRecord within a ResourceContextProvider, or pass a resource prop.');
    }
    var resourceDefinition = (0, core_1.useResourceDefinition)(options);
    var createPath = (0, useCreatePath_1.useCreatePath)();
    var _a = (0, react_1.useState)(link && typeof link !== 'function' && record != null
        ? createPath({
            resource: resource,
            id: record.id,
            type: link,
        })
        : false), path = _a[0], setPath = _a[1];
    // in preparation for the default value, does the user have access to the show and edit pages?
    // (we can't run hooks conditionally, so we need to run them even though the link is specified)
    var canAccessShow = (0, auth_1.useCanAccess)({
        action: 'show',
        resource: resource,
        record: record,
        enabled: link == null && resourceDefinition.hasShow,
    }).canAccess;
    var canAccessEdit = (0, auth_1.useCanAccess)({
        action: 'edit',
        resource: resource,
        record: record,
        enabled: link == null && resourceDefinition.hasEdit,
    }).canAccess;
    (0, react_1.useEffect)(function () {
        if (!record)
            return;
        // Handle the inferred link type case
        if (link == null) {
            // We must check whether the resource has an edit view because if there is no
            // authProvider, canAccessShow will always be true
            if (resourceDefinition.hasShow && canAccessShow) {
                setPath(createPath({
                    resource: resource,
                    id: record.id,
                    type: 'show',
                }));
                return;
            }
            // We must check whether the resource has an edit view because if there is no
            // authProvider, canAccessEdit will always be true
            if (resourceDefinition.hasEdit && canAccessEdit) {
                setPath(createPath({
                    resource: resource,
                    id: record.id,
                    type: 'edit',
                }));
                return;
            }
        }
        // Handle the link function case
        if (typeof link === 'function') {
            var linkResult = link(record, resource);
            if (linkResult instanceof Promise) {
                linkResult.then(function (resolvedPath) { return setPath(resolvedPath); });
                return;
            }
            setPath(linkResult
                ? createPath({
                    resource: resource,
                    id: record.id,
                    type: linkResult,
                })
                : false);
        }
    }, [
        createPath,
        canAccessShow,
        canAccessEdit,
        link,
        record,
        resource,
        resourceDefinition.hasEdit,
        resourceDefinition.hasShow,
    ]);
    return path;
};
exports.useGetPathForRecord = useGetPathForRecord;
//# sourceMappingURL=useGetPathForRecord.js.map