"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.useFieldValue = void 0;
var get_1 = __importDefault(require("lodash/get"));
var controller_1 = require("../controller");
/**
 * A hook that gets the value of a field of the current record.
 * @param params The hook parameters
 * @param params.source The field source
 * @param params.record The record to use. Uses the record from the RecordContext if not provided
 * @param params.defaultValue The value to return when the field value is empty
 * @returns The field value
 *
 * @example
 * const MyField = (props: { source: string }) => {
 *   const value = useFieldValue(props);
 *   return <span>{value}</span>;
 * }
 */
var useFieldValue = function (params) {
    var defaultValue = params.defaultValue, source = params.source;
    // We use the record from the RecordContext and do not rely on the SourceContext on purpose to
    // avoid having the wrong source targeting the record.
    // Indeed, some components may create a sub record context (SimpleFormIterator, TranslatableInputs, etc.). In this case,
    // it they used the SourceContext as well, they would have the wrong source.
    // Inputs needs the SourceContext as they rely on the Form value and you can't have nested forms.
    // Fields needs the RecordContext as they rely on the Record value and you can have nested RecordContext.
    var record = (0, controller_1.useRecordContext)(params);
    return (0, get_1.default)(record, source, defaultValue);
};
exports.useFieldValue = useFieldValue;
//# sourceMappingURL=useFieldValue.js.map