import { UseQueryOptions, QueryObserverResult } from '@tanstack/react-query';
import { UserIdentity } from '../types';
/**
 * Return the current user identity by calling authProvider.getIdentity() on mount
 *
 * The return value updates according to the call state:
 *
 * - mount: { isPending: true }
 * - success: { identity, refetch: () => {}, isPending: false }
 * - error: { error: Error, isPending: false }
 *
 * The implementation is left to the authProvider.
 *
 * @returns The current user identity. Destructure as { isPending, identity, error, refetch }.
 *
 * @example
 * import { useGetIdentity, useGetOne } from 'react-admin';
 *
 * const PostDetail = ({ id }) => {
 *     const { data: post, isPending: postLoading } = useGetOne('posts', { id });
 *     const { identity, isPending: identityLoading } = useGetIdentity();
 *     if (postLoading || identityLoading) return <>Loading...</>;
 *     if (!post.lockedBy || post.lockedBy === identity.id) {
 *         // post isn't locked, or is locked by me
 *         return <PostEdit post={post} />
 *     } else {
 *         // post is locked by someone else and cannot be edited
 *         return <PostShow post={post} />
 *     }
 * }
 */
export declare const useGetIdentity: <ErrorType extends Error = Error>(options?: UseGetIdentityOptions<ErrorType>) => UseGetIdentityResult<ErrorType>;
export interface UseGetIdentityOptions<ErrorType extends Error = Error> extends Omit<UseQueryOptions<UserIdentity, ErrorType>, 'queryKey' | 'queryFn'> {
    onSuccess?: (data: UserIdentity) => void;
    onError?: (err: Error) => void;
    onSettled?: (data?: UserIdentity, error?: Error | null) => void;
}
export type UseGetIdentityResult<ErrorType = Error> = QueryObserverResult<UserIdentity, ErrorType> & {
    identity: UserIdentity | undefined;
};
export default useGetIdentity;
//# sourceMappingURL=useGetIdentity.d.ts.map