var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
import { useCallback, useMemo, useState, useRef, useEffect } from 'react';
import lodashDebounce from 'lodash/debounce';
import removeEmpty from '../../util/removeEmpty';
import { queryReducer, HIDE_FILTER, SET_FILTER, SET_PAGE, SET_PER_PAGE, SET_SORT, SHOW_FILTER, SORT_ASC, } from '../list';
/**
 * Get the reference inputs parameters (page, sort, filters) and modifiers.
 *
 * @returns {Array} A tuple [parameters, modifiers].
 * Destructure as [
 *    { page, perPage, sort, order, filter, filterValues, displayedFilters, requestSignature },
 *    { setFilters, hideFilter, showFilter, setPage, setPerPage, setSort }
 * ]
 *
 * @example
 *
 * const [referenceParams, referenceParamsActions] = useReferenceParams({
 *      resource: 'posts',
 *      filterDefaultValues: {
 *          published: true
 *      },
 *      sort: {
 *          field: 'published_at',
 *          order: 'DESC'
 *      },
 *      perPage: 25
 * });
 *
 * const {
 *      page,
 *      perPage,
 *      sort,
 *      order,
 *      filter,
 *      filterValues,
 *      displayedFilters,
 *      requestSignature
 * } = referenceParams;
 *
 * const {
 *      setFilters,
 *      hideFilter,
 *      showFilter,
 *      setPage,
 *      setPerPage,
 *      setSort,
 * } = referenceParamsActions;
 */
export var useReferenceParams = function (_a) {
    var resource = _a.resource, filter = _a.filter, _b = _a.sort, sort = _b === void 0 ? defaultSort : _b, _c = _a.page, page = _c === void 0 ? 1 : _c, _d = _a.perPage, perPage = _d === void 0 ? 10 : _d, _e = _a.debounce, debounce = _e === void 0 ? 500 : _e;
    var _f = useState(defaultParams), params = _f[0], setParams = _f[1];
    var tempParams = useRef();
    var requestSignature = [
        resource,
        JSON.stringify(params),
        JSON.stringify(filter),
        JSON.stringify(sort),
        page,
        perPage,
    ];
    var query = useMemo(function () {
        return getQuery({
            params: params,
            filterDefaultValues: filter,
            sort: sort,
            page: page,
            perPage: perPage,
        });
    }, requestSignature // eslint-disable-line react-hooks/exhaustive-deps
    );
    var changeParams = useCallback(function (action) {
        if (!tempParams.current) {
            // no other changeParams action dispatched this tick
            var newTempParams_1 = queryReducer(query, action);
            tempParams.current = newTempParams_1;
            // schedule side effects for next tick
            setTimeout(function () {
                setParams(newTempParams_1);
                tempParams.current = undefined;
            }, 0);
        }
        else {
            // side effects already scheduled, just change the params
            tempParams.current = queryReducer(tempParams.current, action);
        }
    }, requestSignature); // eslint-disable-line react-hooks/exhaustive-deps
    var setSort = useCallback(function (sort) {
        return changeParams({
            type: SET_SORT,
            payload: sort,
        });
    }, requestSignature // eslint-disable-line react-hooks/exhaustive-deps
    );
    var setPage = useCallback(function (newPage) { return changeParams({ type: SET_PAGE, payload: newPage }); }, requestSignature // eslint-disable-line react-hooks/exhaustive-deps
    );
    var setPerPage = useCallback(function (newPerPage) {
        return changeParams({ type: SET_PER_PAGE, payload: newPerPage });
    }, requestSignature // eslint-disable-line react-hooks/exhaustive-deps
    );
    var filterValues = query.filter || emptyObject;
    var displayedFilterValues = query.displayedFilters || emptyObject;
    var debouncedSetFilters = useRef(lodashDebounce(function (filter, displayedFilters) {
        if (displayedFilters === void 0) { displayedFilters = undefined; }
        changeParams({
            type: SET_FILTER,
            payload: {
                filter: removeEmpty(filter),
                displayedFilters: displayedFilters,
            },
        });
    }, debounce));
    useEffect(function () {
        return function () {
            // eslint-disable-next-line react-hooks/exhaustive-deps
            debouncedSetFilters.current.cancel();
        };
    }, []);
    var setFilters = useCallback(function (filter, displayedFilters, debounce) {
        if (displayedFilters === void 0) { displayedFilters = undefined; }
        if (debounce === void 0) { debounce = false; }
        debounce
            ? debouncedSetFilters.current(filter, displayedFilters)
            : changeParams({
                type: SET_FILTER,
                payload: {
                    filter: removeEmpty(filter),
                    displayedFilters: displayedFilters,
                },
            });
    }, requestSignature // eslint-disable-line react-hooks/exhaustive-deps
    );
    var hideFilter = useCallback(function (filterName) {
        changeParams({
            type: HIDE_FILTER,
            payload: filterName,
        });
    }, requestSignature); // eslint-disable-line react-hooks/exhaustive-deps
    var showFilter = useCallback(function (filterName, defaultValue) {
        changeParams({
            type: SHOW_FILTER,
            payload: {
                filterName: filterName,
                defaultValue: defaultValue,
            },
        });
    }, requestSignature); // eslint-disable-line react-hooks/exhaustive-deps
    return [
        __assign(__assign({ filterValues: filterValues, requestSignature: requestSignature }, query), { displayedFilters: displayedFilterValues }),
        {
            changeParams: changeParams,
            setPage: setPage,
            setPerPage: setPerPage,
            setSort: setSort,
            setFilters: setFilters,
            hideFilter: hideFilter,
            showFilter: showFilter,
        },
    ];
};
export var validQueryParams = [
    'page',
    'perPage',
    'sort',
    'order',
    'filter',
    'displayedFilters',
];
/**
 * Merge list params from 2 different sources:
 *   - the params stored in the local state
 *   - the props passed to the List component (including the filter defaultValues)
 */
export var getQuery = function (_a) {
    var params = _a.params, filterDefaultValues = _a.filterDefaultValues, sort = _a.sort, page = _a.page, perPage = _a.perPage;
    var query = hasCustomParams(params)
        ? __assign({}, params) : { filter: filterDefaultValues || {} };
    if (!query.sort) {
        query.sort = sort.field;
        query.order = sort.order;
    }
    if (query.page == null) {
        query.page = page;
    }
    if (query.perPage == null) {
        query.perPage = perPage;
    }
    if (query.page == null) {
        query.page = 1;
    }
    return __assign(__assign({}, query), { page: getNumberOrDefault(query.page, 1), perPage: getNumberOrDefault(query.perPage, 10) });
};
/**
 * Check if user has already set custom sort, page, or filters for this list
 *
 * User params come from the Redux store as the params props. By default,
 * this object is:
 *
 * { filter: {}, order: null, page: 1, perPage: null, sort: null }
 *
 * To check if the user has custom params, we must compare the params
 * to these initial values.
 *
 * @param {Object} params
 */
export var hasCustomParams = function (params) {
    return (params &&
        params.filter &&
        (Object.keys(params.filter).length > 0 ||
            params.order != null ||
            params.page !== 1 ||
            params.perPage != null ||
            params.sort != null));
};
export var getNumberOrDefault = function (possibleNumber, defaultValue) {
    if (typeof possibleNumber === 'undefined') {
        return defaultValue;
    }
    var parsedNumber = typeof possibleNumber === 'string'
        ? parseInt(possibleNumber, 10)
        : possibleNumber;
    return isNaN(parsedNumber) ? defaultValue : parsedNumber;
};
var emptyObject = {};
var defaultSort = {
    field: 'id',
    order: SORT_ASC,
};
var defaultParams = {};
//# sourceMappingURL=useReferenceParams.js.map