import { SortPayload, FilterPayload } from '../../types';
export interface ListParams {
    sort: string;
    order: 'ASC' | 'DESC';
    page: number;
    perPage: number;
    filter: any;
    displayedFilters: any;
}
/**
 * Get the list parameters (page, sort, filters) and modifiers.
 *
 * These parameters are merged from 3 sources:
 *   - the query string from the URL
 *   - the params stored in the state (from previous navigation)
 *   - the options passed to the hook (including the filter defaultValues)
 *
 * @returns {Array} A tuple [parameters, modifiers].
 * Destructure as [
 *    { page, perPage, sort, order, filter, filterValues, displayedFilters, requestSignature },
 *    { setFilters, hideFilter, showFilter, setPage, setPerPage, setSort }
 * ]
 *
 * @example
 *
 * const [listParams, listParamsActions] = useListParams({
 *      resource: 'posts',
 *      location: location // From react-router. Injected to your component by react-admin inside a List
 *      filterDefaultValues: {
 *          published: true
 *      },
 *      sort: {
 *          field: 'published_at',
 *          order: 'DESC'
 *      },
 *      perPage: 25
 * });
 *
 * const {
 *      page,
 *      perPage,
 *      sort,
 *      order,
 *      filter,
 *      filterValues,
 *      displayedFilters,
 *      requestSignature
 * } = listParams;
 *
 * const {
 *      setFilters,
 *      hideFilter,
 *      showFilter,
 *      setPage,
 *      setPerPage,
 *      setSort,
 * } = listParamsActions;
 */
export declare const useListParams: ({ debounce, disableSyncWithLocation, filterDefaultValues, perPage, resource, sort, storeKey, }: ListParamsOptions) => [Parameters, Modifiers];
export declare const parseQueryFromLocation: ({ search }: {
    search: any;
}) => Partial<ListParams>;
/**
 * Check if user has already set custom sort, page, or filters for this list
 *
 * User params come from the store as the params props. By default,
 * this object is:
 *
 * { filter: {}, order: null, page: 1, perPage: null, sort: null }
 *
 * To check if the user has custom params, we must compare the params
 * to these initial values.
 *
 * @param {Object} params
 */
export declare const hasCustomParams: (params: ListParams) => any;
/**
 * Merge list params from 3 different sources:
 *   - the query string
 *   - the params stored in the state (from previous navigation)
 *   - the props passed to the List component (including the filter defaultValues)
 */
export declare const getQuery: ({ queryFromLocation, params, filterDefaultValues, sort, perPage, }: {
    queryFromLocation: any;
    params: any;
    filterDefaultValues: any;
    sort: any;
    perPage: any;
}) => ListParams;
export declare const getNumberOrDefault: (possibleNumber: string | number | undefined, defaultValue: number) => number;
export interface ListParamsOptions {
    debounce?: number;
    disableSyncWithLocation?: boolean;
    filterDefaultValues?: FilterPayload;
    perPage?: number;
    resource: string;
    sort?: SortPayload;
    storeKey?: string | false;
}
interface Parameters extends ListParams {
    filterValues: object;
    displayedFilters: {
        [key: string]: boolean;
    };
    requestSignature: any[];
}
interface Modifiers {
    changeParams: (action: any) => void;
    setPage: (page: number) => void;
    setPerPage: (pageSize: number) => void;
    setSort: (sort: SortPayload) => void;
    setFilters: (filters: any, displayedFilters?: any, debounce?: boolean) => void;
    hideFilter: (filterName: string) => void;
    showFilter: (filterName: string, defaultValue: any) => void;
}
export {};
//# sourceMappingURL=useListParams.d.ts.map