import { UseQueryOptions } from '@tanstack/react-query';
import { FilterPayload, RaRecord, SortPayload } from '../types';
/**
 * A hook used to fetch the previous and next record identifiers for a given record and resource.
 *
 * It fetches the list of records according to the filters
 * and the sort order configured in the list, and merges
 * the filters and the sorting order passed as props.
 *
 * usePrevNextController can be used anywhere a record context is provided
 * (often inside a `<Show>` or `<Edit>` component).
 *
 * @example <caption>Simple usage</caption>
 *
 * import { usePrevNextControllerProps } from 'ra-core';
 * const {
 *         hasPrev,
 *         hasNext,
 *         prevPath,
 *         nextPath,
 *         index,
 *         total,
 *         error,
 *         isPending,
 *     } = usePrevNextController(props);
 *
 * @example <caption>Custom PrevNextButton</caption>
 *
 * import { UsePrevNextControllerProps, useTranslate } from 'ra-core';
 * import NavigateBefore from '@mui/icons-material/NavigateBefore';
 * import NavigateNext from '@mui/icons-material/NavigateNext';
 * import ErrorIcon from '@mui/icons-material/Error';
 * import { Link } from 'react-router-dom';
 * import { CircularProgress, IconButton } from '@mui/material';
 *
 * const MyPrevNextButtons = props => {
 *     const {
 *         hasPrev,
 *         hasNext,
 *         nextPath,
 *         prevPath,
 *         index,
 *         total,
 *         error,
 *         isPending,
 *     } = usePrevNextController(props);
 *
 *     const translate = useTranslate();
 *
 *     if (isPending) {
 *         return <CircularProgress size={14} />;
 *     }
 *
 *     if (error) {
 *         return (
 *             <ErrorIcon
 *                 color="error"
 *                 fontSize="small"
 *                 titleAccess="error"
 *                 aria-errormessage={error.message}
 *             />
 *         );
 *     }
 *
 *     return (
 *         <ul>
 *             <li>
 *                 <IconButton
 *                     component={hasPrev ? Link : undefined}
 *                     to={navigateToPrev}
 *                     aria-label={translate('ra.navigation.previous')}
 *                     disabled={!hasPrev}
 *                 >
 *                     <NavigateBefore />
 *                 </IconButton>
 *             </li>
 *             {typeof index === 'number' && (
 *                 <li>
 *                     {index + 1} / {total}
 *                 </li>
 *             )}
 *             <li>
 *                 <IconButton
 *                     component={hasNext ? Link : undefined}
 *                     to={navigateToNext}
 *                     aria-label={translate('ra.navigation.next')}
 *                     disabled={!hasNext}
 *                 >
 *                     <NavigateNext />
 *                 </IconButton>
 *             </li>
 *         </ul>
 *     );
 * };
 */
export declare const usePrevNextController: <RecordType extends RaRecord<import("../types").Identifier> = any>(props: UsePrevNextControllerProps<RecordType>) => UsePrevNextControllerResult;
export interface UsePrevNextControllerProps<RecordType extends RaRecord = any> {
    linkType?: 'edit' | 'show';
    storeKey?: string | false;
    limit?: number;
    filter?: FilterPayload;
    filterDefaultValues?: FilterPayload;
    sort?: SortPayload;
    resource?: string;
    queryOptions?: Omit<UseQueryOptions<{
        data: RecordType[];
        total?: number;
        pageInfo?: {
            hasNextPage?: boolean;
            hasPreviousPage?: boolean;
        };
    }>, 'queryFn' | 'queryKey'> & {
        meta?: any;
    };
}
export type UsePrevNextControllerResult = {
    isFetching: boolean;
    isLoading: boolean;
    isPending: boolean;
    hasPrev: boolean;
    hasNext: boolean;
    prevPath: string | undefined;
    nextPath: string | undefined;
    index: number | undefined;
    total: number | undefined;
    error?: any;
};
//# sourceMappingURL=usePrevNextController.d.ts.map