import { useResourceDefinitions } from './useResourceDefinitions';
import { useTranslate } from '../i18n';
import { humanize, pluralize, singularize } from 'inflection';
/**
 * A hook which returns function to get a translated resource name. It will use the label option of the `Resource` component if it was provided.
 *
 * @returns {GetResourceLabel} A function which takes a resource name and an optional number indicating the number of items (used for pluralization) and returns a translated string.
 * @example
 * const Menu = () => {
 *     const resources = useResourceDefinitions();
 *     const getResourceLabel = useGetResourceLabel();
 *
 *     return (
 *         <ul>
 *             {Object.keys(resources).map(name => (
 *                 <li key={name}>
 *                     {getResourceLabel(name, 2)}
 *                 </li>
 *             ))}
 *         </ul>
 *     )
 * }
 */
export var useGetResourceLabel = function () {
    var translate = useTranslate();
    var definitions = useResourceDefinitions();
    return function (resource, count) {
        if (count === void 0) { count = 2; }
        var resourceDefinition = definitions[resource];
        var label = translate("resources.".concat(resource, ".name"), {
            smart_count: count,
            _: resourceDefinition &&
                resourceDefinition.options &&
                resourceDefinition.options.label
                ? translate(resourceDefinition.options.label, {
                    smart_count: count,
                    _: resourceDefinition.options.label,
                })
                : humanize(count > 1
                    ? pluralize(resource)
                    : singularize(resource)),
        });
        return label;
    };
};
//# sourceMappingURL=useGetResourceLabel.js.map