var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __rest = (this && this.__rest) || function (s, e) {
    var t = {};
    for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p) && e.indexOf(p) < 0)
        t[p] = s[p];
    if (s != null && typeof Object.getOwnPropertySymbols === "function")
        for (var i = 0, p = Object.getOwnPropertySymbols(s); i < p.length; i++) {
            if (e.indexOf(p[i]) < 0 && Object.prototype.propertyIsEnumerable.call(s, p[i]))
                t[p[i]] = s[p[i]];
        }
    return t;
};
import { useEffect, useRef } from 'react';
import { useQuery, useQueryClient, hashKey, } from '@tanstack/react-query';
import { useDataProvider } from './useDataProvider';
import { useEvent } from '../util';
/**
 * Call the dataProvider.getMany() method and return the resolved result
 * as well as the loading state.
 *
 * The return value updates according to the request state:
 *
 * - start: { isPending: true, refetch }
 * - success: { data: [data from store], isPending: false, refetch }
 * - error: { error: [error from response], isPending: false, refetch }
 *
 * This hook will return the cached result when called a second time
 * with the same parameters, until the response arrives.
 *
 * @param {string} resource The resource name, e.g. 'posts'
 * @param {Params} params The getMany parameters { ids, meta }
 * @param {Object} options Options object to pass to the queryClient.
 * May include side effects to be executed upon success or failure, e.g. { onSuccess: () => { refresh(); } }
 *
 * @typedef Params
 * @prop params.ids The ids to get, e.g. [123, 456, 789]
 * @prop params.meta Optional meta parameters
 *
 * @returns The current request state. Destructure as { data, error, isPending, refetch }.
 *
 * @example
 *
 * import { useGetMany } from 'react-admin';
 *
 * const PostTags = ({ post }) => {
 *     const { data, isPending, error } = useGetMany(
 *         'tags',
 *         { ids: post.tags },
 *     );
 *     if (isPending) { return <Loading />; }
 *     if (error) { return <p>ERROR</p>; }
 *     return <ul>{data.map(tag =>
 *         <li key={tag.id}>{tag.name}</li>
 *     )}</ul>;
 * };
 */
export var useGetMany = function (resource, params, options) {
    if (options === void 0) { options = {}; }
    var ids = params.ids, meta = params.meta;
    var dataProvider = useDataProvider();
    var queryClient = useQueryClient();
    var queryCache = queryClient.getQueryCache();
    var _a = options.onError, onError = _a === void 0 ? noop : _a, _b = options.onSuccess, onSuccess = _b === void 0 ? noop : _b, _c = options.onSettled, onSettled = _c === void 0 ? noop : _c, enabled = options.enabled, queryOptions = __rest(options, ["onError", "onSuccess", "onSettled", "enabled"]);
    var onSuccessEvent = useEvent(onSuccess);
    var onErrorEvent = useEvent(onError);
    var onSettledEvent = useEvent(onSettled);
    var result = useQuery(__assign({ queryKey: [
            resource,
            'getMany',
            {
                ids: !ids || ids.length === 0 ? [] : ids.map(function (id) { return String(id); }),
                meta: meta,
            },
        ], queryFn: function (queryParams) {
            if (!ids || ids.length === 0) {
                // no need to call the dataProvider
                return Promise.resolve([]);
            }
            return dataProvider
                .getMany(resource, {
                ids: ids,
                meta: meta,
                signal: dataProvider.supportAbortSignal === true
                    ? queryParams.signal
                    : undefined,
            })
                .then(function (_a) {
                var data = _a.data;
                return data;
            });
        }, placeholderData: function () {
            var records = !ids || ids.length === 0
                ? []
                : ids.map(function (id) {
                    var _a, _b;
                    var queryHash = hashKey([
                        resource,
                        'getOne',
                        { id: String(id), meta: meta },
                    ]);
                    return (_b = (_a = queryCache.get(queryHash)) === null || _a === void 0 ? void 0 : _a.state) === null || _b === void 0 ? void 0 : _b.data;
                });
            if (records.some(function (record) { return record === undefined; })) {
                return undefined;
            }
            else {
                return records;
            }
        }, retry: false, enabled: enabled !== null && enabled !== void 0 ? enabled : ids != null }, queryOptions));
    var metaValue = useRef(meta);
    var resourceValue = useRef(resource);
    useEffect(function () {
        metaValue.current = meta;
    }, [meta]);
    useEffect(function () {
        resourceValue.current = resource;
    }, [resource]);
    useEffect(function () {
        if (result.data === undefined ||
            result.error != null ||
            result.isFetching)
            return;
        // optimistically populate the getOne cache
        result.data.forEach(function (record) {
            queryClient.setQueryData([
                resourceValue.current,
                'getOne',
                { id: String(record.id), meta: metaValue.current },
            ], function (oldRecord) { return oldRecord !== null && oldRecord !== void 0 ? oldRecord : record; });
        });
        onSuccessEvent(result.data);
    }, [
        queryClient,
        onSuccessEvent,
        result.data,
        result.error,
        result.isFetching,
    ]);
    useEffect(function () {
        if (result.error == null || result.isFetching)
            return;
        onErrorEvent(result.error);
    }, [onErrorEvent, result.error, result.isFetching]);
    useEffect(function () {
        if (result.status === 'pending' || result.isFetching)
            return;
        onSettledEvent(result.data, result.error);
    }, [
        onSettledEvent,
        result.data,
        result.error,
        result.status,
        result.isFetching,
    ]);
    return result;
};
var noop = function () { return undefined; };
//# sourceMappingURL=useGetMany.js.map