var _curry2 = require('./internal/_curry2');
var _map = require('./internal/_map');
var curryN = require('./curryN');
var pluck = require('./pluck');


/**
 * Accepts a converging function and a list of branching functions and returns
 * a new function. When invoked, this new function is applied to some
 * arguments, each branching function is applied to those same arguments. The
 * results of each branching function are passed as arguments to the converging
 * function to produce the return value.
 *
 * @func
 * @memberOf R
 * @since v0.4.2
 * @category Function
 * @sig (x1 -> x2 -> ... -> z) -> [(a -> b -> ... -> x1), (a -> b -> ... -> x2), ...] -> (a -> b -> ... -> z)
 * @param {Function} after A function. `after` will be invoked with the return values of
 *        `fn1` and `fn2` as its arguments.
 * @param {Array} functions A list of functions.
 * @return {Function} A new function.
 * @example
 *
 *      var add = (a, b) => a + b;
 *      var multiply = (a, b) => a * b;
 *      var subtract = (a, b) => a - b;
 *
 *      //≅ multiply( add(1, 2), subtract(1, 2) );
 *      R.converge(multiply, [add, subtract])(1, 2); //=> -3
 *
 *      var add3 = (a, b, c) => a + b + c;
 *      R.converge(add3, [multiply, add, subtract])(1, 2); //=> 4
 */
module.exports = _curry2(function converge(after, fns) {
  return curryN(Math.max.apply(Math, pluck('length', fns)), function() {
    var args = arguments;
    var context = this;
    return after.apply(context, _map(function(fn) {
      return fn.apply(context, args);
    }, fns));
  });
});
