/// <reference path="../../typings/main.d.ts" />
"use strict";
var RamlWrapper1Impl = require("../raml1/artifacts/raml10parser");
var path = require("path");
var Opt = require('../Opt');
var jsyaml = require("../raml1/jsyaml/jsyaml2lowLevel");
var hlimpl = require("../raml1/highLevelImpl");
var llimpl = require("../raml1/jsyaml/jsyaml2lowLevel");
var expander = require("../raml1/ast.core/expander");
var universeDef = require("../raml1/tools/universe");
var ramlServices = require("../raml1/definition-system/ramlServices");
var universeProvider = require("../raml1/definition-system/universeProvider");
/***
 * Load API synchronously. Detects RAML version and uses corresponding parser.
 * @param apiPath Path to API: local file system path or Web URL
 * @param options Load options
 * @return Opt&lt;Api&gt;, where Api belongs to RAML 1.0 or RAML 0.8 model.
 ***/
function loadApi(apiPath, arg1, arg2) {
    var api = loadRAMLInternal(apiPath, arg1, arg2);
    // if (false) {
    //     //TODO loaded RAML is API
    //     throw new Error("Loaded RAML is not API");
    // } else {
    return new Opt(api);
    // }
}
exports.loadApi = loadApi;
/***
 * Load RAML synchronously. Detects RAML version and uses corresponding parser.
 * @param ramlPath Path to RAML: local file system path or Web URL
 * @param options Load options
 * @return Opt&lt;RAMLLanguageElement&gt;, where RAMLLanguageElement belongs to RAML 1.0 or RAML 0.8 model.
 ***/
function loadRAML(ramlPath, arg1, arg2) {
    var result = loadRAMLInternal(ramlPath, arg1, arg2);
    return new Opt(result);
}
exports.loadRAML = loadRAML;
function loadRAMLInternal(apiPath, arg1, arg2) {
    var gotArray = Array.isArray(arg1);
    var extensionsAndOverlays = (gotArray ? arg1 : null);
    var options = (gotArray ? arg2 : arg1);
    options = options || {};
    var project = getProject(apiPath, options);
    var pr = apiPath.indexOf("://");
    var unitName = (pr != -1 && pr < 6) ? apiPath : path.basename(apiPath);
    var unit = project.unit(unitName);
    if (arg2 && !extensionsAndOverlays) {
        extensionsAndOverlays = null;
    }
    var api;
    if (unit) {
        if (extensionsAndOverlays && extensionsAndOverlays.length > 0) {
            var extensionUnits = [];
            extensionsAndOverlays.forEach(function (currentPath) {
                if (!currentPath || currentPath.trim().length == 0) {
                    throw new Error("Extensions and overlays list should contain legal file paths");
                }
            });
            extensionsAndOverlays.forEach(function (unitPath) {
                extensionUnits.push(project.unit(unitPath, path.isAbsolute(unitPath)));
            });
            //calling to perform the checks, we do not actually need the api itself
            extensionUnits.forEach(function (extensionUnit) { return toApi(extensionUnit, options); });
            api = toApi(expander.mergeAPIs(unit, extensionUnits, hlimpl.OverlayMergeMode.MERGE), options);
        }
        else {
            api = toApi(unit, options);
            api.highLevel().setMergeMode(hlimpl.OverlayMergeMode.MERGE);
        }
    }
    if (!unit) {
        throw new Error("Can not resolve :" + apiPath);
    }
    if (options.rejectOnErrors && api && api.errors().filter(function (x) { return !x.isWarning; }).length) {
        throw toError(api);
    }
    if (options.attributeDefaults != null && api) {
        api.setAttributeDefaults(options.attributeDefaults);
    }
    else if (api) {
        api.setAttributeDefaults(true);
    }
    return api;
}
/***
 * Load API asynchronously. Detects RAML version and uses corresponding parser.
 * @param apiPath Path to API: local file system path or Web URL
 * @param options Load options
 * @return Promise&lt;Api&gt;, where Api belongs to RAML 1.0 or RAML 0.8 model.
 ***/
function loadApiAsync(apiPath, arg1, arg2) {
    var ramlPromise = loadRAMLAsync(apiPath, arg1, arg2);
    return ramlPromise.then(function (loadedRaml) {
        // if (false) {
        //     //TODO check that loaded RAML is API
        //     return Promise.reject("Specified RAML is not API");
        // } else {
        return loadedRaml;
        // }
    });
}
exports.loadApiAsync = loadApiAsync;
/***
 * Load API asynchronously. Detects RAML version and uses corresponding parser.
 * @param ramlPath Path to RAML: local file system path or Web URL
 * @param options Load options
 * @return Promise&lt;RAMLLanguageElement&gt;, where RAMLLanguageElement belongs to RAML 1.0 or RAML 0.8 model.
 ***/
function loadRAMLAsync(ramlPath, arg1, arg2) {
    var gotArray = Array.isArray(arg1);
    var extensionsAndOverlays = (gotArray ? arg1 : null);
    var options = (gotArray ? arg2 : arg1);
    options = options || {};
    var project = getProject(ramlPath, options);
    var pr = ramlPath.indexOf("://");
    var unitName = (pr != -1 && pr < 6) ? ramlPath : path.basename(ramlPath);
    if (arg2 && !extensionsAndOverlays) {
        extensionsAndOverlays = null;
    }
    if (!extensionsAndOverlays || extensionsAndOverlays.length == 0) {
        return fetchAndLoadApiAsync(project, unitName, options).then(function (masterApi) {
            masterApi.highLevel().setMergeMode(hlimpl.OverlayMergeMode.MERGE);
            if (options.attributeDefaults != null && masterApi) {
                masterApi.setAttributeDefaults(options.attributeDefaults);
            }
            else if (masterApi) {
                masterApi.setAttributeDefaults(true);
            }
            return masterApi;
        });
    }
    else {
        extensionsAndOverlays.forEach(function (currentPath) {
            if (!currentPath || currentPath.trim().length == 0) {
                throw new Error("Extensions and overlays list should contain legal file paths");
            }
        });
        return fetchAndLoadApiAsync(project, unitName, options).then(function (masterApi) {
            var apiPromises = [];
            extensionsAndOverlays.forEach(function (extensionUnitPath) {
                apiPromises.push(fetchAndLoadApiAsync(project, extensionUnitPath, options));
            });
            return Promise.all(apiPromises).then(function (apis) {
                var overlayUnits = [];
                apis.forEach(function (currentApi) { return overlayUnits.push(currentApi.highLevel().lowLevel().unit()); });
                var result = expander.mergeAPIs(masterApi.highLevel().lowLevel().unit(), overlayUnits, hlimpl.OverlayMergeMode.MERGE);
                var wrapperNode = result.wrapperNode();
                if (options.attributeDefaults != null && result) {
                    wrapperNode.setAttributeDefaults(options.attributeDefaults);
                }
                else if (result) {
                    wrapperNode.setAttributeDefaults(true);
                }
                return wrapperNode.highLevel();
            }).then(function (mergedHighLevel) {
                return toApi(mergedHighLevel, options);
            });
        });
    }
}
exports.loadRAMLAsync = loadRAMLAsync;
/**
 * Gets AST node by runtime type, if runtime type matches any.
 * @param runtimeType
 */
function getLanguageElementByRuntimeType(runtimeType) {
    if (runtimeType == null) {
        return null;
    }
    var highLevelNode = runtimeType.getAdapter(ramlServices.RAMLService).getDeclaringNode();
    if (highLevelNode == null) {
        return null;
    }
    return highLevelNode.wrapperNode();
}
exports.getLanguageElementByRuntimeType = getLanguageElementByRuntimeType;
function fetchAndLoadApiAsync(project, unitName, options) {
    return llimpl.fetchIncludesAndMasterAsync(project, unitName).then(function (x) {
        try {
            var api = toApi(x, options);
            if (options.rejectOnErrors && api && api.errors().filter(function (x) { return !x.isWarning; }).length) {
                return Promise.reject(toError(api));
            }
            return api;
        }
        catch (err) {
            return Promise.reject(err);
        }
    });
}
function getProject(apiPath, options) {
    options = options || {};
    var includeResolver = options.fsResolver;
    var httpResolver = options.httpResolver;
    var projectRoot = path.dirname(apiPath);
    var project = new jsyaml.Project(projectRoot, includeResolver, httpResolver);
    return project;
}
;
function toApi(unitOrHighlevel, options, checkApisOverlays) {
    if (checkApisOverlays === void 0) { checkApisOverlays = false; }
    if (!unitOrHighlevel) {
        return null;
    }
    var unit = null;
    var highLevel = null;
    if (unitOrHighlevel.isRAMLUnit) {
        unit = unitOrHighlevel;
    }
    else {
        highLevel = unitOrHighlevel;
        unit = highLevel.lowLevel().unit();
    }
    var api;
    var contents = unit.contents();
    var ramlFirstLine = hlimpl.ramlFirstLine(contents);
    if (!ramlFirstLine) {
        throw new Error("Invalid first line. A RAML document is expected to start with '#%RAML <version> <?fragment type>'.");
    }
    var verStr = ramlFirstLine[1];
    var ramlFileType = ramlFirstLine[2];
    var typeName;
    var apiImpl;
    var ramlVersion;
    if (verStr == '0.8') {
        ramlVersion = 'RAML08';
    }
    else if (verStr == '1.0') {
        ramlVersion = 'RAML10';
    }
    if (!ramlVersion) {
        throw new Error("Unknown version of RAML expected to see one of '#%RAML 0.8' or '#%RAML 1.0'");
    }
    if (ramlVersion == 'RAML08' && checkApisOverlays) {
        throw new Error('Extensions and overlays are not supported in RAML 0.8.');
    }
    //if (!ramlFileType || ramlFileType.trim() === "") {
    //    if (verStr=='0.8') {
    //        typeName = universeDef.Universe08.Api.name;
    //        apiImpl = RamlWrapper08.ApiImpl;
    //    } else if(verStr=='1.0'){
    //        typeName = universeDef.Universe10.Api.name;
    //        apiImpl = RamlWrapper1.ApiImpl;
    //    }
    //} else if (ramlFileType === "Overlay") {
    //    apiImpl = RamlWrapper1.OverlayImpl;
    //    typeName = universeDef.Universe10.Overlay.name;
    //} else if (ramlFileType === "Extension") {
    //    apiImpl = RamlWrapper1.ExtensionImpl;
    //    typeName = universeDef.Universe10.Extension.name;
    //}
    var universe = universeProvider(ramlVersion);
    var apiType = universe.type(typeName);
    if (!highLevel) {
        highLevel = hlimpl.fromUnit(unit);
    }
    //api = new apiImpl(highLevel);
    api = highLevel.wrapperNode();
    return api;
}
;
function toError(api) {
    var error = new Error('Api contains errors.');
    error.parserErrors = api.errors();
    return error;
}
exports.toError = toError;
function loadApis1(projectRoot, cacheChildren, expandTraitsAndResourceTypes) {
    if (cacheChildren === void 0) { cacheChildren = false; }
    if (expandTraitsAndResourceTypes === void 0) { expandTraitsAndResourceTypes = true; }
    var universe = universeProvider("RAML10");
    var apiType = universe.type(universeDef.Universe10.Api.name);
    var p = new jsyaml.Project(projectRoot);
    var result = [];
    p.units().forEach(function (x) {
        var lowLevel = x.ast();
        if (cacheChildren) {
            lowLevel = llimpl.toChildCachingNode(lowLevel);
        }
        var api = new RamlWrapper1Impl.ApiImpl(new hlimpl.ASTNodeImpl(lowLevel, null, apiType, null));
        if (expandTraitsAndResourceTypes) {
            api = expander.expandTraitsAndResourceTypes(api);
        }
        result.push(api);
    });
    return result;
}
exports.loadApis1 = loadApis1;
//# sourceMappingURL=apiLoader.js.map