/// <reference path="../../../typings/main.d.ts" />
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var jsyaml = require("../jsyaml/jsyaml2lowLevel");
var json = require("../jsyaml/json2lowLevel");
var stringify = require("json-stable-stringify");
var proxy = require("../ast.core/LowLevelASTProxy");
var hl = require("../highLevelAST");
var ll = require("../lowLevelAST");
var _ = require("underscore");
var yaml = require("yaml-ast-parser");
var def = require("raml-definition-system");
var hlimpl = require("../highLevelImpl");
var path = require("path");
var fs = require("fs");
var universes = require("../tools/universe");
var universeHelpers = require("../tools/universeHelpers");
var universeProvider = require("../definition-system/universeProvider");
var services = def;
var OverloadingValidator = require("./overloadingValidator");
var expander = require("./expander");
var search = require("./search");
var rtypes = def.rt;
var util = require("../../util/textutil");
var contentprovider = require('../../util/contentprovider');
var resourceRegistry = require('../jsyaml/resourceRegistry');
var su = def.getSchemaUtils();
var mediaTypeParser = require("media-typer");
var xmlutil = require('../../util/xmlutil');
var changeCase = require('change-case');
var pluralize = require('pluralize');
var messageRegistry = require("../../../resources/errorMessages");
var LinterSettings = (function () {
    function LinterSettings() {
        this.validateNotStrictExamples = true;
    }
    return LinterSettings;
}());
var settings = new LinterSettings();
var loophole = require("loophole");
function evalInSandbox(code, thisArg, args) {
    return new loophole.Function(code).call(thisArg, args);
}
var MAX_RECURSION_LEVEL = 400;
exports.RESERVED_TEMPLATE_PARAMETERS = {
    "resourcePathName": "Part of the resource path following the rightmost \"/\"",
    "methodName": "Method name",
    "resourcePath": "Path of the resource"
};
var lintWithFile = function (customLinter, acceptor, astNode) {
    if (fs.existsSync(customLinter)) {
        try {
            var content = fs.readFileSync(customLinter).toString();
            var factr = new LinterExtensionsImpl(acceptor);
            evalInSandbox(content, factr, null);
            factr.visit(astNode);
        }
        catch (e) {
            console.log("Error in custom linter");
            console.log(e);
        }
    }
};
function checkPropertyQuard(n, v) {
    var pr = n.property();
    if (pr) {
        pr.getContextRequirements().forEach(function (x) {
            if (!n.checkContextValue(x.name, x.value, x.value)) {
                v.accept(createIssue1(messageRegistry.CONTEXT_REQUIREMENT, { name: x.name, value: x.value, propName: pr.nameId() }, n));
            }
        });
    }
    return pr;
}
;
function lintNode(astNode, acceptor) {
    var fsEnabled;
    try {
        fsEnabled = astNode.lowLevel().unit().project().fsEnabled();
    }
    catch (exception) {
        fsEnabled = true;
    }
    if (!fsEnabled) {
        return;
    }
    if (typeof fs === "undefined") {
        return;
    }
    if (!fs) {
        return;
    }
    var fsMethods = ['exists', 'readFile', 'writeFile', 'readdir', 'existsSync', 'readFileSync', 'writeFileSync', 'readdirSync'];
    var existingFsFields = Object.keys(fs);
    for (var i = 0; i < fsMethods.length; i++) {
        if (!fs[fsMethods[i]]) {
            return;
        }
    }
    var ps = astNode.lowLevel().unit().absolutePath();
    var dr = path.dirname(ps);
    var customLinter = path.resolve(dr, "raml-lint.js");
    lintWithFile(customLinter, acceptor, astNode);
    var dir = path.resolve(dr, ".raml");
    if (fs.existsSync(dir)) {
        var st = fs.statSync(dir);
        if (st.isDirectory()) {
            var files = fs.readdirSync(dir);
            files.forEach(function (x) {
                if (x.indexOf("-lint.js") != -1) {
                    lintWithFile(path.resolve(dir, x), acceptor, astNode);
                }
                //console.log(x);
            });
        }
    }
}
;
var LinterExtensionsImpl = (function () {
    function LinterExtensionsImpl(acceptor) {
        this.acceptor = acceptor;
        this.nodes = {};
    }
    LinterExtensionsImpl.prototype.error = function (w, message) {
        this.acceptor.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: message }, w.highLevel()));
    };
    LinterExtensionsImpl.prototype.errorOnProperty = function (w, property, message) {
        var pr = w.highLevel().attr(property);
        this.acceptor.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: message }, pr));
    };
    LinterExtensionsImpl.prototype.warningOnProperty = function (w, property, message) {
        var pr = w.highLevel().attr(property);
        this.acceptor.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: message }, pr, true));
    };
    LinterExtensionsImpl.prototype.warning = function (w, message) {
        this.acceptor.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: message }, w.highLevel(), true));
    };
    LinterExtensionsImpl.prototype.registerRule = function (nodeType, rule) {
        var q = this.nodes[nodeType];
        if (!q) {
            q = [];
            this.nodes[nodeType] = q;
        }
        q.push(rule);
    };
    LinterExtensionsImpl.prototype.visit = function (h) {
        var _this = this;
        var nd = h.definition();
        this.process(nd, h);
        nd.allSuperTypes().forEach(function (x) { return _this.process(x, h); });
        h.elements().forEach(function (y) { return _this.visit(y); });
    };
    LinterExtensionsImpl.prototype.process = function (d, h) {
        var _this = this;
        if (d instanceof def.NodeClass) {
            if (!d.getAdapter(services.RAMLService).getDeclaringNode()) {
                var rules = this.nodes[d.nameId()];
                if (rules) {
                    rules.forEach(function (x) { return x(h.wrapperNode(), _this); });
                }
            }
        }
    };
    return LinterExtensionsImpl;
}());
var StackNode = (function () {
    function StackNode() {
    }
    StackNode.prototype.toString = function () {
        if (this.prev) {
            return this.value + "." + this.prev.toString();
        }
        return this.value;
    };
    StackNode.prototype.last = function () {
        if (this.prev) {
            return this.prev.last();
        }
        return this;
    };
    return StackNode;
}());
function isTypeOrSchema(d) {
    return d.nameId() == universes.Universe10.TypeDeclaration.properties.type.name || d.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name;
}
function isDefaultValueProp(d) {
    if (!checkIfDomainIsUserDefined(d)) {
        return false;
    }
    return (d.nameId() == universes.Universe10.TypeDeclaration.properties.default.name);
}
function isExampleProp(d) {
    if (!checkIfDomainIsUserDefined(d)) {
        return false;
    }
    return (d.nameId() == universes.Universe10.TypeDeclaration.properties.example.name);
}
function checkIfDomainIsUserDefined(d) {
    if (!d.domain()) {
        return false;
    }
    if (d.domain().getAdapter(services.RAMLService).isUserDefined()) {
        return false;
    }
    return true;
}
function isSecuredBy(d) {
    if (!checkIfDomainIsUserDefined(d)) {
        return false;
    }
    return (d.nameId() == universes.Universe08.MethodBase.properties.securedBy.name);
}
/**
 * For descendants of templates returns template type. Returns null for all other nodes.
 */
function typeOfContainingTemplate(h) {
    var declRoot = h;
    while (true) {
        if (declRoot.definition().getAdapter(services.RAMLService).isInlinedTemplates()) {
            return declRoot.definition().nameId();
        }
        var np = declRoot.parent();
        if (!np) {
            break;
        }
        else {
            declRoot = np;
        }
    }
    return null;
}
function restrictUnknownNodeError(node) {
    var parentNode = node.parent();
    var issue = null;
    var parentDef = parentNode.definition();
    if (parentNode && parentDef instanceof def.UserDefinedClass) {
        var parentProperty = parentNode.property();
        if (universeHelpers.isIsProperty(parentProperty)
            || universeHelpers.isTypeProperty(parentProperty)) {
            var paramName = node.name();
            if (exports.RESERVED_TEMPLATE_PARAMETERS[paramName] != null) {
                //Handling reserved parameter names;
                issue = createIssue1(messageRegistry.INVALID_PARAMETER_NAME, { paramName: paramName }, node);
            }
            else {
                issue = createIssue1(messageRegistry.UNUSED_PARAMETER, { paramName: paramName }, node);
            }
        }
    }
    if (!issue) {
        var propName = node.name();
        var universe08 = universeProvider("RAML08");
        var isRAML08 = parentDef.universe().version() == universe08.version();
        if (isRAML08) {
            var parameterTypeName = universes.Universe08.Parameter.name;
            var typeDeclarationTypeName = universes.Universe10.TypeDeclaration.name;
            if (isRAML08 && parentDef.isAssignableFrom(parameterTypeName)) {
                var possibleDefs = universe08.type(parameterTypeName).allSubTypes().filter(function (x) {
                    return universes.Universe08[x.nameId()]['properties'][propName] != null;
                });
                var possibleDefsMap = {};
                for (var i = 0; i < possibleDefs.length; i++) {
                    var x = possibleDefs[i];
                    if (possibleDefsMap[x.nameId()]) {
                        continue;
                    }
                    var valueRequirements = x.valueRequirements();
                    if (!(valueRequirements && valueRequirements.length != 0)) {
                        continue;
                    }
                    var typeRquirements = valueRequirements.filter(function (x) { return x.name == 'type'; }).map(function (x) { return x.value; });
                    if (typeRquirements.length == 0) {
                        continue;
                    }
                    var runtimeName = typeRquirements[0];
                    possibleDefsMap[x.nameId()] = runtimeName;
                    x.allSubTypes().forEach(function (y) { return possibleDefs.push(y); });
                }
                var runtimeNames = Object.keys(possibleDefsMap).map(function (x) { return possibleDefsMap[x]; }).sort();
                ;
                if (runtimeNames.length > 0) {
                    var namesStr = runtimeNames.map(function (x, i) {
                        var x1 = "'" + x + "'";
                        if (i == runtimeNames.length - 1) {
                            return x1;
                        }
                        if (i == runtimeNames.length - 2) {
                            return x1 + ' or ';
                        }
                        return x1 + ', ';
                    }).join('');
                    issue = createIssue1(messageRegistry.INVALID_PROPERTY_OWNER_TYPE, { propName: propName, namesStr: namesStr }, node);
                }
            }
        }
    }
    return issue;
}
;
function validateTopLevelNodeSkippingChildren(node, v) {
    // if (!node.parent()){
    //     try {
    //         validateIncludes(<hlimpl.BasicASTNode>node, v);
    //     } finally {
    //         cleanupIncludesFlag(<hlimpl.BasicASTNode>node, v);
    //     }
    // }
    if (node.isElement()) {
        if (node.invalidSequence) {
            var pName = node.property().nameId();
            pName = changeCase.sentenceCase(pluralize.singular(pName));
            v.acceptUnique(createLLIssue1(messageRegistry.SEQUENCE_NOT_ALLOWED_10, { propName: pName }, node.lowLevel().parent().parent(), node));
        }
        var highLevelNode = node.asElement();
        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
            var hasSchemas = false;
            var hasTypes = false;
            var vv;
            highLevelNode.lowLevel().children().forEach(function (x) {
                if (x.key() == "schemas") {
                    hasSchemas = true;
                    vv = x;
                }
                if (x.key() == "types") {
                    hasTypes = true;
                }
            });
            if (hasSchemas && hasTypes) {
                v.accept(localLowLevelError(vv, highLevelNode, hl.IssueCode.ILLEGAL_PROPERTY_VALUE, false, "types and schemas are mutually exclusive", false));
            }
        }
        var hasRequireds = highLevelNode.definition().requiredProperties() && highLevelNode.definition().requiredProperties().length > 0;
        validateBasicFlat(node, v);
        //new UriParametersValidator().validate(highLevelNode,v);
        new CompositeNodeValidator().validate(highLevelNode, v);
        new TemplateCyclesDetector().validate(highLevelNode, v);
    }
    else {
        validateBasicFlat(node, v);
    }
    new OptionalPropertiesValidator().validate(node, v);
}
/**
 * Performs basic validation of a node on a single level, without proceeding to the node high-level children validation.
 * @param node
 * @param v
 * @param requiredOnly
 * @returns {boolean} - whether to continue validation after this one is finished, or there is no point for further validation.
 */
function validateBasicFlat(node, v, requiredOnly) {
    if (requiredOnly === void 0) { requiredOnly = false; }
    var parentNode = node.parent();
    var llValue = node.lowLevel().value();
    if (node.lowLevel()) {
        if (node.lowLevel().keyKind() == yaml.Kind.MAP) {
            v.accept(createIssue1(messageRegistry.NODE_KEY_IS_A_MAP, {}, node));
        }
        if (node.lowLevel().keyKind() == yaml.Kind.SEQ) {
            if (llValue == null) {
                var isPattern = false;
                if (node.isElement()) {
                    if (node.asElement().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
                        isPattern = true;
                    }
                }
                if (!isPattern) {
                    v.accept(createIssue1(messageRegistry.NODE_KEY_IS_A_SEQUENCE, {}, node));
                }
            }
        }
        if (parentNode == null) {
            node.lowLevel().errors().forEach(function (x) {
                var ps = x.mark ? x.mark.position : 0;
                var em = {
                    code: "YAML_ERROR",
                    message: x.message,
                    node: null,
                    start: ps,
                    end: ps + 1,
                    isWarning: false,
                    path: node.lowLevel().unit() == node.root().lowLevel().unit() ? null : node.lowLevel().unit().path(),
                    unit: node.lowLevel().unit()
                };
                v.accept(em);
            });
        }
    }
    if (node.isUnknown()) {
        if (node.name().indexOf("<<") != -1) {
            if (typeOfContainingTemplate(parentNode) != null) {
                new TraitVariablesValidator().validateName(node, v);
                return false;
            }
        }
        if (node.needSequence) {
            v.accept(createIssue1(messageRegistry.SEQUENCE_REQUIRED, { name: node.name() }, node));
        }
        if (node.needMap) {
            if (node.knownProperty) {
                v.accept(createIssue1(messageRegistry.PROPERTY_MUST_BE_A_MAP_10, { propName: node.knownProperty.nameId() }, node));
            }
            else {
                v.accept(createIssue1(messageRegistry.MAP_REQUIRED, {}, node));
            }
            return false;
        }
        if (node.unresolvedRef) {
            v.accept(createIssue1(messageRegistry.UNRESOLVED_REFERENCE, { ref: llValue }, node));
        }
        if (node.knownProperty) {
            //if (!node.lowLevel().)
            if (node.lowLevel().includeErrors().length == 0) {
                if (typeOfContainingTemplate(parentNode)
                    && util.startsWith(llValue, "<<")
                    && util.endsWith(llValue, ">>")) {
                    return false;
                }
                if (node.name() == "body" && node.computedValue("mediaType")) {
                    return false;
                }
                if (node.lowLevel().value() != '~') {
                    v.accept(createIssue1(messageRegistry.SCALAR_PROHIBITED, { propName: node.name() }, node));
                }
            }
        }
        else {
            var issue = restrictUnknownNodeError(node);
            if (!issue) {
                issue = createIssue1(messageRegistry.UNKNOWN_NODE, { name: node.name() }, node);
            }
            v.accept(issue);
        }
    }
    if (node.markCh() && !node.allowRecursive()) {
        if (!node.property()) {
            return false;
        }
        v.accept(createIssue1(messageRegistry.RECURSIVE_DEFINITION, { name: node.name() }, node));
        return false;
    }
    if (node.definition && node.definition().isAssignableFrom(universes.Universe10.Operation.name)) {
        var queryStringNode = node.element(universes.Universe10.Operation.properties.queryString.name);
        var queryParamsNode = node.lowLevel && _.find(node.lowLevel().children(), function (child) { return child.key && child.key() === universes.Universe10.Operation.properties.queryParameters.name; });
        if (queryStringNode && queryParamsNode) {
            v.accept(createIssue1(messageRegistry.PROPERTY_ALREADY_SPECIFIED, { propName: universes.Universe10.Operation.properties.queryParameters.name }, queryStringNode));
            v.accept(createLLIssue1(messageRegistry.PROPERTY_ALREADY_SPECIFIED, { propName: universes.Universe10.Operation.properties.queryString.name }, queryParamsNode, node));
        }
    }
    var isOverlay = node.definition && node.definition() &&
        (node.definition().key() === universes.Universe10.Overlay ||
            node.definition().key() === universes.Universe10.Extension);
    if (isOverlay) {
        validateMasterFlat(node, v, requiredOnly);
    }
    return true;
}
exports.validateBasicFlat = validateBasicFlat;
/**
 * Validates node master, but only on a single level, without recurring to high-level children.
 * @param node
 * @param acceptor
 */
function validateMasterFlat(node, acceptor, requiredOnly) {
    if (requiredOnly === void 0) { requiredOnly = false; }
    if (node.parent())
        return;
    var nodeAsElement = node.asElement();
    if (!nodeAsElement)
        return;
    if (!nodeAsElement.isAuxilary())
        return;
    var master = nodeAsElement.getMaster();
    if (!master)
        return;
    validateTopLevelNodeSkippingChildren(master, acceptor);
}
function validateBasic(node, v, requiredOnly) {
    if (requiredOnly === void 0) { requiredOnly = false; }
    if (!validateBasicFlat(node, v, requiredOnly)) {
        return;
    }
    try {
        var isOverlay = node.definition && node.definition() &&
            (node.definition().key() === universes.Universe10.Overlay ||
                node.definition().key() === universes.Universe10.Extension);
        var children = isOverlay ? node.children() : node.directChildren();
        children.filter(function (child) {
            return !requiredOnly || (child.property && child.property() && child.property().isRequired());
        }).forEach(function (x) {
            if (x && x.errorMessage) {
                var em = x.errorMessage;
                v.accept(createIssue1(em.entry, em.parameters, x.name() ? x : node));
                return;
            }
            x.validate(v);
        });
    }
    finally {
        node.unmarkCh();
    }
}
exports.validateBasic = validateBasic;
function hasTemplateArgs(node) {
    var vl = node.value();
    if (typeof vl == "string") {
        if (vl.indexOf("<<") != -1) {
            return true;
        }
    }
    var x = node.children();
    for (var i = 0; i < x.length; i++) {
        if (hasTemplateArgs(x[i])) {
            return true;
        }
    }
    return false;
}
var createLibraryIssue = function (attr, hlNode) {
    var start = hlNode.lowLevel().start();
    var usesNodes = [];
    if (start < 0) {
        var seq = hlNode.attr("key").value().split(".");
        var nodes = [];
        var parent = hlNode.parent();
        for (var _i = 0, seq_1 = seq; _i < seq_1.length; _i++) {
            var segment = seq_1[_i];
            var n = _.find(parent.elementsOfKind("uses"), function (x) { return x.attr("key") && x.attr("key").value() == segment; });
            nodes.push(n);
            parent = n.lowLevel().unit().resolve(n.attr("value").value()).highLevel().asElement();
        }
        var issues = nodes.map(function (x) { return createIssue1(messageRegistry.ISSUES_IN_THE_LIBRARY, { value: x.attr("value").value() }, x, true); });
        issues = issues.reverse();
        for (var i = 0; i < issues.length - 1; i++) {
            issues[i].extras.push(issues[i + 1]);
        }
        return issues[0];
    }
    else {
        usesNodes.push(hlNode);
    }
    return createIssue1(messageRegistry.ISSUES_IN_THE_LIBRARY, { value: attr.value() }, hlNode, true);
};
function validate(node, v) {
    if (!node.parent()) {
        try {
            validateIncludes(node, v);
        }
        finally {
            cleanupIncludesFlag(node, v);
        }
    }
    if (node.isAttr()) {
        new CompositePropertyValidator().validate(node, v);
    }
    else if (node.isElement()) {
        if (node.invalidSequence) {
            var pName = node.property().nameId();
            var propName = changeCase.sentenceCase(pluralize.singular(pName));
            v.acceptUnique(createLLIssue1(messageRegistry.SEQUENCE_NOT_ALLOWED_10, { propName: propName }, node.lowLevel().parent().parent(), node, false));
        }
        var highLevelNode = node.asElement();
        if (universeHelpers.isExampleSpecType(highLevelNode.definition())) {
            var hlChildren = highLevelNode.children();
            if (hlChildren.length == 0) {
                validateBasic(node, v, true);
                return;
            }
            var content = hlChildren.filter(function (x) {
                var propName = x.lowLevel().key();
                if (!propName) {
                    return true;
                }
                if (propName.charAt(0) == "(" && propName.charAt(propName.length - 1) == ")") {
                    return false;
                }
                return highLevelNode.definition().property(propName) == null;
            });
            if (content.length > 0) {
                validateBasic(node, v, true);
                return;
            }
        }
        if (highLevelNode.definition().isAnnotationType() || highLevelNode.property() && highLevelNode.property().nameId() == "annotations") {
            new FixedFacetsValidator().validate(highLevelNode, v);
            return;
        }
        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.UsesDeclaration.name)) {
            var vn = highLevelNode.attr(universes.Universe10.UsesDeclaration.properties.value.name);
            if (vn && vn.value()) {
                var rs = highLevelNode.lowLevel().unit().resolve(vn.value());
                if (!rs) {
                    v.accept(createIssue1(messageRegistry.INVALID_LIBRARY_PATH, { path: vn.value() }, highLevelNode, false));
                }
                else if (!resourceRegistry.isWaitingFor(vn.value())) {
                    var issues = [];
                    if (rs.contents().trim().length === 0) {
                        v.accept(createIssue1(messageRegistry.EMPTY_FILE, { path: vn.value() }, highLevelNode, false));
                        return;
                    }
                    rs.highLevel().validate(hlimpl.createBasicValidationAcceptor(issues, rs.highLevel()));
                    if (issues.length > 0) {
                        var brand = createLibraryIssue(vn, highLevelNode);
                        issues.forEach(function (x) {
                            x.unit = x.unit == null ? rs : x.unit;
                            if (!x.path) {
                                x.path = rs.absolutePath();
                            }
                        });
                        for (var _i = 0, issues_1 = issues; _i < issues_1.length; _i++) {
                            var issue = issues_1[_i];
                            var _issue = issue;
                            while (_issue.extras.length > 0) {
                                _issue = _issue.extras[0];
                            }
                            if (_issue != brand) {
                                _issue.extras.push(brand);
                            }
                            v.accept(issue);
                        }
                    }
                }
            }
        }
        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
            if (typeOfContainingTemplate(highLevelNode)) {
                if (hasTemplateArgs(highLevelNode.lowLevel())) {
                    return;
                }
            }
            highLevelNode.attrs().forEach(function (a) {
                var range = a.property().range().key();
                if (range == universes.Universe08.RelativeUriString || range == universes.Universe10.RelativeUriString) {
                    new UriValidator().validate(a, v);
                    return;
                }
                if (range == universes.Universe08.FullUriTemplateString || range == universes.Universe10.FullUriTemplateString) {
                    new UriValidator().validate(a, v);
                    return;
                }
                if (a.property().getAdapter(services.RAMLPropertyService).isKey()) {
                    var nameId = node.property() && node.property().nameId();
                    if (nameId == universes.Universe08.Resource.properties.uriParameters.name
                        || nameId == universes.Universe08.Resource.properties.baseUriParameters.name) {
                        //new UrlParameterNameValidator().validate(a, v);
                        return;
                    }
                    if (highLevelNode.property()) {
                        if (highLevelNode.property().nameId() ==
                            universes.Universe10.MethodBase.properties.body.name) {
                            new MediaTypeValidator().validate(a, v);
                            return;
                        }
                    }
                }
            });
            // if (highLevelNode.parent()&&!highLevelNode.parent().parent()){
            //     if (rtypes.builtInTypes().get(highLevelNode.name())){
            //         v.accept(createIssue(hl.IssueCode.ILLEGAL_PROPERTY_VALUE,
            //              `redefining a built in type: '${highLevelNode.name()}'`,highLevelNode));
            //     }
            // }
            new RecurrentOverlayValidator().validate(highLevelNode, v);
            new RecurrentValidateChildrenKeys().validate(highLevelNode, v);
            new NodeSpecificValidator().validate(highLevelNode, v);
            new TypeDeclarationValidator().validate(highLevelNode, v);
            return;
        }
        if (highLevelNode.definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
            var hasSchemas = false;
            var hasTypes = false;
            var vv;
            highLevelNode.lowLevel().children().forEach(function (x) {
                if (x.key() == "schemas") {
                    hasSchemas = true;
                    vv = x;
                }
                if (x.key() == "types") {
                    hasTypes = true;
                }
            });
            if (hasSchemas && hasTypes) {
                v.accept(createLLIssue1(messageRegistry.TYPES_AND_SCHEMAS_ARE_EXCLUSIVE, {}, vv, highLevelNode));
            }
        }
        var hasRequireds = highLevelNode.definition().requiredProperties() && highLevelNode.definition().requiredProperties().length > 0;
        var isAllowAny = highLevelNode.definition().getAdapter(services.RAMLService).getAllowAny();
        if (isAllowAny) {
            if (hasRequireds) {
                validateBasic(node, v, true);
            }
        }
        else {
            validateBasic(node, v);
        }
        new UriParametersValidator().validate(highLevelNode, v);
        new CompositeNodeValidator().validate(highLevelNode, v);
        new TemplateCyclesDetector().validate(highLevelNode, v);
    }
    else {
        validateBasic(node, v);
    }
    new OptionalPropertiesValidator().validate(node, v);
}
exports.validate = validate;
function cleanupIncludesFlag(node, v) {
    if (!node.lowLevel()) {
        return;
    }
    var val = node.lowLevel().actual();
    delete val._inc;
    node.children().forEach(function (x) { return cleanupIncludesFlag(x, v); });
}
function validateIncludes(node, v) {
    var llNode = node.lowLevel();
    if (!llNode) {
        return;
    }
    var val = llNode.actual();
    if (val._inc) {
        return;
    }
    if (node.isElement()) {
        var vl = node.name();
        if (typeof vl == "string") {
            if (vl != null && vl.indexOf(" ") != -1) {
                v.accept(createIssue1(messageRegistry.SPACES_IN_KEY, { value: vl }, node, true));
            }
        }
    }
    val._inc = true;
    if (llNode) {
        llNode.includeErrors().forEach(function (x) {
            var isWarn = false;
            if (node.lowLevel().hasInnerIncludeError()) {
                isWarn = true;
            }
            var em = createIssue1(messageRegistry.INCLUDE_ERROR, { msg: x }, node, isWarn);
            v.accept(em);
        });
        var includePath = llNode.includePath();
        if (includePath != null && !path.isAbsolute(includePath) && !ll.isWebPath(includePath)) {
            var unitPath = llNode.unit().absolutePath();
            var exceeding = calculateExceeding(path.dirname(unitPath), includePath);
            if (exceeding > 0) {
                var em = createIssue1(messageRegistry.PATH_EXCEEDS_ROOT, {}, node, true);
                v.accept(em);
            }
        }
    }
    node.children().forEach(function (x) { return validateIncludes(x, v); });
    if (node.children().length == 0 && llNode != null) {
        llNode.children().forEach(function (x) { return validateIncludesLL(x, v, node); });
    }
}
function validateIncludesLL(llNode, v, node) {
    llNode.includeErrors().forEach(function (x) {
        var isWarn = false;
        if (llNode.hasInnerIncludeError()) {
            isWarn = true;
        }
        var em = createLLIssue1(messageRegistry.INCLUDE_ERROR, { msg: x }, llNode, node, isWarn);
        v.accept(em);
    });
    var includePath = llNode.includePath();
    if (includePath != null && !path.isAbsolute(includePath) && !ll.isWebPath(includePath)) {
        var unitPath = llNode.unit().absolutePath();
        var exceeding = calculateExceeding(path.dirname(unitPath), includePath);
        if (exceeding > 0) {
            var em = createLLIssue1(messageRegistry.PATH_EXCEEDS_ROOT, {}, llNode, node, true);
            v.accept(em);
        }
    }
    llNode.children().forEach(function (x) { return validateIncludesLL(x, v, node); });
}
var actualSegments = function (rootPath) {
    rootPath = rootPath.replace(/\\/g, "/").trim();
    if (rootPath.length > 1 && rootPath.charAt(1) == ":" && /^win/.test(process.platform)) {
        rootPath = rootPath.substring(2);
    }
    var segments = rootPath.split("/");
    if (segments[0].length == 0) {
        segments = segments.slice(1);
    }
    if (segments.length > 0 && segments[segments.length - 1].length == 0) {
        segments = segments.slice(0, segments.length - 1);
    }
    return segments;
};
function calculateExceeding(rootPath, relativePath) {
    var rootSegments = actualSegments(rootPath);
    var relativeSegments = actualSegments(relativePath);
    var count = rootSegments.length;
    var result = 0;
    for (var _i = 0, relativeSegments_1 = relativeSegments; _i < relativeSegments_1.length; _i++) {
        var segment = relativeSegments_1[_i];
        if (segment == "..") {
            count--;
            if (count < 0) {
                result = Math.min(count, result);
            }
        }
        else {
            count++;
        }
    }
    return -1 * result;
}
var validateRegexp = function (cleanedValue, v, node) {
    try {
        new RegExp(cleanedValue);
    }
    catch (Error) {
        v.accept(createIssue1(messageRegistry.ILLEGAL_PATTERN, { value: cleanedValue }, node));
    }
};
var TraitVariablesValidator = (function () {
    function TraitVariablesValidator() {
    }
    TraitVariablesValidator.prototype.validateName = function (node, acceptor) {
        var name = node.name();
        if (name) {
            var start = node.lowLevel().keyStart();
            this.check(name, start, node, acceptor);
        }
    };
    TraitVariablesValidator.prototype.validateValue = function (node, acceptor) {
        var value = node.value();
        if (typeof (value) === 'string') {
            var start = node.lowLevel().valueStart();
            this.check(value, start, node, acceptor);
        }
    };
    TraitVariablesValidator.prototype.hasTraitOrResourceTypeParent = function (node) {
        var parent = node.parent();
        while (parent != null) {
            if (!parent.definition())
                return false;
            if (universeHelpers.isTraitType(parent.definition())
                || universeHelpers.isResourceTypeType(parent.definition())) {
                return true;
            }
            parent = parent.parent();
        }
        return false;
    };
    TraitVariablesValidator.prototype.check = function (str, start, node, acceptor) {
        if (!this.hasTraitOrResourceTypeParent(node))
            return [];
        var errors = [];
        var prev = 0;
        for (var i = str.indexOf('<<'); i >= 0; i = str.indexOf('<<', prev)) {
            var i0 = i;
            i += '<<'.length;
            prev = str.indexOf('>>', i);
            var paramOccurence = str.substring(i, prev);
            var ind = paramOccurence.indexOf('|');
            var paramName = ind >= 0 ? paramOccurence.substring(0, ind) : paramOccurence;
            if (paramName.trim().length == 0) {
                var msg = "Trait or resource type parameter name must contain non whitespace characters";
                var issue = createIssue1(messageRegistry.TEMPLATE_PARAMETER_NAME_MUST_CONTAIN_NONWHITESPACE_CHARACTERS, {}, node);
                issue.start = start + i;
                issue.end = start + prev;
                acceptor.accept(issue);
            }
            if (ind != -1) {
                ind++;
                var transformerNames = paramOccurence.split("|").slice(1).map(function (x) { return x.trim(); });
                var functionNames = expander.getTransformNames();
                for (var _i = 0, transformerNames_1 = transformerNames; _i < transformerNames_1.length; _i++) {
                    var transformerName = transformerNames_1[_i];
                    if (!_.find(functionNames, function (functionName) {
                        return transformerName === functionName || transformerName === ('!' + functionName);
                    })) {
                        var issue = createIssue1(messageRegistry.UNKNOWN_FUNCTION, { transformerName: transformerName }, node, false);
                        issue.start = start + ind;
                        issue.end = start + prev;
                        acceptor.accept(issue);
                    }
                }
            }
            prev += '>>'.length;
        }
        return errors;
    };
    return TraitVariablesValidator;
}());
var MethodBodyValidator = (function () {
    function MethodBodyValidator() {
    }
    MethodBodyValidator.prototype.validate = function (node, validationAcceptor) {
        var methodNode = node.parent();
        if (!methodNode) {
            return;
        }
        if (!(methodNode.definition().isAssignableFrom(universes.Universe08.Method.name) || methodNode.definition().isAssignableFrom(universes.Universe10.Method.name))) {
            return;
        }
        var hasBody = _.find(methodNode.lowLevel() && methodNode.lowLevel().children() || [], function (child) {
            var keyValue = child.key();
            return keyValue && (universes.Universe08.MethodBase.properties.body.name === keyValue || universes.Universe10.MethodBase.properties.body.name === keyValue);
        });
        if (hasBody && _.find(MethodBodyValidator.methodsWithoutRequestBody, function (methodDisabled) { return methodNode.name() === methodDisabled; })) {
            validationAcceptor.accept(createIssue1(messageRegistry.REQUEST_BODY_DISABLED, { methodName: methodNode.name() }, methodNode));
        }
    };
    MethodBodyValidator.methodsWithoutRequestBody = ['trace'];
    return MethodBodyValidator;
}());
var CompositePropertyValidator = (function () {
    function CompositePropertyValidator() {
    }
    CompositePropertyValidator.prototype.validate = function (node, v) {
        var pr = checkPropertyQuard(node, v);
        var vl = node.value();
        if (!node.property().range().hasStructure()) {
            if (vl instanceof hlimpl.StructuredValue && !node.property().isSelfNode()) {
                //TODO THIS SHOULD BE MOVED TO TYPESYSTEM FOR STS AT SOME MOMENT
                if (isTypeOrSchema(node.property())) {
                    if (node.property().domain().key() == universes.Universe08.BodyLike) {
                        var structValue = vl;
                        var newNode = new hlimpl.ASTNodeImpl(node.lowLevel(), node.parent(), node.parent().definition().universe().type(universes.Universe08.BodyLike.name), node.property());
                        newNode.validate(v);
                        return;
                    }
                }
                if (node.parent().definition().universe().version() == "RAML10"
                    && typeOfContainingTemplate(node.parent()) != null) {
                    return;
                }
                v.accept(createIssue1(messageRegistry.SCALAR_EXPECTED, {}, node));
            }
            else {
                var vk = node.lowLevel().valueKind();
                if (node.lowLevel().valueKind() != yaml.Kind.INCLUDE_REF && !node.property().getAdapter(services.RAMLPropertyService).isKey()) {
                    if ((!node.property().isMultiValue())) {
                        var k = node.property().range().key();
                        if (k == universes.Universe08.StringType || k == universes.Universe08.MarkdownString || k == universes.Universe08.MimeType) {
                            if (vk == yaml.Kind.SEQ || vk == yaml.Kind.MAPPING || vk == yaml.Kind.MAP || ((node.property().isRequired() || node.property().nameId() == "mediaType") && (vk == null || vk === undefined))) {
                                if (!node.property().domain().getAdapter(services.RAMLService).isInlinedTemplates()) {
                                    v.accept(createIssue1(messageRegistry.STRING_EXPECTED, { propName: node.name() }, node));
                                }
                            }
                        }
                    }
                }
            }
            if (node.isAnnotatedScalar()) {
                var fvl = new FixedFacetsValidator();
                node.annotations().forEach(function (x) {
                    var vl = x.value();
                    var highLevel = vl.toHighLevel();
                    if (!highLevel) {
                        v.accept(createIssue1(messageRegistry.UNKNOWN_ANNOTATION, { aName: vl.valueName() }, x));
                    }
                    else {
                        fvl.validate(highLevel, v);
                    }
                });
            }
        }
        var refName;
        if (typeof vl == 'string') {
            refName = vl;
        }
        else if (vl instanceof hlimpl.StructuredValue) {
            refName = vl.valueName();
        }
        if (refName && refName.indexOf("<<") != -1) {
            if (refName.indexOf(">>") > refName.indexOf("<<")) {
                new TraitVariablesValidator().validateValue(node, v);
                if (typeOfContainingTemplate(node.parent()) != null) {
                    return;
                }
            }
        }
        new MethodBodyValidator().validate(node, v);
        if ((node.property().range().key() == universes.Universe08.MimeType ||
            node.property().range().key() == universes.Universe10.MimeType) ||
            (node.property().nameId() == universes.Universe10.TypeDeclaration.properties.name.name
                && node.parent().property().nameId() ==
                    universes.Universe10.MethodBase.properties.body.name)) {
            new MediaTypeValidator().validate(node, v);
            return;
        }
        if (isExampleProp(node.property()) || isDefaultValueProp(node.property())) {
            if (node.definition().universe().version() == "RAML08") {
                var llv = node.lowLevel().value();
                if (node.lowLevel().children().length > 0) {
                    var valName = isExampleProp(node.property()) ? "'example'" : "'defaultValue'";
                    v.accept(createIssue1(messageRegistry.STRING_EXPECTED_2, { propName: valName }, node, false));
                }
            }
            new ExampleAndDefaultValueValidator().validate(node, v);
        }
        if (isSecuredBy(node.property())) {
            if (node.definition().universe().version() == "RAML08") {
                var np = node.lowLevel().parent();
                var ysc = yaml.Kind.SEQ;
                if (node.lowLevel() instanceof proxy.LowLevelProxyNode) {
                    if (np.valueKind() != ysc) {
                        v.accept(createIssue1(messageRegistry.SECUREDBY_LIST_08, {}, node, false));
                    }
                }
                else {
                    if (np.kind() != ysc) {
                        v.accept(createIssue1(messageRegistry.SECUREDBY_LIST_08, {}, node, false));
                    }
                }
            }
            new ExampleAndDefaultValueValidator().validate(node, v);
        }
        if (node.property().nameId() == universes.Universe10.TypeDeclaration.properties.name.name) {
            //TODO MOVE TO DEF SYSTEM
            var nameId = node.parent().property() && node.parent().property().nameId();
            if (nameId == universes.Universe08.Resource.properties.uriParameters.name
                || nameId == universes.Universe08.Resource.properties.baseUriParameters.name) {
                //                    new UrlParameterNameValidator().validate(node, v);
                return;
            }
        }
        var range = node.property().range().key();
        if (range == universes.Universe08.RelativeUriString || range == universes.Universe10.RelativeUriString) {
            new UriValidator().validate(node, v);
            return;
        }
        if (range == universes.Universe08.FullUriTemplateString || range == universes.Universe10.FullUriTemplateString) {
            new UriValidator().validate(node, v);
            return;
        }
        if ("pattern" == node.name() && universes.Universe10.StringType == node.definition().key()
            && node.parent().definition().isAssignableFrom("StringTypeDeclaration")) {
            validateRegexp(node.value(), v, node);
        }
        if ("name" == node.name() && universes.Universe10.StringType == node.definition().key()
            && (typeof node.value() == "string")
            && node.value().indexOf("[") == 0
            && node.value().lastIndexOf("]") == node.value().length - 1) {
            if (node.parent() instanceof hlimpl.ASTNodeImpl &&
                universes.Universe10.ObjectTypeDeclaration.properties.properties.name == node.parent().property().nameId()) {
                if (node.parent().parent() instanceof hlimpl.ASTNodeImpl &&
                    universes.Universe10.ObjectTypeDeclaration == node.parent().parent().definition().key()) {
                    var cleanedValue = node.value().substr(1, node.value().length - 2);
                    validateRegexp(cleanedValue, v, node);
                }
            }
        }
        if (pr.isReference() || pr.isDescriminator()) {
            new DescriminatorOrReferenceValidator().validate(node, v);
        }
        else {
            new NormalValidator().validate(node, v);
        }
    };
    return CompositePropertyValidator;
}());
function isValid(t, h, value, p, attr) {
    if (t.hasArrayInHierarchy()) {
        return isValidArray(t, h, value, p, attr);
    }
    else if (t.hasValueTypeInHierarchy()) {
        return isValidValueType(t, h, value, p, attr);
    }
    return true;
}
exports.isValid = isValid;
function isValidArray(t, h, v, p, attr) {
    if (t.arrayInHierarchy().componentType()) {
        return isValid(t.arrayInHierarchy().componentType(), h, v, p);
    }
    return true;
}
var ValidationError = (function (_super) {
    __extends(ValidationError, _super);
    function ValidationError(messageEntry, parameters) {
        if (parameters === void 0) { parameters = {}; }
        _super.call(this);
        this.messageEntry = messageEntry;
        this.parameters = parameters;
    }
    return ValidationError;
}(Error));
function isValidValueType(t, h, v, p, attr) {
    //FIXME
    try {
        if (t.key() == universes.Universe10.AnnotationRef) {
            var targets = search.referenceTargets(p, h);
            var actualAnnotation = _.find(targets, function (x) { return hlimpl.qName(x, h) == v; });
            if (actualAnnotation != null) {
                var attrs = actualAnnotation.attributes("allowedTargets");
                if (attrs) {
                    var aVals = attrs.map(function (x) { return x.value(); });
                    if (aVals.length > 0) {
                        var found = false;
                        //no we should actually check that we are applying annotation properly
                        var tps = h.definition().allSuperTypes();
                        tps = tps.concat([h.definition()]);
                        var tpNames = tps.map(function (x) { return x.nameId(); });
                        aVals.forEach(function (x) {
                            //FIXME this is deeply wrong code
                            if (x == "API") {
                                x = "Api";
                            }
                            if (x == "NamedExample") {
                                x = "ExampleSpec";
                            }
                            if (x == "SecurityScheme") {
                                x = "AbstractSecurityScheme";
                            }
                            if (x == "SecuritySchemeSettings") {
                                x = "SecuritySchemeSettings";
                            }
                            if (_.find(tpNames, function (y) { return y == x; })) {
                                found = true;
                            }
                            else {
                                if (x == "Parameter") {
                                    if (h.computedValue("location")) {
                                        found = true;
                                    }
                                }
                                if (x == "Field") {
                                    if (h.computedValue("field")) {
                                        found = true;
                                    }
                                }
                            }
                        });
                        if (!found) {
                            var list = aVals.map(function (x) { return ("'" + x + "'"); }).join(", ");
                            return new ValidationError(messageRegistry.INVALID_ANNOTATION_LOCATION, { aName: v, aValues: list });
                        }
                    }
                }
            }
            return tm;
        }
        if (t.key() == universes.Universe08.SchemaString || t.key() == universes.Universe10.SchemaString) {
            var tm = su.createSchema(v, contentProvider(h.lowLevel()));
            if (tm instanceof Error) {
                tm.canBeRef = true;
            }
            return tm;
        }
        if (t.key() == universes.Universe08.StatusCodeString || t.key() == universes.Universe10.StatusCodeString) {
            var err = validateResponseString(v);
            if (err != null) {
                return err;
            }
        }
        if (t.key() == universes.Universe08.BooleanType || t.isAssignableFrom(universes.Universe10.BooleanType.name)) {
            if (!(v === 'true' || v === 'false' || v === true || v === false)) {
                return new ValidationError(messageRegistry.BOOLEAN_EXPECTED);
            }
            if (attr) {
                var stringValue = attr.lowLevel().value(true);
                if (!(stringValue === 'true' || stringValue === 'false')) {
                    return new ValidationError(messageRegistry.BOOLEAN_EXPECTED);
                }
            }
        }
        if (t.key() == universes.Universe08.NumberType || t.isAssignableFrom(universes.Universe10.NumberType.name)) {
            var q = parseFloat(v);
            if (isNaN(q)) {
                return new ValidationError(messageRegistry.NUMBER_EXPECTED, { propName: p.nameId() });
            }
        }
        if (t.key() == universes.Universe08.StringType || t.isAssignableFrom(universes.Universe10.StringType.name)) {
            if (v === null) {
                //checking if there is at least something in the node.
                //We have many tests and APIs with the text like 'propertyName:' without a value. I do not know if such cases are
                //actually valid, but not reporting this for now.
                if (h && p) {
                    var highLevelProperty = h.attr(p.nameId());
                    if (highLevelProperty) {
                        var lowLevelChildren = highLevelProperty.lowLevel().children();
                        if (lowLevelChildren && lowLevelChildren.length > 0) {
                            return new ValidationError(messageRegistry.STRING_EXPECTED_3, { propName: p.nameId() });
                        }
                    }
                }
            }
        }
        return true;
    }
    catch (e) {
        e.canBeRef = true; //FIXME
        return e;
    }
}
var NormalValidator = (function () {
    function NormalValidator() {
    }
    NormalValidator.prototype.validate = function (node, cb) {
        var vl = node.value();
        var pr = node.property();
        var range = pr.range();
        var dnode = range.getAdapter(services.RAMLService).getDeclaringNode();
        if (dnode && range.isUserDefined()) {
            var rof = dnode.parsedType();
            var dp = node.parent().lowLevel().dumpToObject();
            var tempVal = dp[node.parent().name()];
            var isVal = pr.canBeValue();
            var val = (isVal || (tempVal === null || tempVal === undefined)) ? tempVal : tempVal[pr.nameId()];
            var validateObject = rof.validate(val, true);
            if (!validateObject.isOk()) {
                validateObject.getErrors().forEach(function (e) { return cb.accept(createIssue(e.getCode(), e.getMessage(), node, false)); });
            }
        }
        var v = cb;
        if (node.lowLevel().keyKind() != yaml.Kind.SEQ) {
            var validation = isValid(pr.range(), node.parent(), vl, pr, node);
        }
        else {
            validation = true;
        }
        if (validation instanceof Error) {
            if (!validation.canBeRef) {
                if (validation instanceof ValidationError) {
                    var ve = validation;
                    v.accept(createIssue1(ve.messageEntry, ve.parameters, node));
                }
                else {
                    v.accept(createIssue1(messageRegistry.SCHEMA_EXCEPTION, { msg: validation.message }, node));
                }
                validation = null;
                return;
            }
        }
        if (!validation || validation instanceof Error) {
            if (pr.nameId() != 'value') {
                if (!checkReference(pr, node, vl, v)) {
                    if (pr.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name
                        || universes.Universe10.TypeDeclaration.properties.type.name) {
                        if (vl && vl.trim() && (pr.domain().key() == universes.Universe08.BodyLike
                            || pr.domain().key() == universes.Universe10.TypeDeclaration)) {
                            var testSchema = vl.trim().charAt(0); //FIXME
                            if (testSchema != '{' && testSchema != '<') {
                                return;
                            }
                        }
                    }
                    var decl = node.findReferencedValue();
                    if (decl instanceof Error) {
                        if (decl instanceof ValidationError) {
                            var ve = decl;
                            v.accept(createIssue1(ve.messageEntry, ve.parameters, node));
                        }
                        else {
                            v.accept(createIssue1(messageRegistry.SCHEMA_EXCEPTION, { msg: decl.message }, node));
                        }
                    }
                    if (!decl) {
                        if (vl) {
                            if (pr.nameId() == universes.Universe10.TypeDeclaration.properties.schema.name) {
                                var z = vl.trim();
                                if (z.charAt(0) != '{' && z.charAt(0) != '<') {
                                    if (vl.indexOf('|') != -1 || vl.indexOf('[]') != -1 || vl.indexOf("(") != -1) {
                                        return;
                                    }
                                }
                            }
                        }
                        if (validation instanceof Error && vl) {
                            if (validation instanceof ValidationError) {
                                var ve = validation;
                                v.accept(createIssue1(ve.messageEntry, ve.parameters, node));
                            }
                            else {
                                v.accept(createIssue1(messageRegistry.SCHEMA_EXCEPTION, { msg: validation.message }, node));
                            }
                            validation = null;
                            return;
                        }
                        if (node.property().isRequired() && node.value() == null) {
                            v.accept(createIssue1(messageRegistry.EMPTY_VALUE_NOT_ALLOWED, {}, node));
                        }
                        else {
                            var ck = node.lowLevel().valueKind();
                            if (ck == yaml.Kind.MAP || ck == yaml.Kind.SEQ || ck == yaml.Kind.MAPPING) {
                                v.accept(createIssue1(messageRegistry.EMPTY_VALUE_NOT_ALLOWED, {}, node));
                            }
                        }
                    }
                }
            }
            else {
                if (validation instanceof Error) {
                    var message = validation.message;
                    v.accept(createIssue1(messageRegistry.SCHEMA_ERROR, { msg: message }, node, pr.range().key() == universes.Universe08.SchemaString));
                }
                else {
                    var vl = node.value();
                    v.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: vl }, node, pr.range().key() == universes.Universe08.SchemaString));
                }
            }
        }
        var values = pr.enumOptions();
        if (values) {
            if (typeof vl !== 'string') {
                return;
            }
            if (typeof values == 'string') {
                if (values != vl) {
                    if (vl && (vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe08.AbstractSecurityScheme.properties.type.name) {
                    }
                    else {
                        v.accept(createIssue1(messageRegistry.INVALID_VALUE, { iValue: vl,
                            aValues: "'" + values + "'" }, node));
                    }
                }
            }
            else if (values.length > 0) {
                if (!_.find(values, function (x) { return x == vl; })) {
                    if (vl && (vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe08.AbstractSecurityScheme.properties.type.name) {
                    }
                    else {
                        v.accept(createIssue1(messageRegistry.INVALID_VALUE, { iValue: vl,
                            aValues: values.map(function (x) { return ("'" + x + "'"); }).join(", ") }, node));
                    }
                }
            }
        }
    };
    return NormalValidator;
}());
var UriValidator = (function () {
    function UriValidator() {
    }
    UriValidator.prototype.validate = function (node, cb) {
        try {
            var values = new UrlParameterNameValidator().parseUrl(node.value());
            if (values.some(function (x) { return x == "version"; }) && node.property().nameId() == "baseUri") {
                var version = node.root().attr("version");
                if (!version) {
                    cb.accept(createIssue1(messageRegistry.MISSING_VERSION, {}, node, false));
                }
            }
            if (values.some(function (x) { return x.length == 0; })) {
                cb.accept(createIssue1(messageRegistry.URI_PARAMETER_NAME_MISSING, {}, node, false));
            }
        }
        catch (e) {
            cb.accept(createIssue1(messageRegistry.URI_EXCEPTION, { msg: e.message }, node, false));
        }
    };
    return UriValidator;
}());
var MediaTypeValidator = (function () {
    function MediaTypeValidator() {
    }
    MediaTypeValidator.prototype.validate = function (node, cb) {
        try {
            var v = node.value();
            if (!v) {
                return;
            }
            if (v == "*/*") {
                return;
            }
            if (v.indexOf("/*") == v.length - 2) {
                v = v.substring(0, v.length - 2) + "/xxx";
            }
            if (node.parent() && node.parent().parent() && node.parent().parent().definition().isAssignableFrom(universes.Universe10.Trait.name)) {
                if (v.indexOf("<<") >= 0) {
                    return;
                }
            }
            if (v == "body") {
                if (node.parent().parent()) {
                    var ppc = node.parent().parent().definition().key();
                    if (ppc === universes.Universe08.Response || ppc === universes.Universe10.Response ||
                        node.parent().parent().definition().isAssignableFrom(universes.Universe10.MethodBase.name)) {
                        v = node.parent().computedValue("mediaType");
                    }
                }
            }
            var res = mediaTypeParser.parse(v);
            //check if type name satisfies RFC6338
            if (!res.type.match(/[\w\d][\w\d!#\$&\-\^_+\.]*/)) {
                cb.accept(createIssue1(messageRegistry.INVALID_MEDIATYPE, { mediaType: res.type }, node));
            }
        }
        catch (e) {
            cb.accept(createIssue1(messageRegistry.MEDIATYPE_EXCEPTION, { msg: e.message }, node));
        }
        if (node.value() && node.value() == ("multipart/form-data") || node.value() == ("application/x-www-form-urlencoded")) {
            if (node.parent() && node.parent().parent() && node.parent().parent().property()) {
                if (node.parent().parent().property().nameId() == universes.Universe10.MethodBase.properties.responses.name) {
                    cb.accept(createIssue1(messageRegistry.FORM_IN_RESPONSE, {}, node, true));
                }
            }
        }
        return;
    };
    return MediaTypeValidator;
}());
//class SignatureValidator implements PropertyValidator{
//    validate(node:hl.IAttribute,cb:hl.ValidationAcceptor){
//        var vl=node.value();
//        var q = vl?vl.trim():"";
//        if (q.length > 0 ) {
//            try {
//                //ramlSignature.validate(vl, node, cb);
//            }catch (e){
//                cb.accept(createIssue(hl.IssueCode.INVALID_VALUE_SCHEMA,"Error during signature parse: "+e.message,node))
//            }
//            return;
//        }
//        return;
//    }
//}
var UrlParameterNameValidator = (function () {
    function UrlParameterNameValidator() {
    }
    UrlParameterNameValidator.prototype.checkBaseUri = function (node, c, vl, v) {
        var bu = c.root().attr("baseUri");
        if (bu) {
            var tnv = bu.value();
            try {
                var pNames = this.parseUrl(tnv);
                if (!_.find(pNames, function (x) { return x == vl; })) {
                    v.accept(createIssue1(messageRegistry.UNUSED_URL_PARAMETER, { paramName: "" }, node));
                }
            }
            catch (e) {
            }
        }
        else {
            v.accept(createIssue1(messageRegistry.UNUSED_URL_PARAMETER, { paramName: "" }, node));
        }
    };
    UrlParameterNameValidator.prototype.parseUrl = function (value) {
        var result = [];
        var temp = "";
        var inPar = false;
        var count = 0;
        for (var a = 0; a < value.length; a++) {
            var c = value[a];
            if (c == '{') {
                count++;
                inPar = true;
                continue;
            }
            if (c == '}') {
                count--;
                inPar = false;
                result.push(temp);
                temp = "";
                continue;
            }
            if (inPar) {
                temp += c;
            }
        }
        if (count > 0) {
            throw new Error("Invalid resource name: unmatched '{'");
        }
        if (count < 0) {
            throw new Error("Invalid resource name: unmatched '}'");
        }
        return result;
    };
    UrlParameterNameValidator.prototype.validate = function (node, cb) {
        var vl = node.value();
        if (node.parent().property().nameId() == universes.Universe10.Api.properties.baseUri.name) {
            var c = node.parent().parent();
            this.checkBaseUri(node, c, vl, cb);
            return;
        }
        var c = node.parent().parent();
        var tn = c.name();
        if (c.definition().key() === universes.Universe10.Api ||
            c.definition().key() === universes.Universe08.Api) {
            this.checkBaseUri(node, c, vl, cb);
            return;
        }
        if (c.definition().key() == universes.Universe10.ResourceType ||
            c.definition().key() == universes.Universe08.ResourceType) {
            return;
        }
        try {
            var pNames = this.parseUrl(tn);
            var foundInLocalParameters = _.find(pNames, function (x) { return x == vl; });
            if (!foundInLocalParameters) {
                var baseUri = node.root().attr(universes.Universe10.Api.properties.baseUri.name);
                if (baseUri && node.name() === universes.Universe08.Api.properties.baseUriParameters.name) {
                    var baseUriValue = baseUri.value();
                    if (baseUriValue) {
                        pNames = this.parseUrl(baseUriValue);
                        if (pNames && pNames.length > 0) {
                            if (_.find(pNames, function (x) { return x == vl; }))
                                return;
                        }
                    }
                }
                cb.accept(createIssue1(messageRegistry.UNUSED_URL_PARAMETER, { paramName: "'" + vl + "'" }, node));
            }
        }
        catch (e) {
        }
    };
    return UrlParameterNameValidator;
}());
exports.UrlParameterNameValidator = UrlParameterNameValidator;
//TODO this should be probably moved to a more general file/module
exports.typeToName = {};
exports.typeToName[universes.Universe08.Trait.name] = "trait";
exports.typeToName[universes.Universe08.ResourceType.name] = "resource type";
exports.typeToName[universes.Universe10.Trait.name] = "trait";
exports.typeToName[universes.Universe10.ResourceType.name] = "resource type";
exports.typeToName[universes.Universe10.AbstractSecurityScheme.name] = "security scheme";
exports.typeToName[universes.Universe10.Method.name] = "method";
exports.typeToName[universes.Universe08.Method.name] = "method";
exports.typeToName[universes.Universe10.Resource.name] = "resource";
exports.typeToName[universes.Universe08.Resource.name] = "resource";
exports.typeToName[universes.Universe10.Api.name] = "api";
exports.typeToName[universes.Universe08.Api.name] = "api";
exports.typeToName[universes.Universe10.Response.name] = "response";
exports.typeToName[universes.Universe08.Response.name] = "response";
exports.typeToName[universes.Universe08.BodyLike.name] = "body";
exports.parameterPropertyToName = {};
exports.parameterPropertyToName[universes.Universe08.MethodBase.properties.headers.name] = "header";
exports.parameterPropertyToName[universes.Universe08.MethodBase.properties.queryParameters.name] = "query parameter";
exports.parameterPropertyToName[universes.Universe08.Api.properties.uriParameters.name] = "uri parameter";
exports.parameterPropertyToName[universes.Universe08.Api.properties.baseUriParameters.name] = "base uri parameter";
exports.parameterPropertyToName[universes.Universe08.BodyLike.properties.formParameters.name] = "form parameter";
exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.headers.name] = "header";
exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.queryParameters.name] = "query parameter";
exports.parameterPropertyToName[universes.Universe10.ResourceBase.properties.uriParameters.name] = "uri parameter";
exports.parameterPropertyToName[universes.Universe10.Api.properties.baseUriParameters.name] = "base uri parameter";
exports.parameterPropertyToName[universes.Universe10.MethodBase.properties.body.name] = "body";
function getHumanReadableNodeName(astNode) {
    if (!astNode)
        return null;
    if (astNode.isElement()) {
        var element = astNode;
        var definition = element.definition();
        if (definition && exports.typeToName.hasOwnProperty(definition.nameId())) {
            return exports.typeToName[definition.nameId()];
        }
        if (definition.isAssignableFrom(universes.Universe10.TypeDeclaration.name) ||
            definition.isAssignableFrom(universes.Universe08.Parameter.name)) {
            if (element.property() && exports.parameterPropertyToName.hasOwnProperty(element.property().nameId())) {
                return exports.parameterPropertyToName[element.property().nameId()];
            }
            if (element.property() && element.parent() &&
                element.property().nameId() == universes.Universe10.LibraryBase.properties.types.name &&
                element.parent().definition() && element.parent().definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
                return "type";
            }
            if (element.property() && element.parent() &&
                element.property().nameId() == universes.Universe10.LibraryBase.properties.securitySchemes.name &&
                element.parent().definition() && element.parent().definition().isAssignableFrom(universes.Universe10.LibraryBase.name)) {
                return "security scheme";
            }
        }
    }
    return null;
}
exports.getHumanReadableNodeName = getHumanReadableNodeName;
function isValidPropertyValue(pr, vl, c) {
    var node = search.declRoot(c);
    if (!node._cach) {
        node._cach = {};
    }
    var id = pr.id();
    if (pr.domain()) {
        id += pr.domain().nameId();
    }
    if (id) {
        var cached = node._cach[id];
        if (cached) {
            return cached[vl] != null;
        }
    }
    var vls = search.enumValues(pr, c);
    var mm = {};
    vls.forEach(function (x) { return mm[x] = 1; });
    if (pr.id()) {
        node._cach[id] = mm;
    }
    return mm[vl] != null;
}
function checkReference(pr, astNode, vl, cb) {
    checkTraitReference(pr, astNode, cb);
    checkResourceTypeReference(pr, astNode, cb);
    if (!vl) {
        return;
    }
    if (vl == 'null') {
        if (pr.isAllowNull()) {
            return;
        }
    }
    var adapter = pr.getAdapter(services.RAMLPropertyService);
    var valid = isValidPropertyValue(pr, vl, astNode.parent());
    if (!valid && astNode.lowLevel().unit().absolutePath() !== astNode.parent().lowLevel().unit().absolutePath()) {
        valid = isValidPropertyValue(pr, vl, hlimpl.fromUnit(astNode.lowLevel().unit()));
    }
    if (!valid) {
        if (typeof vl == 'string') {
            if ((vl.indexOf("x-") == 0) && pr.nameId() == universes.Universe10.TypeDeclaration.properties.type.name) {
                return true;
            }
        }
        var expected = (adapter.isReference && adapter.isReference() && adapter.referencesTo && adapter.referencesTo() && adapter.referencesTo().nameId && adapter.referencesTo().nameId());
        var referencedToName = exports.typeToName[expected] || nameForNonReference(astNode);
        var parameters = {
            referencedToName: referencedToName,
            ref: vl
        };
        var code = referencedToName ? messageRegistry.UNRECOGNIZED_ELEMENT
            : messageRegistry.UNRESOLVED_REFERENCE;
        var spesializedMessage = specializeReferenceError(code, pr, astNode);
        cb.accept(createIssue1(spesializedMessage, parameters, astNode, pr.range().key() === universes.Universe08.SchemaString));
        return true;
    }
    if (isDuplicateSibling(astNode) && universeHelpers.isTraitRefType(astNode.definition())) {
        cb.accept(createIssue1(messageRegistry.DUPLICATE_TRAIT_REFERENCE, { refValue: vl }, astNode));
        return true;
    }
    return false;
}
function isDuplicateSibling(attr) {
    var ramlVersion = attr.property().domain().universe().version();
    var siblingName;
    if (ramlVersion == "RAML10") {
        siblingName = stringify(json.serialize(attr.lowLevel()));
    }
    else {
        siblingName = attr.value() && attr.value().valueName && attr.value().valueName();
    }
    if (!siblingName) {
        return false;
    }
    var parent = attr.parent && attr.parent();
    if (!parent) {
        return false;
    }
    var propertyName = attr.name && attr.name();
    if (!propertyName) {
        return false;
    }
    var siblings = parent.attributes && parent.attributes(propertyName);
    if (!siblings) {
        return false;
    }
    if (siblings.length === 0) {
        return false;
    }
    var count = 0;
    siblings.forEach(function (sibling) {
        var name;
        if (ramlVersion == "RAML10") {
            siblingName = stringify(json.serialize(sibling.lowLevel()));
        }
        else {
            name = sibling.value && sibling.value() && sibling.value().valueName && sibling.value().valueName();
        }
        if (name === siblingName) {
            count++;
        }
    });
    return count > 1;
}
function checkTraitReference(property, astNode, acceptor) {
    //"is" property value must be an array
    if (!universeHelpers.isIsProperty(property)) {
        return;
    }
    var lowLevel = astNode.lowLevel();
    if (lowLevel == null) {
        return;
    }
    //trying to find "is" mapping, looking 2 nodes up max
    var isMappingNode = null;
    var lowLevelParent = lowLevel.parent();
    var lowLevelParentParent = lowLevelParent != null ? lowLevelParent.parent() : null;
    if (lowLevel.kind() == yaml.Kind.MAPPING && lowLevel.key() && lowLevel.key() == "is") {
        isMappingNode = lowLevel;
    }
    else if (lowLevelParent != null &&
        lowLevelParent.kind() == yaml.Kind.MAPPING && lowLevelParent.key() && lowLevelParent.key() == "is") {
        isMappingNode = lowLevelParent;
    }
    else if (lowLevelParentParent != null &&
        lowLevelParentParent.kind() == yaml.Kind.MAPPING && lowLevelParentParent.key() && lowLevelParentParent.key() == "is") {
        isMappingNode = lowLevelParentParent;
    }
    if (isMappingNode == null) {
        return;
    }
    //having a single value is bad
    if (isMappingNode.value() != null && (!isMappingNode.children() || isMappingNode.children().length == 0)) {
        acceptor.accept(createIssue1(messageRegistry.IS_IS_ARRAY, {}, astNode));
    }
    //only maps and scalars are allowed as direct children
    var illegalChildFound = false;
    isMappingNode.children().forEach(function (child) {
        if (child.kind() != yaml.Kind.SCALAR && child.kind() != yaml.Kind.MAP) {
            illegalChildFound = true;
        }
    });
    if (illegalChildFound) {
        acceptor.accept(createIssue1(messageRegistry.IS_IS_ARRAY, {}, astNode));
    }
}
function checkResourceTypeReference(property, astNode, acceptor) {
    if (!universeHelpers.isTypeProperty(property)) {
        return;
    }
    if (!universeHelpers.isResourceTypeRefType(astNode.definition())) {
        return;
    }
    var lowLevel = astNode.lowLevel();
    if (astNode.value() == null && lowLevel && lowLevel.children() &&
        lowLevel.children().length == 0) {
        if (lowLevel.kind() == yaml.Kind.MAPPING && lowLevel.valueKind() != null) {
            //no value, no children in the mapping, but some value, that means empty map or something like this.
            acceptor.accept(createIssue1(messageRegistry.RESOURCE_TYPE_NAME, {}, astNode));
        }
    }
    else if (astNode.value() == null && lowLevel && lowLevel.children() &&
        lowLevel.children().length > 1) {
        //more than a single resource type in a list / map
        acceptor.accept(createIssue1(messageRegistry.MULTIPLE_RESOURCE_TYPES, {}, astNode));
    }
}
/**
 * Sometimes we need a more specialized message for the bad references, which diviate from a general algorithm.
 * Like listing possible values etc.
 * This method is responsible for such cases.
 * @param originalMessage
 * @param pr
 * @param astNode
 * @returns {string}
 */
function specializeReferenceError(originalMessage, property, astNode) {
    if (property.nameId() == "type" && property.domain().universe().version() == "RAML08") {
        if (property.domain().isAssignableFrom(universes.Universe08.Parameter.name)) {
            return messageRegistry.TYPES_VARIETY_RESTRICTION;
        }
    }
    if (astNode.parent() != null && universeHelpers.isSecuritySchemaType(astNode.parent().definition())) {
        return messageRegistry.UNRECOGNIZED_SECURITY_SCHEME;
    }
    return originalMessage;
}
function nameForNonReference(astNode) {
    var propertyName = astNode && astNode.lowLevel() && astNode.lowLevel().key();
    if (propertyName === universes.Universe10.AbstractSecurityScheme.properties.type.name) {
        var domain = astNode.parent() && astNode.parent().definition() && astNode.parent().definition().nameId();
        if (domain === universes.Universe10.AbstractSecurityScheme.name) {
            return "security scheme type";
        }
    }
    else if (propertyName === universes.Universe08.BodyLike.properties.schema.name) {
        var domain = astNode.parent() && astNode.parent().definition() && astNode.parent().definition().nameId();
        if (domain === universes.Universe08.BodyLike.name) {
            return "schema";
        }
    }
}
var DescriminatorOrReferenceValidator = (function () {
    function DescriminatorOrReferenceValidator() {
    }
    DescriminatorOrReferenceValidator.prototype.validate = function (node, cb) {
        var vl = node.value();
        var valueKey = vl;
        var pr = node.property();
        if (typeof vl == 'string') {
            checkReference(pr, node, vl, cb);
            if (pr.range() instanceof def.ReferenceType) {
                var t = pr.range();
                if (true) {
                    var mockNode = jsyaml.createNode("" + vl, node.lowLevel().parent(), node.lowLevel().unit());
                    mockNode._actualNode().startPosition = node.lowLevel().valueStart();
                    mockNode._actualNode().endPosition = node.lowLevel().valueEnd();
                    var stv = new hlimpl.StructuredValue(mockNode, node.parent(), node.property());
                    var hn = stv.toHighLevel();
                    if (hn) {
                        hn.validate(cb);
                    }
                }
            }
        }
        else if (vl != null) {
            var st = vl;
            if (st) {
                valueKey = st.valueName();
                var vn = st.valueName();
                if (!checkReference(pr, node, vn, cb)) {
                    var hnode = st.toHighLevel();
                    if (hnode)
                        hnode.validate(cb);
                }
            }
            else {
                valueKey = null;
            }
        }
        else {
            //there is no value, but still a reference: calling checkReference with null value
            if (node.definition().isAssignableFrom(universes.Universe10.Reference.name)) {
                checkReference(pr, node, null, cb);
            }
        }
        if (valueKey) {
            var validation = isValid(pr.range(), node.parent(), valueKey, pr);
            if (validation instanceof Error) {
                if (validation instanceof ValidationError) {
                    var ve = validation;
                    cb.accept(createIssue1(ve.messageEntry, ve.parameters, node));
                }
                else {
                    cb.accept(createIssue1(messageRegistry.SCHEMA_EXCEPTION, { msg: validation.message }, node));
                }
                validation = null;
            }
        }
    };
    return DescriminatorOrReferenceValidator;
}());
var allowOverride = { resources: 1, queryParameters: 1, headers: 1, body: 1, methods: 1, responses: 1 };
var RAMLVersionAndFragmentValidator = (function () {
    function RAMLVersionAndFragmentValidator() {
    }
    RAMLVersionAndFragmentValidator.prototype.validate = function (node, v) {
        var u = node.universe();
        var tv = u.getTypedVersion();
        if (tv) {
            if (tv !== "0.8" && tv !== "1.0") {
                var i = createIssue1(messageRegistry.UNKNOWN_RAML_VERSION, {}, node);
                v.accept(i);
            }
            var tl = u.getOriginalTopLevelText();
            if (tl) {
                var parameters = { typeName: tl };
                if (tl != node.definition().nameId()) {
                    if (node.definition().nameId() == "Api") {
                        var i = createIssue1(messageRegistry.UNKNOWN_TOPL_LEVEL_TYPE, parameters, node);
                        v.accept(i);
                    }
                }
                else if ("Api" == u.getOriginalTopLevelText()) {
                    var i = createIssue1(messageRegistry.REDUNDANT_FRAGMENT_NAME, parameters, node);
                    v.accept(i);
                }
            }
        }
    };
    return RAMLVersionAndFragmentValidator;
}());
var RequiredPropertiesAndContextRequirementsValidator = (function () {
    function RequiredPropertiesAndContextRequirementsValidator() {
    }
    RequiredPropertiesAndContextRequirementsValidator.prototype.validate = function (node, v) {
        var _this = this;
        (node.definition()).getAdapter(services.RAMLService).getContextRequirements().forEach(function (x) {
            if (!node.checkContextValue(x.name, x.value, x.value)) {
                var parameters = { v1: x.name, v2: x.value, v3: node.definition().nameId() };
                var messageEntry = messageRegistry.CONTEXT_REQUIREMENT_VIOLATION;
                if (x.name == 'location' && x.value == "ParameterLocation.FORM") {
                    messageEntry = messageRegistry.WEB_FORMS;
                }
                v.accept(createIssue1(messageEntry, parameters, node));
            }
        });
        var t;
        var isInlinedTemplate = node.definition().getAdapter(services.RAMLService).isInlinedTemplates();
        if (isInlinedTemplate) {
            var paramsMap = {};
            for (var _i = 0, _a = node.lowLevel().children(); _i < _a.length; _i++) {
                var ch = _a[_i];
                paramsMap[ch.key()] = ch.value(true);
            }
            var templateKind = node.definition().isAssignableFrom(universes.Universe10.Trait.name) ? "trait" : "resource type";
            var unitsChain = expander.toUnits(node);
            var vt = new expander.ValueTransformer(templateKind, node.definition().nameId(), unitsChain, paramsMap);
            var parent = node.parent();
            var def = parent ? parent.definition() : node.definition();
            while (parent != null && !universeHelpers.isResourceType(def) && !universeHelpers.isMethodType(def)) {
                parent = parent.parent();
            }
            t = new expander.DefaultTransformer(parent, vt, unitsChain);
        }
        node.definition().requiredProperties().forEach(function (x) {
            if (isInlinedTemplate) {
                var paths = x.getAdapter(services.RAMLPropertyService).meta("templatePaths");
                if (paths) {
                    var parent = node.parent();
                    var hasSufficientChild = false;
                    for (var _i = 0, paths_1 = paths; _i < paths_1.length; _i++) {
                        var path = paths_1[_i];
                        path = path.map(function (x) { return t.transform(x).value; });
                        if (_this.checkPathSufficiency(parent.lowLevel(), path, parent)) {
                            hasSufficientChild = true;
                            break;
                        }
                    }
                    if (!hasSufficientChild) {
                        return;
                    }
                }
            }
            var r = x.range();
            if (r.hasArrayInHierarchy()) {
                r = r.arrayInHierarchy().componentType();
            }
            if (r.hasValueTypeInHierarchy()) {
                var nm = node.attr(x.nameId());
                var gotValue = false;
                if (nm != null) {
                    if (nm.lowLevel().kind() == yaml.Kind.SCALAR
                        || nm.lowLevel().valueKind() == yaml.Kind.SCALAR
                        || nm.lowLevel().kind() == yaml.Kind.INCLUDE_REF
                        || nm.lowLevel().valueKind() == yaml.Kind.INCLUDE_REF
                        || (nm.lowLevel().valueKind() === null && !isInlinedTemplate)) {
                        //if(nm.value()!=null){
                        gotValue = true;
                    }
                    else if (nm.lowLevel().children().length != 0) {
                        gotValue = true;
                    }
                }
                if (!gotValue) {
                    var parameters = { propName: x.nameId() };
                    var messageEntry = messageRegistry.MISSING_REQUIRED_PROPERTY;
                    if (isInlinedTemplate) {
                        messageEntry = messageRegistry.VALUE_NOT_PROVIDED;
                    }
                    var i = createIssue1(messageEntry, parameters, node);
                    v.accept(i);
                }
            }
            else {
                var el = node.elementsOfKind(x.nameId());
                if (!el || el.length == 0) {
                    var i = createIssue1(messageRegistry.MISSING_REQUIRED_PROPERTY, { propName: x.nameId() }, node);
                    v.accept(i);
                }
            }
        });
    };
    RequiredPropertiesAndContextRequirementsValidator.prototype.checkPathSufficiency = function (node, path, hlParent) {
        if (hlParent == null || hlParent.definition() == null) {
            return false;
        }
        var definition = hlParent.definition();
        if (universeHelpers.isResourceTypeType(definition) || universeHelpers.isTraitType(definition)) {
            return true;
        }
        if (path.length == 0) {
            return false;
        }
        if (node == null) {
            return false;
        }
        var segment = path[0];
        if (segment == null) {
            return false;
        }
        if (segment == "/") {
            return this.checkPathSufficiency(node, path.slice(1), hlParent);
        }
        if (segment.length == 0) {
            return true;
        }
        var children = node.children().filter(function (x) { return x.key() == segment; });
        if (children.length == 0) {
            path.indexOf("/") < 0;
        }
        var lowLevel = children[0];
        if (lowLevel instanceof proxy.LowLevelCompositeNode) {
            lowLevel = lowLevel.primaryNode();
        }
        if (lowLevel == null) {
            return path.indexOf("/") < 0;
        }
        if (lowLevel.key() == "type") {
            return true;
        }
        if (path.length == 1) {
            // if(hlName==prop.nameId()&&node.definition().nameId()==prop.domain().nameId()){
            //     return true;
            // }
            return lowLevel == null || lowLevel.value() == null;
        }
        else {
            var path1 = path.slice(1);
            return this.checkPathSufficiency(lowLevel, path1, hlParent);
        }
    };
    return RequiredPropertiesAndContextRequirementsValidator;
}());
var ScalarQuoteValidator = (function () {
    function ScalarQuoteValidator() {
    }
    ScalarQuoteValidator.prototype.validate = function (node, v) {
        var r = node.lowLevel().unit();
        node.lowLevel().visit(function (x) {
            if (x.unit() != r) {
                return false;
            }
            if (x.value() && x._node && x._node.value) {
                if (x._node.value.doubleQuoted) {
                    var ind = (x.value() + "").indexOf(":");
                    var nl = (x.value() + "").indexOf("\n");
                    if (ind != -1 && nl != -1 && (!x.includePath() || x.includePath().length == 0)) {
                        var i = createIssue1(messageRegistry.SUSPICIOUS_DOUBLEQUOTE, { value: x.value() }, node, true);
                        i.start = x._node.value.startPosition;
                        i.end = x._node.value.endPosition;
                        if (i.start == i.end) {
                            i.end++;
                        }
                        v.accept(i);
                    }
                }
            }
            return true;
        });
    };
    return ScalarQuoteValidator;
}());
var FixedFacetsValidator = (function () {
    function FixedFacetsValidator() {
    }
    FixedFacetsValidator.prototype.validate = function (node, v) {
        var nc = node.definition();
        var dnode = nc.getAdapter(services.RAMLService).getDeclaringNode();
        if (dnode) {
            var rof = dnode.parsedType();
            var dp = node.lowLevel().dumpToObject(true);
            if (dp) {
                dp = dp[Object.keys(dp)[0]];
            }
            var validateObject = rof.validate(dp, false, false);
            if (!validateObject.isOk()) {
                validateObject.getErrors().forEach(function (e) { return v.accept(createIssue(e.getCode(), e.getMessage(), mapPath(node, e), false)); });
            }
        }
    };
    return FixedFacetsValidator;
}());
var TypeDeclarationValidator = (function () {
    function TypeDeclarationValidator() {
        this.annotables = {
            "API": true,
            "DocumentationItem": true,
            "Resource": true,
            "Method": true,
            "Response": true,
            "RequestBody": true,
            "ResponseBody": true,
            "TypeDeclaration": true,
            "Example": true,
            "ResourceType": true,
            "Trait": true,
            "SecurityScheme": true,
            "SecuritySchemeSettings": true,
            "AnnotationType": true,
            "Library": true,
            "Overlay": true,
            "Extension": true
        };
    }
    TypeDeclarationValidator.prototype.validate = function (node, v) {
        var nc = node.definition();
        var rof = node.parsedType();
        var validateObject = rof.validateType(node.types().getAnnotationTypeRegistry());
        if (!validateObject.isOk()) {
            for (var _i = 0, _a = validateObject.getErrors(); _i < _a.length; _i++) {
                var e = _a[_i];
                var n = extractLowLevelNode(e);
                var issue;
                if (n) {
                    issue = createLLIssue(e.getCode(), e.getMessage(), n, mapPath(node, e), e.isWarning(), true);
                }
                else {
                    issue = createIssue(e.getCode(), e.getMessage(), mapPath(node, e), e.isWarning());
                }
                v.accept(issue);
            }
            ;
        }
        var examplesLowLevel = node.lowLevel() && _.find(node.lowLevel().children(), function (x) { return x.key() == 'examples'; });
        if (examplesLowLevel && examplesLowLevel.valueKind && examplesLowLevel.valueKind() === yaml.Kind.SEQ) {
            issue = createLLIssue1(messageRegistry.MAP_EXPECTED, {}, examplesLowLevel, node, false);
            v.accept(issue);
        }
        if ((node.property() && universeHelpers.isAnnotationTypesProperty(node.property()))
            || hlimpl.isAnnotationTypeFragment(node)) {
            var atAttrs = node.attributes(universes.Universe10.TypeDeclaration.properties.allowedTargets.name);
            for (var _b = 0, atAttrs_1 = atAttrs; _b < atAttrs_1.length; _b++) {
                var attr = atAttrs_1[_b];
                this.checkAnnotationTarget(attr, v);
            }
        }
    };
    TypeDeclarationValidator.prototype.checkAnnotationTarget = function (attr, v) {
        var val = attr.value();
        if (val == null) {
            return;
        }
        if (typeof (val) != "string") {
            v.accept(createIssue1(messageRegistry.ANNOTATION_TARGET_MUST_BE_A_STRING, {}, attr, false));
        }
        var str = val;
        if (val.replace(/\w|\s/g, '').length > 0) {
            v.accept(createIssue1(messageRegistry.ALLOWED_TARGETS_MUST_BE_ARRAY, {}, attr, false));
        }
        else if (!this.annotables[str]) {
            v.accept(createIssue1(messageRegistry.UNSUPPORTED_ANNOTATION_TARGET, { aTarget: str }, attr, false));
        }
    };
    return TypeDeclarationValidator;
}());
function mapPath(node, e) {
    var src = e.getValidationPath();
    return findElementAtPath(node, src);
}
function extractLowLevelNode(e) {
    var pn = e.getExtra(rtypes.SOURCE_EXTRA);
    if (pn instanceof hlimpl.LowLevelWrapperForTypeSystem) {
        return pn.node();
    }
    return null;
}
function findElementAtPath(n, p) {
    if (!p) {
        return n;
    }
    var chld = n.children();
    for (var _i = 0, chld_1 = chld; _i < chld_1.length; _i++) {
        var ch = chld_1[_i];
        if (ch.isAttr() && ch.asAttr().isFromKey()) {
            continue;
        }
        if (ch.name() === p.name) {
            return findElementAtPath(ch, p.child);
        }
    }
    if (!n.lowLevel()) {
        return n;
    }
    var lchld = n.lowLevel().children();
    for (var i = 0; i < lchld.length; i++) {
        if (lchld[i].key() === p.name) {
            var nn = new hlimpl.BasicASTNode(lchld[i], n);
            return findElementAtPath(nn, p.child);
        }
    }
    if (!isNaN(p.name)) {
        if (lchld[p.name]) {
            var node = lchld[p.name];
            var nn = new hlimpl.BasicASTNode(node, n);
            return findElementAtPath(nn, p.child);
        }
    }
    return n;
}
var CompositeNodeValidator = (function () {
    function CompositeNodeValidator() {
    }
    CompositeNodeValidator.prototype.validate = function (node, acceptor) {
        if (node.definition().isAnnotationType()) {
            return;
        }
        if (node.lowLevel().keyKind() == yaml.Kind.SEQ) {
            var isPattern = node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
            if (!isPattern) {
                acceptor.accept(createIssue1(messageRegistry.NODE_KEY_IS_A_SEQUENCE, {}, node));
            }
        }
        if (node.definition().key() == universes.Universe08.GlobalSchema) {
            if (node.lowLevel().valueKind() != yaml.Kind.SCALAR) {
                var isString = false;
                if (node.lowLevel().valueKind() == yaml.Kind.ANCHOR_REF || node.lowLevel().valueKind() == yaml.Kind.INCLUDE_REF) {
                    var vl = node.lowLevel().value();
                    if (typeof vl === "string") {
                        isString = true;
                    }
                }
                if (!isString) {
                    acceptor.accept(createIssue1(messageRegistry.SCHEMA_NAME_MUST_BE_STRING, { name: node.name() }, node));
                }
            }
        }
        if (!node.parent()) {
            new RAMLVersionAndFragmentValidator().validate(node, acceptor);
            //Note: overloading validator now checks for oveloading and rejects it
            if (node.definition().key() == universes.Universe08.Api || node.definition().key() == universes.Universe10.Api) {
                new OverloadingValidator().validateApi(node.wrapperNode(), acceptor);
            }
            new ScalarQuoteValidator().validate(node, acceptor);
            lintNode(node, acceptor);
        }
        new OverlayNodesValidator().validate(node, acceptor);
        var nc = node.definition();
        if (nc.key() == universes.Universe08.BodyLike) {
            if (node.lowLevel().children().map(function (x) { return x.key(); }).some(function (x) { return x === "formParameters"; })) {
                if (node.parent() && node.parent().definition().key() == universes.Universe08.Response) {
                    var i = createIssue1(messageRegistry.FORM_PARAMS_IN_RESPONSE, {}, node);
                    acceptor.accept(i);
                }
                else if (node.lowLevel().children().map(function (x) { return x.key(); }).some(function (x) { return x === "schema" || x === "example"; })) {
                    var i = createIssue1(messageRegistry.FORM_PARAMS_WITH_EXAMPLE, {}, node);
                    acceptor.accept(i);
                }
            }
        }
        if (nc.key() == universes.Universe10.OAuth2SecuritySchemeSettings) {
            var requireUrl = false;
            node.attributes("authorizationGrants").forEach(function (x) {
                var vl = x.value();
                if (vl === "authorization_code" || vl === "implicit") {
                    requireUrl = true;
                }
                else if (vl !== "password" && vl !== 'client_credentials') {
                    if (vl && typeof vl === "string" && vl.indexOf("://") == -1 && vl.indexOf(":") == -1) {
                        var i = createIssue1(messageRegistry.AUTHORIZATION_GRANTS_ENUM, {}, x);
                        acceptor.accept(i);
                    }
                }
            });
            if (requireUrl) {
                if (!node.attr("authorizationUri")) {
                    var i = createIssue1(messageRegistry.AUTHORIZATION_URI_REQUIRED, {}, node);
                    acceptor.accept(i);
                }
            }
        }
        //validation of enum values;
        if (node.definition().isAssignableFrom(universes.Universe08.Parameter.name)
            || node.definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name)) {
            var vls = node.attributes("enum").map(function (x) { return x.value(); });
            if (vls.length != _.uniq(vls).length) {
                var i = createIssue1(messageRegistry.REPEATING_COMPONENTS_IN_ENUM, {}, node);
                acceptor.accept(i);
            }
            if (node.definition().isAssignableFrom(universes.Universe08.NumberTypeDeclaration.name) || node.definition().isAssignableFrom(universes.Universe10.NumberTypeDeclaration.name)) {
                var isInteger = node.definition().isAssignableFrom(universes.Universe08.IntegerTypeDeclaration.name) || node.definition().isAssignableFrom(universes.Universe10.IntegerTypeDeclaration.name);
                node.attributes("enum").forEach(function (attribute) {
                    var value = isInteger ? parseInt(attribute.value()) : parseFloat(attribute.value());
                    var isValid = isInteger ? !isNaN(value) && attribute.value().indexOf('.') === -1 : !isNaN(value);
                    if (!isValid) {
                        var issue = createIssue1(isInteger
                            ? messageRegistry.INTEGER_EXPECTED
                            : messageRegistry.NUMBER_EXPECTED_2, {}, attribute);
                        acceptor.accept(issue);
                    }
                });
            }
        }
        if (universeHelpers.isResourceTypeType(node.definition())) {
            if (node.value() == null && node.lowLevel().value(true) === "null") {
                acceptor.accept(createIssue1(messageRegistry.RESOURCE_TYPE_NULL, {}, node));
            }
        }
        checkPropertyQuard(node, acceptor);
        var nodeValue = node.value();
        if ((typeof nodeValue == 'string'
            || typeof nodeValue == 'number'
            || typeof nodeValue == 'boolean')
            && !node.definition().getAdapter(services.RAMLService).allowValue()) {
            if (node.parent()) {
                if (nodeValue != '~') {
                    var i = createIssue1(messageRegistry.SCALAR_PROHIBITED_2, { name: node.name() }, node);
                    acceptor.accept(i);
                }
            }
        }
        new RequiredPropertiesAndContextRequirementsValidator().validate(node, acceptor);
        new ValidateChildrenKeys().validate(node, acceptor);
        new NodeSpecificValidator().validate(node, acceptor);
    };
    return CompositeNodeValidator;
}());
var BaseUriParameterValidator = (function () {
    function BaseUriParameterValidator() {
    }
    BaseUriParameterValidator.prototype.validate = function (node, acceptor) {
        //we cant have "version" base uri parameter
        var nameAttributeValue = node.attrValue(universes.Universe10.TypeDeclaration.properties.name.name);
        if ("version" == nameAttributeValue) {
            acceptor.accept(createIssue1(messageRegistry.VERSION_NOT_ALLOWED, {}, node));
        }
    };
    return BaseUriParameterValidator;
}());
var NodeSpecificValidatorRegistryEntry = (function () {
    /**
     *
     * @param definition - array of definitions from universes
     * @param propertyName - name of the property. May be null, then property is not tested.
     * @param assignableFrom - whether instead of direct definition comparison, the tested node will be checked
     * for assignability from the specified definitions
     */
    function NodeSpecificValidatorRegistryEntry(definitions, propertyName, validator, assignableFrom) {
        if (assignableFrom === void 0) { assignableFrom = false; }
        this.definitions = definitions;
        this.propertyName = propertyName;
        this.assignableFrom = assignableFrom;
        this.validator = validator;
    }
    /**
     * Checks whether this entry is applicable to the node. If so, invokes its validator.
     * @param node
     * @param cb
     */
    NodeSpecificValidatorRegistryEntry.prototype.validate = function (node, acceptor) {
        var nodeDefinition = node.definition();
        if (nodeDefinition == null)
            return;
        var definitionMatched = false;
        if (!this.assignableFrom) {
            definitionMatched =
                this.definitions.some(function (currentDefinition) { return currentDefinition === nodeDefinition; });
        }
        else {
            definitionMatched =
                this.definitions.some(function (currentDefinition) { return nodeDefinition.isAssignableFrom(currentDefinition.name); });
        }
        if (!definitionMatched)
            return;
        if (this.propertyName != null) {
            if (node.property() == null)
                return;
            if (node.property().nameId() != this.propertyName)
                return;
        }
        //definition and property matched, invoking validator
        this.validator.validate(node, acceptor);
    };
    return NodeSpecificValidatorRegistryEntry;
}());
/**
 * A central switch for validations specific to a particular node.
 * In future it would be nice to migrate all node-specific validation scattered around the code here.
 */
var NodeSpecificValidator = (function () {
    function NodeSpecificValidator() {
    }
    NodeSpecificValidator.createRegistry = function () {
        var result = [];
        NodeSpecificValidator.registerValidator(result, [universes.Universe10.TypeDeclaration, universes.Universe08.Parameter], universes.Universe10.Api.properties.baseUriParameters.name, new BaseUriParameterValidator(), true);
        return result;
    };
    NodeSpecificValidator.registerValidator = function (listToAddTo, definitions, propertyName, validator, assignableFrom) {
        if (assignableFrom === void 0) { assignableFrom = false; }
        var entry = new NodeSpecificValidatorRegistryEntry(definitions, propertyName, validator, assignableFrom);
        listToAddTo.push(entry);
    };
    NodeSpecificValidator.prototype.validate = function (node, acceptor) {
        NodeSpecificValidator.entries.forEach(function (entry) { return entry.validate(node, acceptor); });
    };
    NodeSpecificValidator.entries = NodeSpecificValidator.createRegistry();
    return NodeSpecificValidator;
}());
var OverlayNodesValidator = (function () {
    function OverlayNodesValidator() {
    }
    /**
     * Checks that this node is in white list and
     * makes itself and all of its children allowed to exist in overlay even
     * if there is no master counterpart
     * @param node
     * @param root
     * @returns {boolean}
     */
    OverlayNodesValidator.prototype.allowsAnyChildren = function (node, root) {
        var property = node.property();
        var definition = node.definition();
        //accepting new annotation types
        if ((universeHelpers.isAnnotationTypeType(definition) || universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
            && universeHelpers.isAnnotationTypesProperty(property))
            return true;
        //accepting new top-level type declarations
        if (node.parent() == root && universeHelpers.isTypesProperty(property)
            && universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
            return true;
        //as we allow types, it is logical to also allow schemas as "schemas are only aliases for types"
        if (universeHelpers.isSchemasProperty(property)
            && universeHelpers.isTypeDeclarationTypeOrDescendant(definition))
            return true;
        //accepting documentation
        if (node.parent() == root && universeHelpers.isDocumentationProperty(property)
            && universeHelpers.isDocumentationType(definition))
            return true;
        //accepting annotations
        if (universeHelpers.isAnnotationsProperty(property))
            return true;
        //uses allowed
        if (universeHelpers.isUsesProperty(property))
            return true;
        //examples allowed
        if (universeHelpers.isExamplesProperty(property))
            return true;
        return false;
    };
    /**
     * Checks that this node is allowed to exist in overlay even if there is no master counterpart
     * due to it or its parent being in the white list.
     * @param node
     * @param root
     */
    OverlayNodesValidator.prototype.nodeAllowedDueToParent = function (node, root) {
        var currentNode = node;
        while (currentNode != root && currentNode != null) {
            if (this.allowsAnyChildren(currentNode, root)) {
                return true;
            }
            currentNode = currentNode.parent();
        }
        return false;
    };
    OverlayNodesValidator.prototype.validate = function (node, v) {
        var root = node.root();
        var property = node.property();
        var definition = node.definition();
        //we are only validating overlays
        if (!universeHelpers.isOverlayType(root.definition()))
            return;
        //for root only validate properties
        if (node == root) {
            this.validateProperties(node, v);
            return;
        }
        //we have a whitelist of IHighLevelNodes allowed to be added in an overlay like new types, annotation types,
        //annotation etc. The contents of such nodes is checked here.
        if (this.nodeAllowedDueToParent(node, root))
            return;
        //checking for a node, this node overrides
        var overrides = root.knownIds();
        if (!overrides) {
            //should never happen
            return;
        }
        var override = overrides.hasOwnProperty(node.id());
        if (override) {
            //overrides are allowed, but we need to check properties, this override potentially brings in or changes:
            this.validateProperties(node, v);
            return;
        }
        //otherwise reporting an illegal node:
        v.accept(createIssue1(messageRegistry.INVALID_OVERLAY_NODE, { nodeId: node.id() }, node));
    };
    OverlayNodesValidator.prototype.validateProperties = function (node, acceptor) {
        node.attrs().forEach(function (attribute) {
            //ignoring key properties as they are not overriding anything
            if (attribute.property().getAdapter(services.RAMLPropertyService).isKey()) {
                return;
            }
            //ignoring nodes, which are not coming from this node, but from is master chain
            if (attribute.parent() != node) {
                return;
            }
            //yes, that also happens!
            if (attribute.isElement()) {
                return;
            }
            //title allowed
            if (universeHelpers.isTitlePropertyName(attribute.name()))
                return;
            //description allowed
            if (universeHelpers.isDescriptionPropertyName(attribute.name()))
                return;
            //displayName allowed
            if (universeHelpers.isDisplayNamePropertyName(attribute.name()))
                return;
            //usage allowed
            if (universeHelpers.isUsagePropertyName(attribute.name()))
                return;
            //example allowed
            if (universeHelpers.isExampleProperty(attribute.property()))
                return;
            //masterRef allowed
            if (universeHelpers.isMasterRefProperty(attribute.property()))
                return;
            //annotations allowed
            if (universeHelpers.isAnnotationsProperty(attribute.property()))
                return;
            //uses allowed
            if (universeHelpers.isUsesProperty(attribute.property()))
                return;
            //reporting the error
            acceptor.accept(createIssue1(messageRegistry.INVALID_OVERRIDE_IN_OVERLAY, { propName: attribute.name() }, attribute));
        });
    };
    return OverlayNodesValidator;
}());
var RecurrentOverlayValidator = (function () {
    function RecurrentOverlayValidator() {
    }
    RecurrentOverlayValidator.prototype.validate = function (node, v) {
        var _this = this;
        var z = new OverlayNodesValidator();
        z.validate(node, v);
        node.directChildren().forEach(function (x) { if (x.isElement()) {
            _this.validate(x.asElement(), v);
        } });
    };
    return RecurrentOverlayValidator;
}());
var RecurrentValidateChildrenKeys = (function () {
    function RecurrentValidateChildrenKeys() {
    }
    RecurrentValidateChildrenKeys.prototype.val = function (node, v, p) {
        var _this = this;
        if (node.kind() == yaml.Kind.MAP || node.kind() == yaml.Kind.MAPPING) {
            var ms = {};
            node.children().forEach(function (x) {
                var c = x.key();
                if (c) {
                    if (ms.hasOwnProperty(c)) {
                        var issue = createIssue1(messageRegistry.KEYS_SHOULD_BE_UNIQUE, {}, p, false);
                        if (x.unit() == p.lowLevel().unit()) {
                            issue.start = x.keyStart();
                            issue.end = x.keyEnd();
                        }
                        v.accept(issue);
                    }
                    ms[c] = 1;
                }
            });
        }
        node.children().forEach(function (x) {
            _this.val(x, v, p);
        });
    };
    RecurrentValidateChildrenKeys.prototype.validate = function (node, v) {
        this.val(node.lowLevel(), v, node);
    };
    return RecurrentValidateChildrenKeys;
}());
var ValidateChildrenKeys = (function () {
    function ValidateChildrenKeys() {
    }
    ValidateChildrenKeys.prototype.validate = function (node, acceptor) {
        //validation is being performed at high level instead of low-level
        //to provide more meaningful and specific error messages
        this.validateChildElements(node, acceptor);
        var lowLevelChildren = node.lowLevel().children();
        var keyToLowLevelChildren = _.groupBy(lowLevelChildren.filter(function (x) { return x.key() != null; }), function (x) { return x.key(); });
        this.validateChildAttributes(node, keyToLowLevelChildren, acceptor);
        this.validateUnrecognizedLowLevelChildren(node, keyToLowLevelChildren, acceptor);
    };
    ValidateChildrenKeys.prototype.validateChildElements = function (node, acceptor) {
        //testing for child elements having equal keys
        var keyToElements = {};
        var childElements = node.directChildren().filter(function (x) { return x.isElement(); });
        childElements.forEach(function (childNode) {
            var childElement = childNode;
            if (childElement["_computed"]) {
                return;
            }
            if (!childElement.name()) {
                return; //handling nodes with no key (documentation)
            }
            var elementKey = childElement.name() + childElement.property().nameId();
            if (keyToElements.hasOwnProperty(elementKey)) {
                if (!childElement.isNamePatch()) {
                    keyToElements[elementKey].push(childElement);
                }
            }
            else {
                keyToElements[elementKey] = [childElement];
            }
        });
        Object.keys(keyToElements).forEach(function (key) {
            var childElements = keyToElements[key];
            if (!childElements || childElements.length < 2)
                return;
            childElements.forEach(function (childElement) {
                var message = "";
                var humanReadableName = getHumanReadableNodeName(childElement);
                var parameters = { name: childElement.name() };
                var messageEntry = messageRegistry.ALREADY_EXISTS_IN_CONTEXT;
                if (humanReadableName) {
                    parameters.capitalized = changeCase.upperCaseFirst(humanReadableName);
                    messageEntry = messageRegistry.ALREADY_EXISTS;
                }
                var issue = createIssue1(messageEntry, parameters, childElement);
                acceptor.accept(issue);
            });
        });
    };
    ValidateChildrenKeys.prototype.validateChildAttributes = function (node, keyToLowLevelChildren, acceptor) {
        var highLevelAttributes = this.getHighLevelAttributes(node);
        var nameToHighLevelAttributes = _.groupBy(highLevelAttributes, function (x) { return x.name(); });
        var allowsAnyAndHasRequireds = this.allowsAnyAndHasRequireds(node);
        Object.keys(nameToHighLevelAttributes).forEach(function (attributeName) {
            if (nameToHighLevelAttributes[attributeName].length < 2) {
                return;
            }
            var isUnknown = nameToHighLevelAttributes[attributeName][0].isUnknown();
            var isMultiValue = !isUnknown && nameToHighLevelAttributes[attributeName][0].property().isMultiValue();
            if (isMultiValue && (node.definition().isAssignableFrom(universes.Universe08.SecuritySchemeSettings.name) ||
                node.definition().isAssignableFrom(universes.Universe10.SecuritySchemeSettings.name))) {
                isMultiValue = keyToLowLevelChildren[attributeName] && keyToLowLevelChildren[attributeName].length === 1;
            }
            if ((isUnknown && allowsAnyAndHasRequireds) || !isMultiValue ||
                (isMultiValue && keyToLowLevelChildren[attributeName] != null && keyToLowLevelChildren[attributeName].length > 1)) {
                //we blame even multivalue properties if they have duplicated low-level keys as YAML forbids this
                nameToHighLevelAttributes[attributeName].forEach(function (attribute) {
                    var parameters = { propName: attribute.property() ? attribute.property().nameId() : attribute.name() };
                    var messageEntry = messageRegistry.PROPERTY_USED;
                    var humanReadableParent = getHumanReadableNodeName(attribute.parent());
                    if (humanReadableParent) {
                        parameters.parent = changeCase.upperCaseFirst(humanReadableParent);
                        messageEntry = messageRegistry.PARENT_PROPERTY_USED;
                    }
                    var issue = createIssue1(messageEntry, parameters, attribute);
                    acceptor.accept(issue);
                });
            }
        });
    };
    ValidateChildrenKeys.prototype.validateUnrecognizedLowLevelChildren = function (node, keyToLowLevelChildren, acceptor) {
        var highLevelChildren = node.directChildren();
        var nameToHighLevelChildren = _.groupBy(highLevelChildren, function (x) { return x.name(); });
        Object.keys(keyToLowLevelChildren).forEach(function (lowLevelChildKey) {
            if (lowLevelChildKey) {
                if (keyToLowLevelChildren[lowLevelChildKey].length > 1 && !nameToHighLevelChildren[lowLevelChildKey]) {
                    if (node.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name)) {
                        return;
                    }
                    var parameters = { propName: lowLevelChildKey };
                    var messageEntry = messageRegistry.PROPERTY_USED;
                    var humanReadableNode = getHumanReadableNodeName(node);
                    if (humanReadableNode) {
                        parameters.parent = changeCase.upperCaseFirst(humanReadableNode);
                        messageEntry = messageRegistry.PARENT_PROPERTY_USED;
                    }
                    keyToLowLevelChildren[lowLevelChildKey].forEach(function (lowLevelChild) {
                        var i = createLLIssue1(messageEntry, parameters, lowLevelChild, node);
                        i.start = lowLevelChild.keyStart();
                        i.end = lowLevelChild.keyEnd();
                        acceptor.accept(i);
                    });
                }
            }
        });
    };
    ValidateChildrenKeys.prototype.filterMultiValueAnnotations = function (node, keyToLowLevelChildren, acceptor) {
        var highLevelAttributes = this.getHighLevelAttributes(node);
        var computedAnnotationsMultiplValues = false;
        Object.keys(keyToLowLevelChildren).forEach(function (lowLevelChildKey) {
            if (lowLevelChildKey.charAt(0) !== '(' || keyToLowLevelChildren[lowLevelChildKey].length < 2) {
                return;
            }
        });
    };
    ValidateChildrenKeys.prototype.getHighLevelAttributes = function (node) {
        var allowsAnyAndHasRequireds = this.allowsAnyAndHasRequireds(node);
        return node.directChildren().filter(function (x) { return x.isAttr() || allowsAnyAndHasRequireds; });
    };
    ValidateChildrenKeys.prototype.allowsAnyAndHasRequireds = function (node) {
        var requireds = node.definition().requiredProperties();
        var hasRequireds = requireds && requireds.length > 0;
        var ramlService = node.definition().getAdapter(services.RAMLService);
        var isAllowAny = ramlService && ramlService.getAllowAny();
        var anyExceptRequireds = isAllowAny && hasRequireds;
        return anyExceptRequireds;
    };
    return ValidateChildrenKeys;
}());
function contentProvider(lowLevel) {
    var root = lowLevel && lowLevel.includeBaseUnit() && ((lowLevel.includePath && lowLevel.includePath()) ? lowLevel.includeBaseUnit().resolve(lowLevel.includePath()) : lowLevel.includeBaseUnit());
    return new contentprovider.ContentProvider(root);
}
/**
 * validates examples
 */
var ExampleAndDefaultValueValidator = (function () {
    function ExampleAndDefaultValueValidator() {
    }
    ExampleAndDefaultValueValidator.prototype.validate = function (node, cb) {
        //check if we expect to do strict validation
        var strictValidation = this.isStrict(node);
        if (!strictValidation) {
            if (!settings.validateNotStrictExamples) {
                return;
            }
        }
        var pObj = this.parseObject(node, cb, strictValidation);
        if (!pObj) {
            return;
        }
        var schema = this.aquireSchema(node);
        if (schema) {
            schema.validate(pObj, cb, strictValidation);
        }
    };
    ExampleAndDefaultValueValidator.prototype.isExampleNode = function (node) {
        return this.isSingleExampleNode(node) || this.isExampleNodeInMultipleDecl(node);
    };
    ExampleAndDefaultValueValidator.prototype.isSingleExampleNode = function (node) {
        return node.name() == universes.Universe10.TypeDeclaration.properties.example.name;
    };
    ExampleAndDefaultValueValidator.prototype.isExampleNodeInMultipleDecl = function (node) {
        var parent = node.parent();
        if (parent) {
            return universeHelpers.isExampleSpecType(parent.definition());
        }
        return false;
    };
    ExampleAndDefaultValueValidator.prototype.findParentSchemaOrTypeAttribute = function (node) {
        var attribute = node.parent().attr(universes.Universe10.TypeDeclaration.properties.schema.name);
        if (attribute) {
            return attribute;
        }
        attribute = node.parent().attr(universes.Universe10.TypeDeclaration.properties.type.name);
        if (attribute) {
            return attribute;
        }
        if (!node.parent()) {
            return null;
        }
        attribute = node.parent().parent().attr(universes.Universe10.TypeDeclaration.properties.schema.name);
        if (attribute) {
            return attribute;
        }
        attribute = node.parent().parent().attr(universes.Universe10.TypeDeclaration.properties.type.name);
        if (attribute) {
            return attribute;
        }
        return null;
    };
    ExampleAndDefaultValueValidator.prototype.aquireSchema = function (node) {
        var sp = node.parent().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name);
        if (this.isExampleNode(node)) {
            var sampleRoot = node;
            if (this.isExampleNodeInMultipleDecl(node)) {
                sampleRoot = node.parent();
            }
            if (sampleRoot.parent()) {
                if (sampleRoot.parent().definition().isAssignableFrom(universes.Universe10.TypeDeclaration.name) && sampleRoot.parent().parent() === null) {
                    sp = false;
                }
                else if (sampleRoot.parent().property().nameId() == universes.Universe10.LibraryBase.properties.types.name) {
                    sp = false;
                }
                if (sampleRoot.parent().parent()) {
                    var ppc = sampleRoot.parent().parent().definition().key();
                    if (ppc == universes.Universe08.Method || ppc == universes.Universe10.Method) {
                        if (sampleRoot.parent().property().nameId() == universes.Universe10.MethodBase.properties.queryParameters.name) {
                        }
                        else {
                            sp = true;
                        }
                    }
                    if (ppc == universes.Universe08.Response || ppc == universes.Universe10.Response) {
                        sp = true;
                    }
                }
            }
        }
        if (node.parent().definition().key() == universes.Universe08.BodyLike || sp) {
            //FIXME MULTIPLE INHERITANCE
            var sa = this.findParentSchemaOrTypeAttribute(node);
            if (sa) {
                var val = sa.value();
                if (val instanceof hlimpl.StructuredValue) {
                    return null;
                }
                var strVal = ("" + val).trim();
                var so = null;
                if (strVal.charAt(0) == "{") {
                    try {
                        so = su.getJSONSchema(strVal, contentProvider(sa.lowLevel()));
                    }
                    catch (e) {
                        return null;
                    }
                }
                if (strVal.charAt(0) == "<") {
                    try {
                        so = su.getXMLSchema(strVal);
                    }
                    catch (e) {
                        return null;
                    }
                }
                if (so) {
                    return {
                        validate: function (pObje, cb, strict) {
                            try {
                                if (pObje.__$validated) {
                                    return;
                                }
                                if (so instanceof Error) {
                                    cb.accept(createIssue1(messageRegistry.INVALID_VALUE_SCHEMA, { iValue: so.message }, node, !strict));
                                    return;
                                }
                                so.validateObject(pObje);
                            }
                            catch (e) {
                                var illegalRequiredMessageStart = "Cannot assign to read only property '__$validated' of ";
                                if (e.message && e.message.indexOf(illegalRequiredMessageStart) == 0) {
                                    var propertyName = e.message.substr(illegalRequiredMessageStart.length, e.message.length - illegalRequiredMessageStart.length);
                                    cb.accept(createIssue1(messageRegistry.INVALID_JSON_SCHEMA, { propName: propertyName }, sa, !strict));
                                    return;
                                }
                                if (e.message == "Object.keys called on non-object") {
                                    return;
                                }
                                cb.accept(createIssue1(messageRegistry.EXAMPLE_SCHEMA_FAILURE, { msg: e.message }, node, !strict));
                                return;
                            }
                            //validate using classical schema;
                        }
                    };
                }
                else {
                    if (strVal.length > 0) {
                        var nodeParent = node.parent();
                        var grandParent = nodeParent && nodeParent.parent();
                        var owner = nodeParent && nodeParent.definition() && nodeParent.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name) && nodeParent;
                        owner = owner || (grandParent && grandParent.definition() && grandParent.definition().isAssignableFrom(universes.Universe10.ObjectTypeDeclaration.name) && grandParent);
                        if (owner) {
                            return this.typeValidator(owner, node);
                        }
                    }
                }
            }
        }
        return this.getSchemaFromModel(node);
    };
    ExampleAndDefaultValueValidator.prototype.getSchemaFromModel = function (node) {
        var p = node.parent();
        // if (node.property().nameId()==universes.Universe10.ExampleSpec.properties.content.name){
        //     p=p.parent();
        // }
        return this.typeValidator(p, node);
    };
    ExampleAndDefaultValueValidator.prototype.typeValidator = function (p, node) {
        var newVar = {
            validate: function (pObje, cb, strict) {
                var pt = p.parsedType();
                if (pt && !pt.isUnknown()) {
                    if (typeof pObje === "number" && pt.isString()) {
                        pObje = "" + pObje;
                    }
                    if (typeof pObje === "boolean" && pt.isString()) {
                        pObje = "" + pObje;
                    }
                    if (pt.getExtra("repeat")) {
                        pObje = [pObje];
                    }
                    var validateObject = pt.validate(pObje, false);
                    if (!validateObject.isOk()) {
                        validateObject.getErrors().forEach(function (e) { return cb.accept(createIssue(e.getCode(), e.getMessage(), node, !strict)); });
                    }
                }
            }
        };
        return newVar;
    };
    ;
    ExampleAndDefaultValueValidator.prototype.toObject = function (h, v, cb) {
        var res = v.lowLevel().dumpToObject(true);
        this.testDublication(h, v.lowLevel(), cb);
        if (res["example"]) {
            return res["example"];
        }
        if (res["content"]) {
            return res["content"];
        }
    };
    ExampleAndDefaultValueValidator.prototype.testDublication = function (h, v, cb) {
        var _this = this;
        var map = {};
        v.children().forEach(function (x) {
            if (x.key()) {
                if (map[x.key()]) {
                    cb.accept(createIssue1(messageRegistry.KEYS_SHOULD_BE_UNIQUE, {}, new hlimpl.BasicASTNode(x, h.parent())));
                }
                map[x.key()] = x;
            }
            _this.testDublication(h, x, cb);
        });
    };
    ExampleAndDefaultValueValidator.prototype.parseObject = function (node, cb, strictValidation) {
        var pObj = null;
        var vl = node.value();
        var mediaType = getMediaType(node);
        if (vl instanceof hlimpl.StructuredValue) {
            //validate in context of type/schema
            pObj = this.toObject(node, vl, cb);
        }
        else {
            if (mediaType) {
                if (isJson(mediaType)) {
                    try {
                        pObj = JSON.parse(vl);
                    }
                    catch (e) {
                        cb.accept(createIssue1(messageRegistry.CAN_NOT_PARSE_JSON, { msg: e.message }, node, !strictValidation));
                        return;
                    }
                }
                if (isXML(mediaType)) {
                    try {
                        pObj = xmlutil(vl);
                    }
                    catch (e) {
                        cb.accept(createIssue1(messageRegistry.CAN_NOT_PARSE_XML, { msg: e.message }, node, !strictValidation));
                        return;
                    }
                }
            }
            else {
                try {
                    if (vl && vl.length > 0 && (vl.trim().charAt(0) == '[' || vl.trim().charAt(0) == '{' || vl.trim().charAt(0) == '<')) {
                        pObj = JSON.parse(vl);
                    }
                    else {
                        if (vl == "true") {
                            return true;
                        }
                        else if (vl == "false") {
                            return false;
                        }
                        else {
                            var n = parseFloat(vl);
                            if (!isNaN(n)) {
                                return n;
                            }
                        }
                        return vl;
                    }
                }
                catch (e) {
                    if (vl.trim().indexOf("<") == 0) {
                        try {
                            pObj = xmlutil(vl);
                        }
                        catch (e) {
                            cb.accept(createIssue1(messageRegistry.CAN_NOT_PARSE_XML, { msg: e.message }, node, !strictValidation));
                            return;
                        }
                    }
                    else {
                        //cb.accept(createIssue1(messageRegistry.CAN_NOT_PARSE_XML,
                        // {msg: e.message}, node, !strictValidation));
                        return vl;
                    }
                }
            }
        }
        return pObj;
    };
    ExampleAndDefaultValueValidator.prototype.isStrict = function (node) {
        if (universeHelpers.isDefaultValue(node.property())) {
            return true;
        }
        if (universeHelpers.isExampleProperty(node.property())
            && node.parent().definition().universe().version() == "RAML08") {
            //for RAML 0.8 we do not validate examples strictly
            return false;
        }
        var strictValidation = false;
        var strict = node.parent().attr("strict");
        if (strict) {
            if (strict.value() == 'true') {
                strictValidation = true;
            }
        }
        return strictValidation;
    };
    return ExampleAndDefaultValueValidator;
}());
exports.ExampleAndDefaultValueValidator = ExampleAndDefaultValueValidator;
var toReadableName = function (template, toLowerCase, pluralize_) {
    var templateName = changeCase.sentence(template);
    if (!toLowerCase) {
        templateName = changeCase.ucFirst(templateName);
    }
    if (pluralize_) {
        templateName = pluralize.plural(templateName);
    }
    return templateName;
};
var OptionalPropertiesValidator = (function () {
    function OptionalPropertiesValidator() {
    }
    OptionalPropertiesValidator.prototype.validate = function (node, v) {
        if (node.isAttr()) {
            if (!node.optional()) {
                return;
            }
            var attr = node;
            var prop = attr.property();
            if (prop.isMultiValue() || prop.range().isArray()) {
                return;
            }
            if (!prop.isFromParentKey()) {
                var template = typeOfContainingTemplate(attr.parent());
                if (template) {
                    if (prop.isValueProperty()) {
                        var templateNamePlural = toReadableName(template, true, true);
                        var issue = createIssue1(messageRegistry.OPTIONAL_SCLARAR_PROPERTIES_10, {
                            templateNamePlural: templateNamePlural,
                            propName: attr.name()
                        }, attr, false);
                        v.accept(issue);
                    }
                }
            }
        }
        else if (node.isElement()) {
            var aNode = node;
            var prop = aNode.property();
            var allowsQuestion = aNode.allowsQuestion();
            if (!allowsQuestion) {
                var propName = prop ? toReadableName(prop.nameId(), true, true) : 'API root';
                aNode.optionalProperties().forEach(function (x) {
                    aNode.children().forEach(function (y) {
                        var parameters = {
                            propName: propName,
                            oPropName: y.lowLevel().key()
                        };
                        var issue = createIssue1(messageRegistry.OPTIONAL_PROPERTIES_10, parameters, node, false);
                        v.accept(issue);
                    });
                });
            }
            var def = node.asElement().definition();
            if (node.optional() && def.universe().version() == "RAML10") {
                var prop = node.property();
                var isParam = universeHelpers.isQueryParametersProperty(prop)
                    || universeHelpers.isUriParametersProperty(prop)
                    || universeHelpers.isHeadersProperty(prop);
                if (!universeHelpers.isMethodType(def) && !(universeHelpers.isTypeDeclarationType(def) && isParam)) {
                    var issue = createIssue1(messageRegistry.ONLY_METHODS_CAN_BE_OPTIONAL, {}, node, false);
                    v.accept(issue);
                }
            }
        }
    };
    return OptionalPropertiesValidator;
}());
var UriParametersValidator = (function () {
    function UriParametersValidator() {
    }
    UriParametersValidator.prototype.validate = function (node, v) {
        var def = node.definition();
        var baseUriPropName = universes.Universe10.Api.properties.baseUri.name;
        var baseUriParamsPropName = universes.Universe10.Api.properties.baseUriParameters.name;
        var uriPropName = universes.Universe10.Resource.properties.relativeUri.name;
        var uriParamsPropName = universes.Universe10.ResourceBase.properties.uriParameters.name;
        if (universeHelpers.isApiSibling(def)) {
            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName);
        }
        else if (universeHelpers.isResourceType(def)) {
            var rootNode = node.root();
            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName, rootNode);
            this.inspectParameters(node, v, uriPropName, uriParamsPropName);
        }
        else if (universeHelpers.isResourceTypeType(def)) {
            var rootNode = node.root();
            this.inspectParameters(node, v, baseUriPropName, baseUriParamsPropName, rootNode);
        }
    };
    UriParametersValidator.prototype.inspectParameters = function (node, v, uriPropName, paramsPropName, rootNode) {
        rootNode = rootNode || node;
        var uriValue = '';
        var uriAttr = rootNode.attr(uriPropName);
        if (uriAttr) {
            uriValue = uriAttr.value();
            if (!uriValue || typeof (uriValue) != "string") {
                uriValue = '';
            }
        }
        var paramElements = node.elementsOfKind(paramsPropName);
        paramElements.forEach(function (x) {
            var nameAttr = x.attr(universes.Universe10.TypeDeclaration.properties.name.name);
            if (nameAttr) {
                var name = nameAttr.value();
                if (name != null) {
                    if (uriValue.indexOf('{' + name + '}') < 0) {
                        if (universeHelpers.isResourceTypeType(node.definition())) {
                            if (name.indexOf('<<') >= 0) {
                                return;
                            }
                        }
                        var propNameReadable = changeCase.upperCaseFirst(pluralize.singular(changeCase.sentence(paramsPropName)));
                        var issue = createIssue1(messageRegistry.PROPERTY_UNUSED, { propName: propNameReadable }, x, true);
                        v.accept(issue);
                    }
                }
            }
        });
    };
    return UriParametersValidator;
}());
var TemplateCyclesDetector = (function () {
    function TemplateCyclesDetector() {
        this.nameProperty = universes.Universe10.ResourceType.properties.name.name;
    }
    TemplateCyclesDetector.prototype.validate = function (node, v) {
        var definition = node.definition();
        if (!(universeHelpers.isLibraryBaseSibling(definition) || universeHelpers.isApiType(definition))) {
            return;
        }
        var resourceTypesProp = universes.Universe10.LibraryBase.properties.resourceTypes.name;
        var typeProp = universes.Universe10.ResourceBase.properties.type.name;
        var traitsProp = universes.Universe10.LibraryBase.properties.traits.name;
        var isProp = universes.Universe10.MethodBase.properties.is.name;
        var allResourceTypes = search.globalDeclarations(node)
            .filter(function (x) { return universeHelpers.isResourceTypeType(x.definition()); });
        var alltraits = search.globalDeclarations(node)
            .filter(function (x) { return universeHelpers.isTraitType(x.definition()); });
        this.checkCycles(allResourceTypes, typeProp, v);
        this.checkCycles(alltraits, isProp, v);
    };
    TemplateCyclesDetector.prototype.checkCycles = function (templates, propName, v) {
        var _this = this;
        var templatesMap = {};
        templates.forEach(function (x) {
            var name = _this.templateName(x);
            templatesMap[name] = x;
        });
        var templatesWithCycle = {};
        templates.forEach(function (template) {
            var name = _this.templateName(template);
            if (templatesWithCycle[name]) {
                //skip checking templates which are already known to have cycles in definition;
                return;
            }
            _this.findCyclesInDefinition(template, propName, templatesMap).forEach(function (cycle) {
                //mark templates which have cycles in definitions
                cycle.forEach(function (x) { return templatesWithCycle[x] = true; });
                cycle = cycle.reverse();
                var typeName = toReadableName(template.definition().nameId());
                var cycleStr = cycle.join(" -> ");
                var parameters = {
                    typeName: typeName,
                    cycle: cycleStr
                };
                var issue = createIssue1(messageRegistry.CYCLE_IN_DEFINITION, parameters, template, false);
                v.accept(issue);
            });
        });
    };
    TemplateCyclesDetector.prototype.templateName = function (node) {
        var nameAttribute = node.attr(this.nameProperty);
        if (!nameAttribute) {
            return null;
        }
        return nameAttribute.value();
    };
    TemplateCyclesDetector.prototype.findCyclesInDefinition = function (node, propName, templatesMap, occuredTemplates) {
        if (occuredTemplates === void 0) { occuredTemplates = {}; }
        var name = this.templateName(node);
        if (occuredTemplates[name]) {
            return [[name]];
        }
        var nextOccuredTemplates = {};
        Object.keys(occuredTemplates).forEach(function (x) { return nextOccuredTemplates[x] = occuredTemplates[x]; });
        nextOccuredTemplates[name] = true;
        var occuredCycles = [];
        var templatesRefs = node.attributes(propName);
        for (var i = 0; i < templatesRefs.length; i++) {
            var ref = templatesRefs[i];
            var val = ref.value();
            if (val) {
                var refName = typeof (val) == 'string' ? val : val.valueName();
                var template = templatesMap[refName];
                if (template != null) {
                    var newCycles = this.findCyclesInDefinition(template, propName, templatesMap, nextOccuredTemplates);
                    newCycles.forEach(function (x) { return occuredCycles.push(x); });
                }
            }
        }
        occuredCycles.forEach(function (x) { return x.push(name); });
        return occuredCycles;
    };
    return TemplateCyclesDetector;
}());
function isJson(s) {
    return s.indexOf("json") != -1;
}
exports.isJson = isJson;
function isXML(s) {
    return s.indexOf("xml") != -1;
}
exports.isXML = isXML;
function getMediaType(node) {
    var vl = getMediaType2(node);
    if (vl == 'body') {
        var rootMedia = node.root().attr("mediaType");
        if (rootMedia) {
            return rootMedia.value();
        }
        return null;
    }
    return vl;
}
exports.getMediaType = getMediaType;
function getMediaType2(node) {
    if (node.parent()) {
        var pc = node.parent().definition();
        if (pc.key() == universes.Universe08.BodyLike) {
            return node.parent().name();
        }
        if (node.parent().parent()) {
            var ppc = node.parent().parent().definition().key();
            if (ppc == universes.Universe08.Response || ppc == universes.Universe10.Response) {
                if (node.parent().property().nameId() == universes.Universe08.Response.properties.headers.name) {
                    return null;
                }
                return node.parent().name();
            }
            if (ppc == universes.Universe08.Method || ppc == universes.Universe10.Method) {
                if (node.parent().property().nameId() == universes.Universe10.MethodBase.properties.queryParameters.name
                    || node.parent().property().nameId() == universes.Universe10.MethodBase.properties.headers.name) {
                    return null;
                }
                return node.parent().name();
            }
        }
    }
    return null;
}
var localError = function (node, c, w, message, p, prop, positionsSource) {
    var llNode = positionsSource ? positionsSource : node.lowLevel();
    var contents = llNode.unit() && llNode.unit().contents();
    var contentLength = contents && contents.length;
    var st = llNode.start();
    var et = llNode.end();
    if (contentLength && contentLength < et) {
        et = contentLength - 1;
    }
    if (llNode.key() && llNode.keyStart()) {
        var ks = llNode.keyStart();
        if (ks > 0) {
            st = ks;
        }
        var ke = llNode.keyEnd();
        if (ke > 0) {
            et = ke;
        }
    }
    if (et < st) {
        et = st + 1;
        //this happens for empty APIs, when we basically have nothing to parse
        if (node.isElement()) {
            var definition = node.definition();
            if (universeHelpers.isApiType(definition)) {
                st = contentLength == 0 ? 0 : contentLength - 1;
                et = st;
            }
        }
    }
    if (prop && !prop.getAdapter(services.RAMLPropertyService).isMerged() && node.parent() == null) {
        var nm = _.find(llNode.children(), function (x) { return x.key() == prop.nameId(); });
        if (nm) {
            var ks = nm.keyStart();
            var ke = nm.keyEnd();
            if (ks > 0 && ke > ks) {
                st = ks;
                et = ke;
            }
        }
    }
    return {
        code: c,
        isWarning: w,
        message: message,
        node: node,
        start: st,
        end: et,
        path: p ? (llNode.unit() ? llNode.unit().path() : "") : null,
        extras: [],
        unit: node ? llNode.unit() : null
    };
};
var localLowLevelError = function (node, highLevelAnchor, issueCode, isWarning, message, path) {
    var contents = node.unit() && node.unit().contents();
    var contentLength = contents && contents.length;
    var st = node.start();
    var et = node.end();
    if (contentLength && contentLength >= et) {
        et = contentLength - 1;
    }
    if (node.key() && node.keyStart()) {
        var ks = node.keyStart();
        if (ks > 0) {
            st = ks;
        }
        var ke = node.keyEnd();
        if (ke > 0) {
            et = ke;
        }
    }
    return {
        code: issueCode,
        isWarning: isWarning,
        message: message,
        node: highLevelAnchor,
        start: st,
        end: et,
        path: path ? (node.unit() ? node.unit().path() : "") : null,
        extras: [],
        unit: node ? node.unit() : null
    };
};
function toIssue(error, node) {
    return createIssue(error.getCode(), error.getMessage(), mapPath(node, error), error.isWarning());
}
exports.toIssue = toIssue;
function createIssue1(messageEntry, parameters, node, isWarning) {
    if (isWarning === void 0) { isWarning = false; }
    var msg = applyTemplate(messageEntry, parameters);
    return createIssue(messageEntry.code, msg, node, isWarning);
}
exports.createIssue1 = createIssue1;
function createIssue(issueCode, message, node, isWarning) {
    if (isWarning === void 0) { isWarning = false; }
    //console.log(node.name()+node.lowLevel().start()+":"+node.id());
    var original = null;
    var pr = null;
    if (node.lowLevel() instanceof proxy.LowLevelProxyNode) {
        var proxyNode = node.lowLevel();
        while (!proxyNode.primaryNode()) {
            if (!original) {
                original = localError(node, issueCode, isWarning, message, true, pr);
            }
            node = node.parent();
            proxyNode = node.lowLevel();
        }
    }
    if (node) {
        pr = node.property();
        if (node.lowLevel().unit() != node.root().lowLevel().unit()) {
            original = localError(node, issueCode, isWarning, message, true, pr);
            var v = node.lowLevel().unit();
            if (v) {
            }
            while (node.lowLevel().unit() != node.root().lowLevel().unit()) {
                pr = node.property();
                node = node.parent();
            }
        }
    }
    if (original) {
        if (node.property() && node.property().nameId() == universes.Universe10.FragmentDeclaration.properties.uses.name && node.parent() != null) {
            pr = node.property(); //FIXME there should be other cases
            node = node.parent();
        }
    }
    var error = localError(node, issueCode, isWarning, message, false, pr);
    if (original) {
        original.extras.push(error);
        if (node.lowLevel().valueKind() == yaml.Kind.INCLUDE_REF) {
            var messageEntry = messageRegistry.ERROR_IN_INCLUDED_FILE;
            error.code = messageEntry.code;
            error.message = applyTemplate(messageEntry, { msg: error.message });
        }
        error = original;
    }
    //console.log(error.start+":"+error.end)
    return error;
}
exports.createIssue = createIssue;
function createLLIssue1(messageEntry, parameters, node, rootCalculationAnchor, isWarning, p) {
    if (isWarning === void 0) { isWarning = false; }
    if (p === void 0) { p = false; }
    var msg = applyTemplate(messageEntry, parameters);
    return createLLIssue(messageEntry.code, msg, node, rootCalculationAnchor, isWarning, p);
}
function createLLIssue(issueCode, message, node, rootCalculationAnchor, isWarning, p) {
    if (isWarning === void 0) { isWarning = false; }
    if (p === void 0) { p = false; }
    var original = null;
    if (node) {
        if (rootCalculationAnchor.lowLevel().unit() != rootCalculationAnchor.root().lowLevel().unit()) {
            original = localLowLevelError(node, rootCalculationAnchor, issueCode, isWarning, message, true);
            var v = rootCalculationAnchor.lowLevel().unit();
            if (v) {
                message = message + " " + v.path();
            }
            while (rootCalculationAnchor.lowLevel().unit() != rootCalculationAnchor.root().lowLevel().unit()) {
                rootCalculationAnchor = rootCalculationAnchor.parent();
            }
        }
    }
    if (original) {
        if (rootCalculationAnchor.property() && rootCalculationAnchor.property().nameId()
            == universes.Universe10.FragmentDeclaration.properties.uses.name && rootCalculationAnchor.parent() != null) {
            rootCalculationAnchor = rootCalculationAnchor.parent();
        }
    }
    var error = localLowLevelError(node, rootCalculationAnchor, issueCode, isWarning, message, p);
    if (original) {
        original.extras.push(error);
        if (node.valueKind() == yaml.Kind.INCLUDE_REF) {
            error.message = "Error in the included file: " + error.message;
        }
        error = original;
    }
    //console.log(error.start+":"+error.end)
    return error;
}
exports.createLLIssue = createLLIssue;
function validateResponseString(v) {
    if (v.length != 3) {
        return new ValidationError(messageRegistry.STATUS_MUST_BE_3NUMBER);
    }
    for (var i = 0; i < v.length; i++) {
        var c = v[i];
        if (!_.find(['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'], function (x) { return x == c; })) {
            return new ValidationError(messageRegistry.STATUS_MUST_BE_3NUMBER);
        }
    }
    return null;
}
exports.validateResponseString = validateResponseString;
function applyTemplate(messageEntry, params) {
    var result = "";
    var msg = messageEntry.message;
    var prev = 0;
    for (var ind = msg.indexOf("{{"); ind >= 0; ind = msg.indexOf("{{", prev)) {
        result += msg.substring(prev, ind);
        prev = msg.indexOf("}}", ind);
        if (prev < 0) {
            prev = ind;
            break;
        }
        ind += "{{".length;
        var paramName = msg.substring(ind, prev);
        prev += "}}".length;
        var paramValue = params[paramName];
        if (paramValue === undefined) {
            throw new Error("Message parameter '" + paramName + "' has no value specified.");
        }
        result += paramValue;
    }
    result += msg.substring(prev, msg.length);
    return result;
}
;
//# sourceMappingURL=linter.js.map