/// <reference path="../../../typings/main.d.ts" />
"use strict";
var ll = require("../lowLevelAST");
var path = require("path");
var universes = require("../tools/universe");
var universeHelpers = require("../tools/universeHelpers");
var resourceRegistry = require('../jsyaml/resourceRegistry');
var NamespaceResolver = (function () {
    function NamespaceResolver() {
        this.expandedAbsToNsMap = {};
        this.byPathMap = {};
        this.byNsMap = {};
        this._hasFragments = {};
    }
    NamespaceResolver.prototype.resolveNamespace = function (from, to) {
        if (to == null) {
            return null;
        }
        var toPath = to.absolutePath();
        var unitMap = this.expandedPathMap(from);
        if (!unitMap) {
            return null;
        }
        var usesInfo = unitMap[toPath];
        return usesInfo != null ? usesInfo.namespace() : null;
    };
    NamespaceResolver.prototype.expandedPathMap = function (unit) {
        var fromPath = unit.absolutePath();
        var unitMap = this.expandedAbsToNsMap[fromPath];
        if (unitMap === undefined) {
            unitMap = this.calculateExpandedNamespaces(unit);
            if (Object.keys(unitMap).length == 0) {
                unitMap = null;
            }
            this.expandedAbsToNsMap[fromPath] = unitMap;
        }
        return unitMap;
    };
    NamespaceResolver.prototype.calculateExpandedNamespaces = function (_unit) {
        var _this = this;
        var rootPath = path.dirname(_unit.absolutePath());
        var result = {};
        var usesInfoArray = [];
        while (_unit) {
            usesInfoArray.push(new UsesInfo([], _unit, ""));
            var u = _unit;
            _unit = null;
            var hlNode = u.highLevel();
            if (hlNode.isElement()) {
                var hlElem = hlNode.asElement();
                var definition = hlElem.definition();
                if (universeHelpers.isOverlayType(definition) || universeHelpers.isExtensionType(definition)) {
                    var eValue = hlElem.attrValue(universes.Universe10.Extension.properties.extends.name);
                    if (eValue) {
                        _unit = u.resolve(eValue);
                        if (_unit && resourceRegistry.isWaitingFor(_unit.absolutePath())) {
                            _unit = null;
                        }
                    }
                }
            }
        }
        for (var i = 0; i < usesInfoArray.length; i++) {
            var visited = {};
            var usedUnits = {};
            var info = usesInfoArray[i];
            var unit = info.unit;
            var hlPR = unit.highLevel();
            if (!hlPR.isElement()) {
                continue;
            }
            var steps = info.steps() + 1;
            var visit = function (x) {
                if (!x) {
                    return;
                }
                var children = x.children();
                if (x.parent() == null) {
                    var nodeUnit = x.unit();
                    var localPath = nodeUnit.absolutePath();
                    if (visited[localPath]) {
                        return;
                    }
                    visited[localPath] = true;
                    if (localPath != unit.absolutePath()) {
                        _this._hasFragments[unit.absolutePath()] = true;
                    }
                    var map = _this.pathMap(nodeUnit);
                    if (map) {
                        for (var _i = 0, _a = Object.keys(map); _i < _a.length; _i++) {
                            var absPath = _a[_i];
                            var childInfo = map[absPath];
                            var segments = info.namespaceSegments.concat(childInfo.namespaceSegments);
                            var existing = result[absPath];
                            if (existing) {
                                if (existing.steps() < steps) {
                                    continue;
                                }
                                else if (existing.steps() == steps
                                    && _this.lexLessEq(existing.namespaceSegments, segments)) {
                                    continue;
                                }
                            }
                            var includePath;
                            var childInclude = childInfo.includePath;
                            var absChildPath = childInfo.absolutePath();
                            if (path.isAbsolute(info.includePath) || ll.isWebPath(info.includePath)) {
                                includePath = absChildPath;
                            }
                            else if (path.isAbsolute(childInclude) || ll.isWebPath(childInclude)) {
                                includePath = absChildPath;
                            }
                            else {
                                if (ll.isWebPath(rootPath) != ll.isWebPath(absChildPath)) {
                                    includePath = absChildPath;
                                }
                                else if (rootPath.length > 0 && absChildPath.length > 0
                                    && rootPath.charAt(0).toLowerCase() != absChildPath.charAt(0).toLowerCase()) {
                                    //Windows case of library being located on different drive
                                    includePath = absChildPath;
                                }
                                else {
                                    includePath = path.relative(rootPath, absChildPath);
                                }
                            }
                            includePath = includePath.replace(/\\/g, "/");
                            var ui = new UsesInfo(segments, childInfo.unit, includePath);
                            if (!usedUnits[ui.absolutePath()]) {
                                result[absPath] = ui;
                                usesInfoArray.push(ui);
                                usedUnits[ui.absolutePath()] = true;
                            }
                        }
                    }
                }
                children.forEach(function (y) {
                    if (y.includedFrom()) {
                        y = y.parent();
                    }
                    visit(y);
                });
                if (x.parent() == null) {
                    visited[x.unit().absolutePath()] = false;
                }
            };
            visit(unit.ast());
        }
        var namespaces = {};
        for (var _i = 0, _a = Object.keys(result); _i < _a.length; _i++) {
            var key = _a[_i];
            var info = result[key];
            var ns = info.namespace();
            var i = 0;
            while (namespaces[ns]) {
                ns = info.namespace() + i++;
            }
            if (ns != info.namespace()) {
                info.namespaceSegments = ns.split(".");
            }
            namespaces[ns] = true;
        }
        return result;
    };
    NamespaceResolver.prototype.pathMap = function (unit) {
        var fromPath = unit.absolutePath();
        var unitMap = this.byPathMap[fromPath];
        if (unitMap === undefined) {
            unitMap = this.calculateNamespaces(unit);
            if (Object.keys(unitMap).length == 0) {
                unitMap = null;
            }
            this.byPathMap[fromPath] = unitMap;
        }
        return unitMap;
    };
    NamespaceResolver.prototype.nsMap = function (unit) {
        var fromPath = unit.absolutePath();
        var unitMap = this.byNsMap[fromPath];
        if (unitMap === undefined) {
            var map = this.pathMap(unit);
            if (map == null) {
                unitMap = null;
            }
            else {
                unitMap = {};
                for (var _i = 0, _a = Object.keys(map); _i < _a.length; _i++) {
                    var aPath = _a[_i];
                    var info = map[aPath];
                    unitMap[info.namespace()] = info;
                }
            }
            this.byNsMap[fromPath] = unitMap;
        }
        return unitMap;
    };
    NamespaceResolver.prototype.calculateNamespaces = function (unit) {
        var rootPath = path.dirname(unit.absolutePath());
        var result = {};
        var rootNode = unit.ast();
        var usesNodes = rootNode.children().filter(function (x) { return x.key()
            == universes.Universe10.FragmentDeclaration.properties.uses.name; });
        if (rootNode.actual() && rootNode.actual()["usesNode"]) {
            usesNodes = [rootNode.actual()["usesNode"]];
        }
        if (usesNodes.length == 0) {
            return result;
        }
        var usesDeclarationNodes = [];
        for (var _i = 0, usesNodes_1 = usesNodes; _i < usesNodes_1.length; _i++) {
            var un = usesNodes_1[_i];
            usesDeclarationNodes = usesDeclarationNodes.concat(un.children());
        }
        if (usesDeclarationNodes.length == 0) {
            return result;
        }
        for (var _a = 0, usesDeclarationNodes_1 = usesDeclarationNodes; _a < usesDeclarationNodes_1.length; _a++) {
            var un = usesDeclarationNodes_1[_a];
            var key = un.key();
            var value = un.value();
            var libUnit = unit.resolve(value);
            if (libUnit == null) {
                continue;
            }
            var segments = [key];
            var absPath = libUnit.absolutePath();
            var includePath;
            if (path.isAbsolute(value) || ll.isWebPath(value)) {
                includePath = libUnit.absolutePath();
            }
            else {
                includePath = path.relative(rootPath, libUnit.absolutePath());
            }
            includePath = includePath.replace(/\\/g, "/");
            var ui = new UsesInfo(segments, libUnit, includePath);
            result[absPath] = ui;
        }
        var hlNode = unit.highLevel();
        if (hlNode.isElement()) {
            var hlElem = hlNode.asElement();
            var definition = hlElem.definition();
            if (universeHelpers.isOverlayType(definition) || universeHelpers.isExtensionType(definition)) {
                var eValue = hlElem.attr(universes.Universe10.Extension.properties.extends.name).value();
                var extendedUnit;
                try {
                    extendedUnit = unit.resolve(eValue);
                }
                catch (e) { }
                if (extendedUnit) {
                    var m = this.pathMap(extendedUnit);
                    if (m) {
                        for (var _b = 0, _c = Object.keys(m); _b < _c.length; _b++) {
                            var k = _c[_b];
                            result[k] = m[k];
                        }
                    }
                }
            }
        }
        return result;
    };
    NamespaceResolver.prototype.lexLessEq = function (a, b) {
        if (a.length > b.length) {
            return false;
        }
        if (a.length < b.length) {
            return true;
        }
        for (var i = 0; i < a.length; i++) {
            var seg_a = a[i];
            var seg_b = b[i];
            if (seg_a < seg_b) {
                return true;
            }
            else if (seg_a > seg_b) {
                return false;
            }
        }
        return true;
    };
    NamespaceResolver.prototype.hasFragments = function (unit) {
        this.calculateExpandedNamespaces(unit);
        return this._hasFragments[unit.absolutePath()] ? true : false;
    };
    return NamespaceResolver;
}());
exports.NamespaceResolver = NamespaceResolver;
var UsesInfo = (function () {
    function UsesInfo(namespaceSegments, unit, includePath) {
        this.namespaceSegments = namespaceSegments;
        this.unit = unit;
        this.includePath = includePath;
    }
    UsesInfo.prototype.steps = function () {
        return this.namespaceSegments.length;
    };
    UsesInfo.prototype.namespace = function () {
        return this.namespaceSegments.join(".");
    };
    UsesInfo.prototype.absolutePath = function () {
        return this.unit.absolutePath();
    };
    return UsesInfo;
}());
exports.UsesInfo = UsesInfo;
//# sourceMappingURL=namespaceResolver.js.map