/**
 * Cursor rule:
 * 1. Only `showSearch` enabled
 * 2. Only `open` is `true`
 * 3. When typing, set `open` to `true` which hit rule of 2
 *
 * Accessibility:
 * - https://www.w3.org/TR/wai-aria-practices/examples/combobox/aria1.1pattern/listbox-combo.html
 */
import * as React from 'react';
import KeyCode from "rc-util/es/KeyCode";
import MultipleSelector from './MultipleSelector';
import SingleSelector from './SingleSelector';

var Selector = function Selector(props, ref) {
  var inputRef = React.useRef(null);
  var prefixCls = props.prefixCls,
      multiple = props.multiple,
      onSearch = props.onSearch,
      onToggleOpen = props.onToggleOpen,
      onInputKeyDown = props.onInputKeyDown; // ======================= Ref =======================

  React.useImperativeHandle(ref, function () {
    return {
      focus: function focus() {
        inputRef.current.focus();
      },
      blur: function blur() {
        inputRef.current.blur();
      }
    };
  }); // ====================== Input ======================

  var onInternalInputKeyDown = function onInternalInputKeyDown(event) {
    var which = event.which;

    if (which === KeyCode.UP || which === KeyCode.DOWN) {
      event.preventDefault();
    }

    if (onInputKeyDown) {
      onInputKeyDown(event);
    }

    if (![KeyCode.SHIFT, KeyCode.TAB, KeyCode.BACKSPACE, KeyCode.ESC].includes(which)) {
      onToggleOpen(true);
    }
  };

  var onInputChange = function onInputChange(_ref) {
    var value = _ref.target.value;

    if (onSearch(value) !== false) {
      onToggleOpen(true);
    }
  }; // ====================== Focus ======================
  // Should focus input if click the selector


  var onClick = function onClick(_ref2) {
    var target = _ref2.target;

    if (target !== inputRef.current) {
      inputRef.current.focus();
    }
  };

  var onMouseDown = function onMouseDown(event) {
    if (event.target !== inputRef.current) {
      event.preventDefault();
    }

    onToggleOpen();
  }; // ================= Inner Selector ==================


  var sharedProps = {
    inputRef: inputRef,
    onInputKeyDown: onInternalInputKeyDown,
    onInputChange: onInputChange
  };
  var selectNode = multiple ? React.createElement(MultipleSelector, Object.assign({}, props, sharedProps)) : React.createElement(SingleSelector, Object.assign({}, props, sharedProps));
  return React.createElement("div", {
    className: "".concat(prefixCls, "-selector"),
    onClick: onClick,
    onMouseDown: onMouseDown
  }, selectNode);
};

var ForwardSelector = React.forwardRef(Selector);
ForwardSelector.displayName = 'Selector';
export default ForwardSelector;