'use strict';

var React = require('react');
var d3 = require('d3');
var DataSeries = require('./DataSeries');

var _require = require('../common');

var Chart = _require.Chart;
var XAxis = _require.XAxis;
var YAxis = _require.YAxis;
var Tooltip = _require.Tooltip;

var _require2 = require('../mixins');

var CartesianChartPropsMixin = _require2.CartesianChartPropsMixin;
var DefaultAccessorsMixin = _require2.DefaultAccessorsMixin;
var ViewBoxMixin = _require2.ViewBoxMixin;
var TooltipMixin = _require2.TooltipMixin;


module.exports = React.createClass({

  displayName: 'BarChart',

  propTypes: {
    chartClassName: React.PropTypes.string,
    data: React.PropTypes.array.isRequired,
    hoverAnimation: React.PropTypes.bool,
    margins: React.PropTypes.object,
    rangeRoundBandsPadding: React.PropTypes.number,
    // https://github.com/mbostock/d3/wiki/Stack-Layout#offset
    stackOffset: React.PropTypes.oneOf(['silhouette', 'expand', 'wigget', 'zero']),
    grouped: React.PropTypes.bool,
    valuesAccessor: React.PropTypes.func,
    title: React.PropTypes.string,
    xAxisClassName: React.PropTypes.string,
    yAxisClassName: React.PropTypes.string,
    yAxisTickCount: React.PropTypes.number,
    xAccessor: React.PropTypes.any, // TODO: prop types?
    yAccessor: React.PropTypes.any
  },

  mixins: [CartesianChartPropsMixin, DefaultAccessorsMixin, ViewBoxMixin, TooltipMixin],

  getDefaultProps: function getDefaultProps() {
    return {
      chartClassName: 'rd3-barchart',
      hoverAnimation: true,
      margins: { top: 10, right: 20, bottom: 40, left: 45 },
      rangeRoundBandsPadding: 0.25,
      stackOffset: 'zero',
      grouped: false,
      valuesAccessor: function valuesAccessor(d) {
        return d.values;
      },
      xAxisClassName: 'rd3-barchart-xaxis',
      yAxisClassName: 'rd3-barchart-yaxis',
      yAxisTickCount: 4
    };
  },
  _getStackedValuesMaxY: function _getStackedValuesMaxY(_data) {
    // in stacked bar chart, the maximum height we need for
    // yScale domain is the sum of y0 + y
    var valuesAccessor = this.props.valuesAccessor;

    return d3.max(_data, function (d) {
      return d3.max(valuesAccessor(d), function (d2) {
        return(
          // where y0, y is generated by d3.layout.stack()
          d2.y0 + d2.y
        );
      });
    });
  },
  _getStackedValuesMinY: function _getStackedValuesMinY(_data) {
    var valuesAccessor = this.props.valuesAccessor;

    return d3.min(_data, function (d) {
      return d3.min(valuesAccessor(d), function (d2) {
        return(
          // where y0, y is generated by d3.layout.stack()
          d2.y0 + d2.y
        );
      });
    });
  },
  _getLabels: function _getLabels(firstSeries) {
    // we only need first series to get all the labels
    var _props = this.props;
    var valuesAccessor = _props.valuesAccessor;
    var xAccessor = _props.xAccessor;

    return valuesAccessor(firstSeries).map(xAccessor);
  },
  _stack: function _stack() {
    // Only support columns with all positive or all negative values
    // https://github.com/mbostock/d3/issues/2265
    var _props2 = this.props;
    var stackOffset = _props2.stackOffset;
    var xAccessor = _props2.xAccessor;
    var yAccessor = _props2.yAccessor;
    var valuesAccessor = _props2.valuesAccessor;

    return d3.layout.stack().offset(stackOffset).x(xAccessor).y(yAccessor).values(valuesAccessor);
  },
  render: function render() {
    var props = this.props;
    var yOrient = this.getYOrient();

    var domain = props.domain || {};

    if (props.data.length === 0) {
      return null;
    }
    var _data = this._stack()(props.data);

    var _getDimensions = this.getDimensions();

    var innerHeight = _getDimensions.innerHeight;
    var innerWidth = _getDimensions.innerWidth;
    var trans = _getDimensions.trans;
    var svgMargins = _getDimensions.svgMargins;


    var xDomain = domain.x || this._getLabels(_data[0]);
    var xScale = d3.scale.ordinal().domain(xDomain).rangeRoundBands([0, innerWidth], props.rangeRoundBandsPadding);

    var minYDomain = Math.min(0, this._getStackedValuesMinY(_data));
    var maxYDomain = this._getStackedValuesMaxY(_data);
    var yDomain = domain.y || [minYDomain, maxYDomain];
    var yScale = d3.scale.linear().range([innerHeight, 0]).domain(yDomain);

    var series = props.data.map(function (item) {
      return item.name;
    });

    return React.createElement(
      'span',
      null,
      React.createElement(
        Chart,
        {
          viewBox: this.getViewBox(),
          legend: props.legend,
          data: props.data,
          margins: props.margins,
          colors: props.colors,
          colorAccessor: props.colorAccessor,
          width: props.width,
          height: props.height,
          title: props.title,
          shouldUpdate: !this.state.changeState
        },
        React.createElement(
          'g',
          { transform: trans, className: props.chartClassName },
          React.createElement(YAxis, {
            yAxisClassName: props.yAxisClassName,
            yAxisTickValues: props.yAxisTickValues,
            yAxisLabel: props.yAxisLabel,
            yAxisLabelOffset: props.yAxisLabelOffset,
            yScale: yScale,
            margins: svgMargins,
            yAxisTickCount: props.yAxisTickCount,
            tickFormatting: props.yAxisFormatter,
            tickStroke: props.xAxisTickStroke,
            tickTextStroke: props.xAxisTickTextStroke,
            width: innerWidth,
            height: innerHeight,
            horizontalChart: props.horizontal,
            xOrient: props.xOrient,
            yOrient: yOrient,
            gridHorizontal: props.gridHorizontal,
            gridHorizontalStroke: props.gridHorizontalStroke,
            gridHorizontalStrokeWidth: props.gridHorizontalStrokeWidth,
            gridHorizontalStrokeDash: props.gridHorizontalStrokeDash
          }),
          React.createElement(XAxis, {
            xAxisClassName: props.xAxisClassName,
            xAxisTickValues: props.xAxisTickValues,
            xAxisLabel: props.xAxisLabel,
            xAxisLabelOffset: props.xAxisLabelOffset,
            xScale: xScale,
            margins: svgMargins,
            tickFormatting: props.xAxisFormatter,
            tickStroke: props.yAxisTickStroke,
            tickTextStroke: props.yAxisTickTextStroke,
            width: innerWidth,
            height: innerHeight,
            horizontalChart: props.horizontal,
            xOrient: props.xOrient,
            yOrient: yOrient,
            gridVertical: props.gridVertical,
            gridVerticalStroke: props.gridVerticalStroke,
            gridVerticalStrokeWidth: props.gridVerticalStrokeWidth,
            gridVerticalStrokeDash: props.gridVerticalStrokeDash
          }),
          React.createElement(DataSeries, {
            yScale: yScale,
            xScale: xScale,
            margins: svgMargins,
            _data: _data,
            series: series,
            width: innerWidth,
            height: innerHeight,
            grouped: props.grouped,
            colors: props.colors,
            colorAccessor: props.colorAccessor,
            hoverAnimation: props.hoverAnimation,
            valuesAccessor: props.valuesAccessor,
            onMouseOver: this.onMouseOver,
            onMouseLeave: this.onMouseLeave
          })
        )
      ),
      props.showTooltip ? React.createElement(Tooltip, this.state.tooltip) : null
    );
  }
});