import type * as RDF from '@rdfjs/types';
import { BlankNode } from './BlankNode';
import { DefaultGraph } from './DefaultGraph';
import { Literal } from './Literal';
import { NamedNode } from './NamedNode';
import { Quad } from './Quad';
import { Variable } from './Variable';
/**
 * A factory for instantiating RDF terms and quads.
 */
export declare class DataFactory<Q extends RDF.BaseQuad = RDF.Quad> implements RDF.DataFactory<Q> {
    private readonly blankNodePrefix;
    private blankNodeCounter;
    constructor(options?: IDataFactoryOptions);
    /**
     * @param value The IRI for the named node.
     * @return A new instance of NamedNode.
     * @see NamedNode
     */
    namedNode<Iri extends string = string>(value: Iri): NamedNode<Iri>;
    /**
     * @param value The optional blank node identifier.
     * @return A new instance of BlankNode.
     *         If the `value` parameter is undefined a new identifier
     *         for the blank node is generated for each call.
     * @see BlankNode
     */
    blankNode(value?: string): BlankNode;
    /**
     * @param value              The literal value.
     * @param languageOrDatatype The optional language or datatype.
     *                           If `languageOrDatatype` is a NamedNode,
     *                           then it is used for the value of `NamedNode.datatype`.
     *                           Otherwise `languageOrDatatype` is used for the value
     *                           of `NamedNode.language`.
     * @return A new instance of Literal.
     * @see Literal
     */
    literal(value: string, languageOrDatatype?: string | RDF.NamedNode): Literal;
    /**
     * This method is optional.
     * @param value The variable name
     * @return A new instance of Variable.
     * @see Variable
     */
    variable(value: string): Variable;
    /**
     * @return An instance of DefaultGraph.
     */
    defaultGraph(): DefaultGraph;
    /**
     * @param subject   The quad subject term.
     * @param predicate The quad predicate term.
     * @param object    The quad object term.
     * @param graph     The quad graph term.
     * @return A new instance of Quad.
     * @see Quad
     */
    quad(subject: Q['subject'], predicate: Q['predicate'], object: Q['object'], graph?: Q['graph']): Q & Quad;
    /**
     * Create a deep copy of the given term using this data factory.
     * @param original An RDF term.
     * @return A deep copy of the given term.
     */
    fromTerm<T extends RDF.Term>(original: T): (T extends RDF.NamedNode ? NamedNode : (T extends RDF.BlankNode ? BlankNode : (T extends RDF.Literal ? Literal : (T extends RDF.Variable ? Variable : (T extends RDF.DefaultGraph ? DefaultGraph : (T extends Q ? Q : unknown))))));
    /**
     * Create a deep copy of the given quad using this data factory.
     * @param original An RDF quad.
     * @return A deep copy of the given quad.
     */
    fromQuad(original: Q): Q;
    /**
     * Reset the internal blank node counter.
     */
    resetBlankNodeCounter(): void;
}
export interface IDataFactoryOptions {
    blankNodePrefix?: string;
}
