"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Translates between an RDF literal and a JavaScript primitive.
 */
class Translator {
    constructor() {
        this.supportedRdfDatatypes = [];
        this.fromRdfHandlers = {};
        this.toRdfHandlers = {};
    }
    static incorrectRdfDataType(literal) {
        throw new Error(`Invalid RDF ${literal.datatype.value} value: '${literal.value}'`);
    }
    registerHandler(handler, rdfDatatypes, javaScriptDataTypes) {
        for (const rdfDatatype of rdfDatatypes) {
            this.supportedRdfDatatypes.push(rdfDatatype);
            this.fromRdfHandlers[rdfDatatype.value] = handler;
        }
        for (const javaScriptDataType of javaScriptDataTypes) {
            let existingToRdfHandlers = this.toRdfHandlers[javaScriptDataType];
            if (!existingToRdfHandlers) {
                this.toRdfHandlers[javaScriptDataType] = existingToRdfHandlers = [];
            }
            existingToRdfHandlers.push(handler);
        }
    }
    fromRdf(literal, validate) {
        const handler = this.fromRdfHandlers[literal.datatype.value];
        if (handler) {
            return handler.fromRdf(literal, validate);
        }
        else {
            return literal.value;
        }
    }
    toRdf(value, dataFactory) {
        const handlers = this.toRdfHandlers[typeof value];
        if (handlers) {
            for (const handler of handlers) {
                const ret = handler.toRdf(value, dataFactory);
                if (ret) {
                    return ret;
                }
            }
        }
        throw new Error(`Invalid JavaScript value: '${value}'`);
    }
    /**
     * @return {NamedNode[]} An array of all supported RDF datatypes.
     */
    getSupportedRdfDatatypes() {
        return this.supportedRdfDatatypes;
    }
    /**
     * @return {string[]} An array of all supported JavaScript types.
     */
    getSupportedJavaScriptPrimitives() {
        return Object.keys(this.toRdfHandlers);
    }
}
exports.Translator = Translator;
//# sourceMappingURL=Translator.js.map