import * as RDF from "rdf-js";
/**
 * The possible quad term entries in an RDFJS quad.
 */
export declare type QuadTermName = 'subject' | 'predicate' | 'object' | 'graph';
/**
 * All available quad term names.
 * @type {[string , string , string , string]}
 */
export declare const QUAD_TERM_NAMES: QuadTermName[];
/**
 * All available triple term names.
 * @type {[string , string , string]}
 */
export declare const TRIPLE_TERM_NAMES: QuadTermName[];
/**
 * An RDFJS term with a quad term name key.
 */
export interface INamedTerm {
    /**
     * A quad term name.
     */
    key: QuadTermName;
    /**
     * An RDFJS term.
     */
    value: RDF.Term;
}
/**
 * Get all terms in the given quad.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {boolean} ignoreDefaultGraph If true and the quad has the default graph as graph,
 *                                     this term will not be returned in the array.
 *                                     (default: false)
 * @return {Term[]} The available terms in the quad.
 */
export declare function getTerms(quad: RDF.BaseQuad, ignoreDefaultGraph?: boolean): RDF.Term[];
/**
 * Convert the given quad to an array of named terms.
 * This is the reverse operation of {@link collectNamedTerms}.
 * @param {BaseQuad} quad An RDFJS quad.
 * @return {INamedTerm[]} An array of named terms.
 */
export declare function getNamedTerms(quad: RDF.BaseQuad): INamedTerm[];
/**
 * Convert an array of named terms to an RDFJS quad.
 * This is the reverse operation of {@link getNamedTerms}.
 * @param {INamedTerm[]} namedTerms An array of named terms.
 * @param {(termName: QuadTermName) => Term} defaultCb An optional callback for when
 *                                                     certain terms are not available in the array.
 * @param {RDF.DataFactory} dataFactory A custom data factory to create quads.
 * @return {Q} The resulting RDFJS quad.
 * @template Q The type of quad to output, defaults to RDF.Quad.
 */
export declare function collectNamedTerms<Q extends RDF.BaseQuad = RDF.Quad>(namedTerms: INamedTerm[], defaultCb?: (termName: QuadTermName) => RDF.Term, dataFactory?: RDF.DataFactory): Q;
/**
 * Iterats over each term.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} cb A callback function.
 */
export declare function forEachTerms(quad: RDF.BaseQuad, cb: (value: RDF.Term, key: QuadTermName) => void): void;
/**
 * Get all terms in the given quad that return true on the given filter function.
 * @param {BaseQuad} quad A quad.
 * @param {(value: Term, key: QuadTermName) => boolean} filter A filter callback.
 * @return {Term[]} The list of matching terms.
 */
export declare function filterTerms(quad: RDF.BaseQuad, filter: (value: RDF.Term, key: QuadTermName) => boolean): RDF.Term[];
/**
 * Get all quad term names in the given quad that return true on the given filter function.
 * @param {BaseQuad} quad A quad.
 * @param {(value: Term, key: QuadTermName, all: INamedTerm[]) => boolean} filter A filter callback.
 * @return {QuadTermName[]} The list of matching quad term names.
 */
export declare function filterQuadTermNames(quad: RDF.BaseQuad, filter: (value: RDF.Term, key: QuadTermName) => boolean): QuadTermName[];
/**
 * Map all terms of a quad.
 * @param {Quad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName) => Term} mapper A mapper function.
 * @param {RDF.DataFactory} dataFactory A custom data factory to create quads.
 * @return {Quad} A new RDFJS quad.
 * @template Q The type of quad, defaults to RDF.Quad.
 */
export declare function mapTerms<Q extends RDF.BaseQuad = RDF.Quad>(quad: Q, mapper: (term: RDF.Term, key: QuadTermName) => RDF.Term, dataFactory?: RDF.DataFactory): Q;
/**
 * Reduce all terms of a quad.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(previousValue: U, currentValue: Term, key: QuadTermName) => U} reducer A reduce function.
 * @param {U} initialValue The initial value.
 * @return {U} The final value.
 */
export declare function reduceTerms<U>(quad: RDF.BaseQuad, reducer: (previousValue: U, currentValue: RDF.Term, key: QuadTermName) => U, initialValue: U): U;
/**
 * Determines whether all terms satisfy the specified test.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} checker A checker function.
 * @return {boolean} If all terms satisfy the specified test.
 */
export declare function everyTerms(quad: RDF.BaseQuad, checker: (value: RDF.Term, key: QuadTermName) => boolean): boolean;
/**
 * Determines whether at least one term satisfies the specified test.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} checker A checker function.
 * @return {boolean} If at least one term satisfies the specified test.
 */
export declare function someTerms(quad: RDF.BaseQuad, checker: (value: RDF.Term, key: QuadTermName) => boolean): boolean;
/**
 * Check if the given quad matches with the given quad terms.
 *
 * Each term must match at least one of the following:
 * * Term is undefined.
 * * Term is a variable.
 * * Quad term and term are equal (`quadTerm.equals(term)` return true)
 *
 * @param {BaseQuad} quad A quad to match with (can not contain variables).
 * @param {Term} subject An optional subject.
 * @param {Term} predicate An optional predicate.
 * @param {Term} object An optional object.
 * @param {Term} graph An optional graph.
 * @return {boolean} If the quad matches with the quad terms.
 */
export declare function matchPattern(quad: RDF.BaseQuad, subject?: RDF.Term, predicate?: RDF.Term, object?: RDF.Term, graph?: RDF.Term): boolean;
/**
 * Check if the first quad matches with all terms from the second quad.
 *
 * Each term must match at least one of the following:
 * * Quad2 term is a variable.
 * * Quad1 term and Quad2 term are equal (`term1.equals(term2)` return true)
 *
 * @param {BaseQuad} quad A quad (can not contain variables).
 * @param {BaseQuad} pattern A quad pattern (can contain variables).
 * @return {boolean} If the quad terms match.
 */
export declare function matchPatternComplete(quad: RDF.BaseQuad, pattern: RDF.BaseQuad): boolean;
