"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const DataFactory = require("@rdfjs/data-model");
/**
 * All available quad term names.
 * @type {[string , string , string , string]}
 */
exports.QUAD_TERM_NAMES = ['subject', 'predicate', 'object', 'graph'];
/**
 * All available triple term names.
 * @type {[string , string , string]}
 */
exports.TRIPLE_TERM_NAMES = ['subject', 'predicate', 'object'];
/**
 * Get all terms in the given quad.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {boolean} ignoreDefaultGraph If true and the quad has the default graph as graph,
 *                                     this term will not be returned in the array.
 *                                     (default: false)
 * @return {Term[]} The available terms in the quad.
 */
function getTerms(quad, ignoreDefaultGraph) {
    if (ignoreDefaultGraph && quad.graph.termType === 'DefaultGraph') {
        return [quad.subject, quad.predicate, quad.object];
    }
    return [quad.subject, quad.predicate, quad.object, quad.graph];
}
exports.getTerms = getTerms;
/**
 * Convert the given quad to an array of named terms.
 * This is the reverse operation of {@link collectNamedTerms}.
 * @param {BaseQuad} quad An RDFJS quad.
 * @return {INamedTerm[]} An array of named terms.
 */
function getNamedTerms(quad) {
    return [
        { key: 'subject', value: quad.subject },
        { key: 'predicate', value: quad.predicate },
        { key: 'object', value: quad.object },
        { key: 'graph', value: quad.graph },
    ];
}
exports.getNamedTerms = getNamedTerms;
/**
 * Convert an array of named terms to an RDFJS quad.
 * This is the reverse operation of {@link getNamedTerms}.
 * @param {INamedTerm[]} namedTerms An array of named terms.
 * @param {(termName: QuadTermName) => Term} defaultCb An optional callback for when
 *                                                     certain terms are not available in the array.
 * @param {RDF.DataFactory} dataFactory A custom data factory to create quads.
 * @return {Q} The resulting RDFJS quad.
 * @template Q The type of quad to output, defaults to RDF.Quad.
 */
function collectNamedTerms(namedTerms, defaultCb, dataFactory) {
    const elements = {};
    namedTerms.forEach((namedTerm) => elements[namedTerm.key] = namedTerm.value);
    if (defaultCb) {
        elements.subject = elements.subject || defaultCb('subject');
        elements.predicate = elements.predicate || defaultCb('predicate');
        elements.object = elements.object || defaultCb('object');
        elements.graph = elements.graph || defaultCb('graph');
    }
    return (dataFactory || DataFactory).quad(elements.subject, elements.predicate, elements.object, elements.graph);
}
exports.collectNamedTerms = collectNamedTerms;
/**
 * Iterats over each term.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} cb A callback function.
 */
function forEachTerms(quad, cb) {
    cb(quad.subject, 'subject');
    cb(quad.predicate, 'predicate');
    cb(quad.object, 'object');
    cb(quad.graph, 'graph');
}
exports.forEachTerms = forEachTerms;
/**
 * Get all terms in the given quad that return true on the given filter function.
 * @param {BaseQuad} quad A quad.
 * @param {(value: Term, key: QuadTermName) => boolean} filter A filter callback.
 * @return {Term[]} The list of matching terms.
 */
function filterTerms(quad, filter) {
    const terms = [];
    if (filter(quad.subject, 'subject')) {
        terms.push(quad.subject);
    }
    if (filter(quad.predicate, 'predicate')) {
        terms.push(quad.predicate);
    }
    if (filter(quad.object, 'object')) {
        terms.push(quad.object);
    }
    if (filter(quad.graph, 'graph')) {
        terms.push(quad.graph);
    }
    return terms;
}
exports.filterTerms = filterTerms;
/**
 * Get all quad term names in the given quad that return true on the given filter function.
 * @param {BaseQuad} quad A quad.
 * @param {(value: Term, key: QuadTermName, all: INamedTerm[]) => boolean} filter A filter callback.
 * @return {QuadTermName[]} The list of matching quad term names.
 */
function filterQuadTermNames(quad, filter) {
    const names = [];
    if (filter(quad.subject, 'subject')) {
        names.push('subject');
    }
    if (filter(quad.predicate, 'predicate')) {
        names.push('predicate');
    }
    if (filter(quad.object, 'object')) {
        names.push('object');
    }
    if (filter(quad.graph, 'graph')) {
        names.push('graph');
    }
    return names;
}
exports.filterQuadTermNames = filterQuadTermNames;
/**
 * Map all terms of a quad.
 * @param {Quad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName) => Term} mapper A mapper function.
 * @param {RDF.DataFactory} dataFactory A custom data factory to create quads.
 * @return {Quad} A new RDFJS quad.
 * @template Q The type of quad, defaults to RDF.Quad.
 */
function mapTerms(quad, mapper, dataFactory) {
    return (dataFactory || DataFactory).quad(mapper(quad.subject, 'subject'), mapper(quad.predicate, 'predicate'), mapper(quad.object, 'object'), mapper(quad.graph, 'graph'));
}
exports.mapTerms = mapTerms;
/**
 * Reduce all terms of a quad.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(previousValue: U, currentValue: Term, key: QuadTermName) => U} reducer A reduce function.
 * @param {U} initialValue The initial value.
 * @return {U} The final value.
 */
function reduceTerms(quad, reducer, initialValue) {
    let value = initialValue;
    value = reducer(value, quad.subject, 'subject');
    value = reducer(value, quad.predicate, 'predicate');
    value = reducer(value, quad.object, 'object');
    return reducer(value, quad.graph, 'graph');
}
exports.reduceTerms = reduceTerms;
/**
 * Determines whether all terms satisfy the specified test.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} checker A checker function.
 * @return {boolean} If all terms satisfy the specified test.
 */
function everyTerms(quad, checker) {
    return checker(quad.subject, 'subject')
        && checker(quad.predicate, 'predicate')
        && checker(quad.object, 'object')
        && checker(quad.graph, 'graph');
}
exports.everyTerms = everyTerms;
/**
 * Determines whether at least one term satisfies the specified test.
 * @param {BaseQuad} quad An RDFJS quad.
 * @param {(value: Term, key: QuadTermName} checker A checker function.
 * @return {boolean} If at least one term satisfies the specified test.
 */
function someTerms(quad, checker) {
    return checker(quad.subject, 'subject')
        || checker(quad.predicate, 'predicate')
        || checker(quad.object, 'object')
        || checker(quad.graph, 'graph');
}
exports.someTerms = someTerms;
/**
 * Check if the given quad matches with the given quad terms.
 *
 * Each term must match at least one of the following:
 * * Term is undefined.
 * * Term is a variable.
 * * Quad term and term are equal (`quadTerm.equals(term)` return true)
 *
 * @param {BaseQuad} quad A quad to match with (can not contain variables).
 * @param {Term} subject An optional subject.
 * @param {Term} predicate An optional predicate.
 * @param {Term} object An optional object.
 * @param {Term} graph An optional graph.
 * @return {boolean} If the quad matches with the quad terms.
 */
function matchPattern(quad, subject, predicate, object, graph) {
    return (!subject || subject.termType === 'Variable' || quad.subject.equals(subject))
        && (!predicate || predicate.termType === 'Variable' || quad.predicate.equals(predicate))
        && (!object || object.termType === 'Variable' || quad.object.equals(object))
        && (!graph || graph.termType === 'Variable' || quad.graph.equals(graph));
}
exports.matchPattern = matchPattern;
/**
 * Check if the first quad matches with all terms from the second quad.
 *
 * Each term must match at least one of the following:
 * * Quad2 term is a variable.
 * * Quad1 term and Quad2 term are equal (`term1.equals(term2)` return true)
 *
 * @param {BaseQuad} quad A quad (can not contain variables).
 * @param {BaseQuad} pattern A quad pattern (can contain variables).
 * @return {boolean} If the quad terms match.
 */
function matchPatternComplete(quad, pattern) {
    return matchPattern(quad, pattern.subject, pattern.predicate, pattern.object, pattern.graph);
}
exports.matchPatternComplete = matchPatternComplete;
//# sourceMappingURL=QuadTermUtil.js.map