"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * All known term types.
 * @see RDF.Term
 * @type {[string , string , string , string , string]}
 */
exports.TERM_TYPES = ['NamedNode', 'BlankNode', 'Literal', 'Variable', 'DefaultGraph'];
/*
 * Utility methods for handling RDFJS terms.
 */
/**
 * Create an array of unique terms from the given array.
 * @param {T[]} terms An array of RDFJS terms.
 * @return {T[]} A new array of unique RDFJS terms.
 */
function uniqTerms(terms) {
    return require('lodash.uniqwith')(terms, (termA, termB) => termA.equals(termB));
}
exports.uniqTerms = uniqTerms;
/**
 * Find all terms of the given type in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @param {"NamedNode" | "BlankNode" | "Literal" | "Variable" | "DefaultGraph"} termType A term type.
 * @return {Term[]} A new array with elements from the given array only containing elements of the given type.
 */
function getTermsOfType(terms, termType) {
    return terms.filter((term) => term.termType === termType);
}
exports.getTermsOfType = getTermsOfType;
/**
 * Find all named nodes in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @return {NamedNode[]} A new array with elements from the given array only containing named nodes.
 */
function getNamedNodes(terms) {
    return getTermsOfType(terms, 'NamedNode');
}
exports.getNamedNodes = getNamedNodes;
/**
 * Find all blank nodes in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @return {BlankNode[]} A new array with elements from the given array only containing blank nodes.
 */
function getBlankNodes(terms) {
    return getTermsOfType(terms, 'BlankNode');
}
exports.getBlankNodes = getBlankNodes;
/**
 * Find all literals in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @return {Literal[]} A new array with elements from the given array only containing literals.
 */
function getLiterals(terms) {
    return getTermsOfType(terms, 'Literal');
}
exports.getLiterals = getLiterals;
/**
 * Find all variables in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @return {Variable[]} A new array with elements from the given array only containing variables.
 */
function getVariables(terms) {
    return getTermsOfType(terms, 'Variable');
}
exports.getVariables = getVariables;
/**
 * Find all default graphs in the given array.
 * @param {Term[]} terms An array of RDFJS terms.
 * @return {DefaultGraph[]} A new array with elements from the given array only containing default graphs.
 */
function getDefaultGraphs(terms) {
    return getTermsOfType(terms, 'DefaultGraph');
}
exports.getDefaultGraphs = getDefaultGraphs;
//# sourceMappingURL=TermUtil.js.map