"use strict";
/**
  * @reach/listbox v0.18.0-pre.5
  *
  * Copyright (c) 2018-2022, React Training LLC
  *
  * This source code is licensed under the MIT license found in the
  * LICENSE.md file in the root directory of this source tree.
  *
  * @license MIT
  */


// src/reach-listbox.tsx
import * as React from "react";
import { useId } from "@reach/auto-id";
import { Popover, positionMatchWidth } from "@reach/popover";
import {
  createDescendantContext,
  DescendantProvider,
  useDescendant,
  useDescendantKeyDown,
  useDescendants,
  useDescendantsInit
} from "@reach/descendants";
import {
  composeEventHandlers,
  createNamedContext,
  isBoolean,
  isFunction,
  isRightClick,
  isString,
  makeId,
  useComposedRefs,
  useControlledSwitchWarning,
  useIsomorphicLayoutEffect as useLayoutEffect,
  useStableCallback,
  useStatefulRefValue
} from "@reach/utils";
import { useMachine, useCreateMachine } from "@reach/machine";

// src/machine.ts
import { assign } from "@reach/machine";
import { getOwnerDocument } from "@reach/utils";
var clearNavigationValue = assign({
  navigationValue: null
});
var clearTypeahead = assign({
  typeaheadQuery: null
});
var assignValue = assign({
  value: (_, event) => event.value
});
var resetValue = assign({
  value: (data, event) => data.initialValue
});
var navigate = assign({
  navigationValue: (data, event) => event.value
});
var navigateFromCurrentValue = assign({
  navigationValue: (data) => {
    let selected = findOptionFromValue(data.value, data.options);
    if (selected && !selected.disabled) {
      return data.value;
    } else {
      return data.options.find((option) => !option.disabled)?.value || null;
    }
  }
});
function listboxLostFocus(data, event) {
  if (event.type === "BLUR" /* Blur */) {
    let { list, popover } = event.refs;
    let { relatedTarget } = event;
    let ownerDocument = getOwnerDocument(popover);
    return !!(ownerDocument?.activeElement !== list && popover && !popover.contains(relatedTarget || ownerDocument?.activeElement));
  }
  return false;
}
function clickedOutsideOfListbox(data, event) {
  if (event.type === "OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */ || event.type === "OUTSIDE_MOUSE_UP" /* OutsideMouseUp */) {
    let { button, popover } = event.refs;
    let { relatedTarget } = event;
    return !!(relatedTarget !== button && button && !button.contains(relatedTarget) && popover && !popover.contains(relatedTarget));
  }
  return false;
}
function optionIsActive(data, event) {
  return !!data.options.find((option) => option.value === data.navigationValue);
}
function shouldNavigate(data, event) {
  let { popover, list } = event.refs;
  let { relatedTarget } = event;
  if (popover && relatedTarget && popover.contains(relatedTarget) && relatedTarget !== list) {
    return false;
  }
  return optionIsActive(data, event);
}
function focusList(data, event) {
  requestAnimationFrame(() => {
    event.refs.list && event.refs.list.focus();
  });
}
function focusButton(data, event) {
  event.refs.button && event.refs.button.focus();
}
function listboxIsNotDisabled(data, event) {
  return !event.disabled;
}
function listboxIsNotControlled(data, event) {
  return "isControlled" in event ? !event.isControlled : false;
}
function optionIsNavigable(data, event) {
  if (event.type === "OPTION_TOUCH_START" /* OptionTouchStart */) {
    if (event && event.disabled) {
      return false;
    }
  }
  return true;
}
function optionIsSelectable(data, event) {
  if ("disabled" in event && event.disabled) {
    return false;
  }
  if ("value" in event) {
    return event.value != null;
  }
  return data.navigationValue != null;
}
function selectOption(data, event) {
  event.callback && event.callback(event.value);
}
function submitForm(data, event) {
  if (event.type !== "KEY_DOWN_ENTER" /* KeyDownEnter */) {
    return;
  }
  let { hiddenInput } = event.refs;
  if (hiddenInput && hiddenInput.form) {
    let submitButton = hiddenInput.form.querySelector("button:not([type]),[type='submit']");
    submitButton && submitButton.click();
  }
}
var setTypeahead = assign({
  typeaheadQuery: (data, event) => {
    return (data.typeaheadQuery || "") + event.query;
  }
});
var setValueFromTypeahead = assign({
  value: (data, event) => {
    if (event.type === "UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */ && event.query) {
      let match = findOptionFromTypeahead(data.options, event.query);
      if (match && !match.disabled) {
        event.callback && event.callback(match.value);
        return match.value;
      }
    }
    return data.value;
  }
});
var setNavSelectionFromTypeahead = assign({
  navigationValue: (data, event) => {
    if (event.type === "UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */ && event.query) {
      let match = findOptionFromTypeahead(data.options, event.query);
      if (match && !match.disabled) {
        return match.value;
      }
    }
    return data.navigationValue;
  }
});
var commonEvents = {
  ["GET_DERIVED_DATA" /* GetDerivedData */]: {
    actions: assign((ctx, event) => {
      return {
        ...ctx,
        ...event.data
      };
    })
  },
  ["VALUE_CHANGE" /* ValueChange */]: {
    actions: [assignValue, selectOption]
  }
};
var createMachineDefinition = ({
  value
}) => ({
  id: "listbox",
  initial: "IDLE" /* Idle */,
  context: {
    initialValue: value,
    value,
    options: [],
    navigationValue: null,
    typeaheadQuery: null
  },
  states: {
    ["IDLE" /* Idle */]: {
      on: {
        ...commonEvents,
        ["BUTTON_MOUSE_DOWN" /* ButtonMouseDown */]: {
          target: "OPEN" /* Open */,
          actions: [navigateFromCurrentValue],
          cond: listboxIsNotDisabled
        },
        ["KEY_DOWN_SPACE" /* KeyDownSpace */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, focusList],
          cond: listboxIsNotDisabled
        },
        ["KEY_DOWN_SEARCH" /* KeyDownSearch */]: {
          target: "IDLE" /* Idle */,
          actions: setTypeahead,
          cond: listboxIsNotDisabled
        },
        ["UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */]: {
          target: "IDLE" /* Idle */,
          actions: [setValueFromTypeahead],
          cond: listboxIsNotDisabled
        },
        ["CLEAR_TYPEAHEAD" /* ClearTypeahead */]: {
          target: "IDLE" /* Idle */,
          actions: clearTypeahead
        },
        ["KEY_DOWN_NAVIGATE" /* KeyDownNavigate */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, clearTypeahead, focusList],
          cond: listboxIsNotDisabled
        },
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: {
          actions: [submitForm],
          cond: listboxIsNotDisabled
        },
        ["RESET_VALUE" /* ResetValue */]: {
          actions: [resetValue, selectOption],
          cond: listboxIsNotControlled
        }
      }
    },
    ["INTERACTING" /* Interacting */]: {
      entry: [clearNavigationValue],
      on: {
        ...commonEvents,
        ["CLEAR_NAV_SELECTION" /* ClearNavSelection */]: {
          actions: [clearNavigationValue, focusList]
        },
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["KEY_DOWN_SPACE" /* KeyDownSpace */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["BUTTON_MOUSE_DOWN" /* ButtonMouseDown */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["KEY_DOWN_ESCAPE" /* KeyDownEscape */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["OPTION_MOUSE_DOWN" /* OptionMouseDown */]: {
          target: "DRAGGING" /* Dragging */
        },
        ["OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "DRAGGING" /* Dragging */,
            actions: clearTypeahead,
            cond: optionIsActive
          }
        ],
        ["OUTSIDE_MOUSE_UP" /* OutsideMouseUp */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: "INTERACTING" /* Interacting */,
        ["BLUR" /* Blur */]: [
          {
            target: "IDLE" /* Idle */,
            cond: listboxLostFocus,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: shouldNavigate
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["OPTION_TOUCH_START" /* OptionTouchStart */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["OPTION_CLICK" /* OptionClick */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_PRESS" /* OptionPress */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_MOUSE_ENTER" /* OptionMouseEnter */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["KEY_DOWN_NAVIGATE" /* KeyDownNavigate */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead, focusList]
        }
      }
    },
    ["OPEN" /* Open */]: {
      on: {
        ...commonEvents,
        ["CLEAR_NAV_SELECTION" /* ClearNavSelection */]: {
          actions: [clearNavigationValue]
        },
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["KEY_DOWN_SPACE" /* KeyDownSpace */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["BUTTON_MOUSE_DOWN" /* ButtonMouseDown */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["KEY_DOWN_ESCAPE" /* KeyDownEscape */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["OPTION_MOUSE_DOWN" /* OptionMouseDown */]: {
          target: "DRAGGING" /* Dragging */
        },
        ["OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "DRAGGING" /* Dragging */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["OUTSIDE_MOUSE_UP" /* OutsideMouseUp */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["BLUR" /* Blur */]: [
          {
            target: "IDLE" /* Idle */,
            cond: listboxLostFocus,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: shouldNavigate
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["BUTTON_MOUSE_UP" /* ButtonMouseUp */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, focusList]
        },
        ["LIST_MOUSE_UP" /* ListMouseUp */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, focusList]
        },
        ["OPTION_TOUCH_START" /* OptionTouchStart */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["OPTION_CLICK" /* OptionClick */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_PRESS" /* OptionPress */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["KEY_DOWN_NAVIGATE" /* KeyDownNavigate */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead, focusList]
        },
        ["KEY_DOWN_SEARCH" /* KeyDownSearch */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: setTypeahead
        },
        ["UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */]: {
          actions: [setNavSelectionFromTypeahead]
        },
        ["CLEAR_TYPEAHEAD" /* ClearTypeahead */]: {
          actions: clearTypeahead
        },
        ["OPTION_MOUSE_MOVE" /* OptionMouseMove */]: [
          {
            target: "DRAGGING" /* Dragging */,
            actions: [navigate],
            cond: optionIsNavigable
          },
          {
            target: "DRAGGING" /* Dragging */
          }
        ]
      }
    },
    ["DRAGGING" /* Dragging */]: {
      on: {
        ...commonEvents,
        ["CLEAR_NAV_SELECTION" /* ClearNavSelection */]: {
          actions: [clearNavigationValue]
        },
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["KEY_DOWN_SPACE" /* KeyDownSpace */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["BUTTON_MOUSE_DOWN" /* ButtonMouseDown */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["KEY_DOWN_ESCAPE" /* KeyDownEscape */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["OPTION_MOUSE_DOWN" /* OptionMouseDown */]: {
          target: "DRAGGING" /* Dragging */
        },
        ["OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["OUTSIDE_MOUSE_UP" /* OutsideMouseUp */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive,
            actions: focusList
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: [clearTypeahead, focusList]
          }
        ],
        ["BLUR" /* Blur */]: [
          {
            target: "IDLE" /* Idle */,
            cond: listboxLostFocus,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: shouldNavigate
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["BUTTON_MOUSE_UP" /* ButtonMouseUp */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, focusList]
        },
        ["OPTION_TOUCH_START" /* OptionTouchStart */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["OPTION_CLICK" /* OptionClick */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_PRESS" /* OptionPress */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_MOUSE_ENTER" /* OptionMouseEnter */]: {
          target: "DRAGGING" /* Dragging */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["KEY_DOWN_NAVIGATE" /* KeyDownNavigate */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead, focusList]
        },
        ["KEY_DOWN_SEARCH" /* KeyDownSearch */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: setTypeahead
        },
        ["UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */]: {
          actions: [setNavSelectionFromTypeahead]
        },
        ["CLEAR_TYPEAHEAD" /* ClearTypeahead */]: {
          actions: clearTypeahead
        },
        ["OPTION_MOUSE_MOVE" /* OptionMouseMove */]: [
          {
            target: "NAVIGATING" /* Navigating */,
            actions: [navigate],
            cond: optionIsNavigable
          },
          {
            target: "NAVIGATING" /* Navigating */
          }
        ],
        ["OPTION_MOUSE_UP" /* OptionMouseUp */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        }
      }
    },
    ["NAVIGATING" /* Navigating */]: {
      on: {
        ...commonEvents,
        ["CLEAR_NAV_SELECTION" /* ClearNavSelection */]: {
          actions: [clearNavigationValue, focusList]
        },
        ["KEY_DOWN_ENTER" /* KeyDownEnter */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["KEY_DOWN_SPACE" /* KeyDownSpace */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["BUTTON_MOUSE_DOWN" /* ButtonMouseDown */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["KEY_DOWN_ESCAPE" /* KeyDownEscape */]: {
          target: "IDLE" /* Idle */,
          actions: [focusButton]
        },
        ["OPTION_MOUSE_DOWN" /* OptionMouseDown */]: {
          target: "DRAGGING" /* Dragging */
        },
        ["OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["OUTSIDE_MOUSE_UP" /* OutsideMouseUp */]: [
          {
            target: "IDLE" /* Idle */,
            cond: clickedOutsideOfListbox,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: optionIsActive
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["BLUR" /* Blur */]: [
          {
            target: "IDLE" /* Idle */,
            cond: listboxLostFocus,
            actions: clearTypeahead
          },
          {
            target: "NAVIGATING" /* Navigating */,
            cond: shouldNavigate
          },
          {
            target: "INTERACTING" /* Interacting */,
            actions: clearTypeahead
          }
        ],
        ["BUTTON_MOUSE_UP" /* ButtonMouseUp */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigateFromCurrentValue, focusList]
        },
        ["OPTION_TOUCH_START" /* OptionTouchStart */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["OPTION_CLICK" /* OptionClick */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_PRESS" /* OptionPress */]: {
          target: "IDLE" /* Idle */,
          actions: [assignValue, clearTypeahead, focusButton, selectOption],
          cond: optionIsSelectable
        },
        ["OPTION_MOUSE_ENTER" /* OptionMouseEnter */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead],
          cond: optionIsNavigable
        },
        ["KEY_DOWN_NAVIGATE" /* KeyDownNavigate */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: [navigate, clearTypeahead, focusList]
        },
        ["KEY_DOWN_SEARCH" /* KeyDownSearch */]: {
          target: "NAVIGATING" /* Navigating */,
          actions: setTypeahead
        },
        ["UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */]: {
          actions: [setNavSelectionFromTypeahead]
        },
        ["CLEAR_TYPEAHEAD" /* ClearTypeahead */]: {
          actions: clearTypeahead
        },
        ["OPTION_MOUSE_MOVE" /* OptionMouseMove */]: [
          {
            target: "NAVIGATING" /* Navigating */,
            actions: [navigate],
            cond: optionIsNavigable
          },
          {
            target: "NAVIGATING" /* Navigating */
          }
        ]
      }
    }
  }
});
function findOptionFromTypeahead(options, string = "") {
  if (!string)
    return null;
  const found = options.find((option) => !option.disabled && option.label && option.label.toLowerCase().startsWith(string.toLowerCase()));
  return found || null;
}
function findOptionFromValue(value, options) {
  return value ? options.find((option) => option.value === value) : void 0;
}

// src/reach-listbox.tsx
var DEBUG = false;
var ListboxDescendantContext = createDescendantContext("ListboxDescendantContext");
var ListboxContext = createNamedContext("ListboxContext", {});
var ListboxGroupContext = createNamedContext("ListboxGroupContext", {});
var ListboxInput = React.forwardRef(function ListboxInput2({
  as: Comp = "div",
  "aria-labelledby": ariaLabelledBy,
  "aria-label": ariaLabel,
  children,
  defaultValue,
  disabled = false,
  form,
  name,
  onChange,
  required,
  value: valueProp,
  __componentName = "ListboxInput",
  ...props
}, forwardedRef) {
  let isControlled = React.useRef(valueProp != null);
  let [options, setOptions] = useDescendantsInit();
  let buttonRef = React.useRef(null);
  let hiddenInputRef = React.useRef(null);
  let highlightedOptionRef = React.useRef(null);
  let inputRef = React.useRef(null);
  let listRef = React.useRef(null);
  let popoverRef = React.useRef(null);
  let selectedOptionRef = React.useRef(null);
  let machine = useCreateMachine(createMachineDefinition({
    value: (isControlled.current ? valueProp : defaultValue) || null
  }));
  let [state, send] = useMachine(machine, {
    button: buttonRef,
    hiddenInput: hiddenInputRef,
    highlightedOption: highlightedOptionRef,
    input: inputRef,
    list: listRef,
    popover: popoverRef,
    selectedOption: selectedOptionRef
  }, DEBUG);
  function handleValueChange(newValue) {
    if (newValue !== state.context.value) {
      onChange?.(newValue);
    }
  }
  let _id = useId(props.id);
  let id = props.id || makeId("listbox-input", _id);
  let ref = useComposedRefs(inputRef, forwardedRef);
  let valueLabel = React.useMemo(() => {
    let selected = options.find((option) => option.value === state.context.value);
    return selected ? selected.label : null;
  }, [options, state.context.value]);
  let isExpanded = isListboxExpanded(state.value);
  let context = {
    ariaLabel,
    ariaLabelledBy,
    buttonRef,
    disabled,
    highlightedOptionRef,
    isExpanded,
    listboxId: id,
    listboxValueLabel: valueLabel,
    listRef,
    onValueChange: handleValueChange,
    popoverRef,
    selectedOptionRef,
    send,
    state: state.value,
    stateData: state.context
  };
  let mounted = React.useRef(false);
  if (!isControlled.current && defaultValue == null && !mounted.current && options.length) {
    mounted.current = true;
    let first = options.find((option) => !option.disabled);
    if (first && first.value) {
      send({
        type: "VALUE_CHANGE" /* ValueChange */,
        value: first.value
      });
    }
  }
  useControlledSwitchWarning(valueProp, "value", __componentName);
  useControlledStateSync(valueProp, state.context.value, () => {
    send({
      type: "VALUE_CHANGE" /* ValueChange */,
      value: valueProp
    });
  });
  useLayoutEffect(() => {
    send({
      type: "GET_DERIVED_DATA" /* GetDerivedData */,
      data: { options }
    });
  }, [options, send]);
  React.useEffect(() => {
    function handleMouseDown(event) {
      let { target, relatedTarget } = event;
      if (!popoverContainsEventTarget(popoverRef.current, target)) {
        send({
          type: "OUTSIDE_MOUSE_DOWN" /* OutsideMouseDown */,
          relatedTarget: relatedTarget || target
        });
      }
    }
    if (isExpanded) {
      window.addEventListener("mousedown", handleMouseDown);
    }
    return () => {
      window.removeEventListener("mousedown", handleMouseDown);
    };
  }, [send, isExpanded]);
  React.useEffect(() => {
    function handleMouseUp(event) {
      let { target, relatedTarget } = event;
      if (!popoverContainsEventTarget(popoverRef.current, target)) {
        send({
          type: "OUTSIDE_MOUSE_UP" /* OutsideMouseUp */,
          relatedTarget: relatedTarget || target
        });
      }
    }
    if (isExpanded) {
      window.addEventListener("mouseup", handleMouseUp);
    }
    return () => {
      window.removeEventListener("mouseup", handleMouseUp);
    };
  }, [send, isExpanded]);
  React.useEffect(() => {
    let form2 = hiddenInputRef.current?.form;
    if (!form2)
      return;
    function handleReset(event) {
      send({
        type: "RESET_VALUE" /* ResetValue */,
        isControlled: isControlled.current
      });
    }
    form2.addEventListener("reset", handleReset);
    return () => {
      form2?.removeEventListener("reset", handleReset);
    };
  }, [inputRef, isControlled, send]);
  return /* @__PURE__ */ React.createElement(Comp, {
    ...props,
    ref,
    "data-reach-listbox-input": "",
    "data-state": isExpanded ? "expanded" : "closed",
    "data-value": state.context.value,
    id
  }, /* @__PURE__ */ React.createElement(ListboxContext.Provider, {
    value: context
  }, /* @__PURE__ */ React.createElement(DescendantProvider, {
    context: ListboxDescendantContext,
    items: options,
    set: setOptions
  }, isFunction(children) ? children({
    id,
    isExpanded,
    value: state.context.value,
    selectedOptionRef,
    highlightedOptionRef,
    valueLabel,
    expanded: isExpanded
  }) : children, (form || name || required) && /* @__PURE__ */ React.createElement("input", {
    ref: hiddenInputRef,
    "data-reach-listbox-hidden-input": "",
    disabled,
    form,
    name,
    readOnly: true,
    required,
    tabIndex: -1,
    type: "hidden",
    value: state.context.value || ""
  }))));
});
ListboxInput.displayName = "ListboxInput";
var Listbox = React.forwardRef(function Listbox2({ arrow = "\u25BC", button, children, portal = true, ...props }, forwardedRef) {
  return /* @__PURE__ */ React.createElement(ListboxInput, {
    ...props,
    __componentName: "Listbox",
    ref: forwardedRef
  }, ({ value, valueLabel }) => /* @__PURE__ */ React.createElement(React.Fragment, null, /* @__PURE__ */ React.createElement(ListboxButton2, {
    arrow,
    children: button ? isFunction(button) ? button({ value, label: valueLabel }) : button : void 0
  }), /* @__PURE__ */ React.createElement(ListboxPopover2, {
    portal
  }, /* @__PURE__ */ React.createElement(ListboxList, null, children))));
});
Listbox.displayName = "Listbox";
var ListboxButtonImpl = React.forwardRef(function ListboxButton({
  "aria-label": ariaLabel,
  arrow = false,
  as: Comp = "span",
  children,
  onKeyDown,
  onMouseDown,
  onMouseUp,
  ...props
}, forwardedRef) {
  let {
    buttonRef,
    send,
    ariaLabelledBy,
    disabled,
    isExpanded,
    listboxId,
    stateData,
    listboxValueLabel
  } = React.useContext(ListboxContext);
  let listboxValue = stateData.value;
  let ref = useComposedRefs(buttonRef, forwardedRef);
  let handleKeyDown = useKeyDown();
  function handleMouseDown(event) {
    if (!isRightClick(event.nativeEvent)) {
      event.preventDefault();
      event.stopPropagation();
      send({
        type: "BUTTON_MOUSE_DOWN" /* ButtonMouseDown */,
        disabled
      });
    }
  }
  function handleMouseUp(event) {
    if (!isRightClick(event.nativeEvent)) {
      event.preventDefault();
      event.stopPropagation();
      send({ type: "BUTTON_MOUSE_UP" /* ButtonMouseUp */ });
    }
  }
  let id = makeId("button", listboxId);
  let label = React.useMemo(() => {
    if (!children) {
      return listboxValueLabel;
    } else if (isFunction(children)) {
      return children({
        isExpanded,
        label: listboxValueLabel,
        value: listboxValue,
        expanded: isExpanded
      });
    }
    return children;
  }, [children, listboxValueLabel, isExpanded, listboxValue]);
  return /* @__PURE__ */ React.createElement(Comp, {
    "aria-disabled": disabled || void 0,
    "aria-expanded": isExpanded || void 0,
    "aria-haspopup": "listbox",
    "aria-labelledby": ariaLabel ? void 0 : [ariaLabelledBy, id].filter(Boolean).join(" "),
    "aria-label": ariaLabel,
    role: "button",
    tabIndex: disabled ? -1 : 0,
    ...props,
    ref,
    "data-reach-listbox-button": "",
    id,
    onKeyDown: composeEventHandlers(onKeyDown, handleKeyDown),
    onMouseDown: composeEventHandlers(onMouseDown, handleMouseDown),
    onMouseUp: composeEventHandlers(onMouseUp, handleMouseUp)
  }, label, arrow && /* @__PURE__ */ React.createElement(ListboxArrow2, null, isBoolean(arrow) ? null : arrow));
});
ListboxButtonImpl.displayName = "ListboxButton";
var ListboxButton2 = React.memo(ListboxButtonImpl);
var ListboxArrowImpl = React.forwardRef(function ListboxArrow({ as: Comp = "span", children, ...props }, forwardedRef) {
  let { isExpanded } = React.useContext(ListboxContext);
  return /* @__PURE__ */ React.createElement(Comp, {
    "aria-hidden": true,
    ...props,
    ref: forwardedRef,
    "data-reach-listbox-arrow": "",
    "data-expanded": isExpanded ? "" : void 0
  }, isFunction(children) ? children({
    isExpanded,
    expanded: isExpanded
  }) : children || "\u25BC");
});
ListboxArrowImpl.displayName = "ListboxArrow";
var ListboxArrow2 = React.memo(ListboxArrowImpl);
var ListboxPopoverImpl = React.forwardRef(function ListboxPopover({
  as: Comp = "div",
  position = positionMatchWidth,
  onBlur,
  onKeyDown,
  onMouseUp,
  portal = true,
  unstable_observableRefs,
  ...props
}, forwardedRef) {
  let { isExpanded, buttonRef, popoverRef, send } = React.useContext(ListboxContext);
  let ref = useComposedRefs(popoverRef, forwardedRef);
  let handleKeyDown = useKeyDown();
  function handleMouseUp() {
    send({
      type: "LIST_MOUSE_UP" /* ListMouseUp */
    });
  }
  let commonProps = {
    hidden: !isExpanded,
    tabIndex: -1,
    ...props,
    ref,
    "data-reach-listbox-popover": "",
    onMouseUp: composeEventHandlers(onMouseUp, handleMouseUp),
    onBlur: composeEventHandlers(onBlur, handleBlur),
    onKeyDown: composeEventHandlers(onKeyDown, handleKeyDown)
  };
  function handleBlur(event) {
    let { nativeEvent } = event;
    requestAnimationFrame(() => {
      send({
        type: "BLUR" /* Blur */,
        relatedTarget: nativeEvent.relatedTarget || nativeEvent.target
      });
    });
  }
  return portal ? /* @__PURE__ */ React.createElement(Popover, {
    ...commonProps,
    as: Comp,
    targetRef: buttonRef,
    position,
    unstable_observableRefs,
    unstable_skipInitialPortalRender: true
  }) : /* @__PURE__ */ React.createElement(Comp, {
    ...commonProps
  });
});
ListboxPopoverImpl.displayName = "ListboxPopover";
var ListboxPopover2 = React.memo(ListboxPopoverImpl);
var ListboxList = React.forwardRef(function ListboxList2({ as: Comp = "ul", ...props }, forwardedRef) {
  let {
    listRef,
    ariaLabel,
    ariaLabelledBy,
    isExpanded,
    listboxId,
    stateData: { value, navigationValue }
  } = React.useContext(ListboxContext);
  let ref = useComposedRefs(forwardedRef, listRef);
  return /* @__PURE__ */ React.createElement(Comp, {
    "aria-activedescendant": useOptionId(isExpanded ? navigationValue : value),
    "aria-labelledby": ariaLabel ? void 0 : ariaLabelledBy,
    "aria-label": ariaLabel,
    role: "listbox",
    tabIndex: -1,
    ...props,
    ref,
    "data-reach-listbox-list": "",
    id: makeId("listbox", listboxId)
  });
});
ListboxList.displayName = "ListboxList";
var ListboxOption = React.forwardRef(function ListboxOption2({
  as: Comp = "li",
  children,
  disabled,
  index: indexProp,
  label: labelProp,
  onClick,
  onMouseDown,
  onMouseEnter,
  onMouseLeave,
  onMouseMove,
  onMouseUp,
  onTouchStart,
  value,
  ...props
}, forwardedRef) {
  let {
    highlightedOptionRef,
    selectedOptionRef,
    send,
    isExpanded,
    onValueChange,
    state,
    stateData: { value: listboxValue, navigationValue }
  } = React.useContext(ListboxContext);
  let [labelState, setLabel] = React.useState(labelProp);
  let label = labelProp || labelState || "";
  let ownRef = React.useRef(null);
  let [element, handleRefSet] = useStatefulRefValue(ownRef, null);
  let descendant = React.useMemo(() => {
    return {
      element,
      value,
      label,
      disabled: !!disabled
    };
  }, [disabled, element, label, value]);
  useDescendant(descendant, ListboxDescendantContext, indexProp);
  let getLabelFromDomNode = React.useCallback((node) => {
    if (!labelProp && node) {
      setLabel((prevState) => {
        if (node.textContent && prevState !== node.textContent) {
          return node.textContent;
        }
        return prevState || "";
      });
    }
  }, [labelProp]);
  let isHighlighted = navigationValue ? navigationValue === value : false;
  let isSelected = listboxValue === value;
  let ref = useComposedRefs(getLabelFromDomNode, forwardedRef, handleRefSet, isSelected ? selectedOptionRef : null, isHighlighted ? highlightedOptionRef : null);
  function handleMouseEnter() {
    send({
      type: "OPTION_MOUSE_ENTER" /* OptionMouseEnter */,
      value,
      disabled: !!disabled
    });
  }
  function handleTouchStart() {
    send({
      type: "OPTION_TOUCH_START" /* OptionTouchStart */,
      value,
      disabled: !!disabled
    });
  }
  function handleMouseLeave() {
    send({ type: "CLEAR_NAV_SELECTION" /* ClearNavSelection */ });
  }
  function handleMouseDown(event) {
    if (!isRightClick(event.nativeEvent)) {
      event.preventDefault();
      send({ type: "OPTION_MOUSE_DOWN" /* OptionMouseDown */ });
    }
  }
  function handleMouseUp(event) {
    if (!isRightClick(event.nativeEvent)) {
      send({
        type: "OPTION_MOUSE_UP" /* OptionMouseUp */,
        value,
        callback: onValueChange,
        disabled: !!disabled
      });
    }
  }
  function handleClick(event) {
    if (!isRightClick(event.nativeEvent)) {
      send({
        type: "OPTION_CLICK" /* OptionClick */,
        value,
        callback: onValueChange,
        disabled: !!disabled
      });
    }
  }
  function handleMouseMove() {
    if (state === "OPEN" /* Open */ || navigationValue !== value) {
      send({
        type: "OPTION_MOUSE_MOVE" /* OptionMouseMove */,
        value,
        disabled: !!disabled
      });
    }
  }
  return /* @__PURE__ */ React.createElement(Comp, {
    "aria-selected": (isExpanded ? isHighlighted : isSelected) || void 0,
    "aria-disabled": disabled || void 0,
    role: "option",
    ...props,
    ref,
    id: useOptionId(value),
    "data-reach-listbox-option": "",
    "data-current-nav": isHighlighted ? "" : void 0,
    "data-current-selected": isSelected ? "" : void 0,
    "data-label": label,
    "data-value": value,
    onClick: composeEventHandlers(onClick, handleClick),
    onMouseDown: composeEventHandlers(onMouseDown, handleMouseDown),
    onMouseEnter: composeEventHandlers(onMouseEnter, handleMouseEnter),
    onMouseLeave: composeEventHandlers(onMouseLeave, handleMouseLeave),
    onMouseMove: composeEventHandlers(onMouseMove, handleMouseMove),
    onMouseUp: composeEventHandlers(onMouseUp, handleMouseUp),
    onTouchStart: composeEventHandlers(onTouchStart, handleTouchStart)
  }, children);
});
ListboxOption.displayName = "ListboxOption";
var ListboxGroup = React.forwardRef(function ListboxGroup2({ as: Comp = "div", label, children, ...props }, forwardedRef) {
  let { listboxId } = React.useContext(ListboxContext);
  let labelId = makeId("label", useId(props.id), listboxId);
  return /* @__PURE__ */ React.createElement(ListboxGroupContext.Provider, {
    value: { labelId }
  }, /* @__PURE__ */ React.createElement(Comp, {
    "aria-labelledby": labelId,
    role: "group",
    ...props,
    "data-reach-listbox-group": "",
    ref: forwardedRef
  }, label && /* @__PURE__ */ React.createElement(ListboxGroupLabel, null, label), children));
});
ListboxGroup.displayName = "ListboxGroup";
var ListboxGroupLabel = React.forwardRef(function ListboxGroupLabel2({ as: Comp = "span", ...props }, forwardedRef) {
  let { labelId } = React.useContext(ListboxGroupContext);
  return /* @__PURE__ */ React.createElement(Comp, {
    role: "presentation",
    ...props,
    ref: forwardedRef,
    "data-reach-listbox-group-label": "",
    id: labelId
  });
});
ListboxGroupLabel.displayName = "ListboxGroupLabel";
function useListboxContext() {
  let {
    highlightedOptionRef,
    selectedOptionRef,
    listboxId,
    listboxValueLabel,
    isExpanded,
    stateData: { value }
  } = React.useContext(ListboxContext);
  return React.useMemo(() => ({
    id: listboxId,
    isExpanded,
    selectedOptionRef,
    highlightedOptionRef,
    value,
    valueLabel: listboxValueLabel
  }), [
    listboxId,
    isExpanded,
    value,
    listboxValueLabel,
    selectedOptionRef,
    highlightedOptionRef
  ]);
}
function isListboxExpanded(state) {
  return [
    "NAVIGATING" /* Navigating */,
    "OPEN" /* Open */,
    "DRAGGING" /* Dragging */,
    "INTERACTING" /* Interacting */
  ].includes(state);
}
function useKeyDown() {
  let {
    send,
    disabled: listboxDisabled,
    onValueChange,
    stateData: { navigationValue, typeaheadQuery }
  } = React.useContext(ListboxContext);
  let options = useDescendants(ListboxDescendantContext);
  let stableOnValueChange = useStableCallback(onValueChange);
  React.useEffect(() => {
    if (typeaheadQuery) {
      send({
        type: "UPDATE_AFTER_TYPEAHEAD" /* UpdateAfterTypeahead */,
        query: typeaheadQuery,
        callback: stableOnValueChange
      });
    }
    let timeout = window.setTimeout(() => {
      if (typeaheadQuery != null) {
        send({ type: "CLEAR_TYPEAHEAD" /* ClearTypeahead */ });
      }
    }, 1e3);
    return () => {
      window.clearTimeout(timeout);
    };
  }, [stableOnValueChange, send, typeaheadQuery]);
  let index = options.findIndex(({ value }) => value === navigationValue);
  let handleKeyDown = composeEventHandlers(function(event) {
    let { key } = event;
    let isSearching = isString(key) && key.length === 1;
    let navOption = options.find((option) => option.value === navigationValue);
    switch (key) {
      case "Enter":
        send({
          type: "KEY_DOWN_ENTER" /* KeyDownEnter */,
          value: navigationValue,
          callback: onValueChange,
          disabled: !!(navOption?.disabled || listboxDisabled)
        });
        return;
      case " ":
        event.preventDefault();
        send({
          type: "KEY_DOWN_SPACE" /* KeyDownSpace */,
          value: navigationValue,
          callback: onValueChange,
          disabled: !!(navOption?.disabled || listboxDisabled)
        });
        return;
      case "Escape":
        send({ type: "KEY_DOWN_ESCAPE" /* KeyDownEscape */ });
        return;
      case "Tab":
        let eventType = event.shiftKey ? "KEY_DOWN_SHIFT_TAB" /* KeyDownShiftTab */ : "KEY_DOWN_TAB" /* KeyDownTab */;
        send({ type: eventType });
        return;
      default:
        if (isSearching) {
          send({
            type: "KEY_DOWN_SEARCH" /* KeyDownSearch */,
            query: key,
            disabled: listboxDisabled
          });
        }
        return;
    }
  }, useDescendantKeyDown(ListboxDescendantContext, {
    currentIndex: index,
    orientation: "vertical",
    key: "index",
    rotate: true,
    filter: (option) => !option.disabled,
    callback(nextIndex) {
      send({
        type: "KEY_DOWN_NAVIGATE" /* KeyDownNavigate */,
        value: options[nextIndex].value,
        disabled: listboxDisabled
      });
    }
  }));
  return handleKeyDown;
}
function useOptionId(value) {
  let { listboxId } = React.useContext(ListboxContext);
  return value ? makeId(`option-${value}`, listboxId) : void 0;
}
function popoverContainsEventTarget(popover, target) {
  return !!(popover && popover.contains(target));
}
function useControlledStateSync(controlPropValue, internalValue, send) {
  let { current: isControlled } = React.useRef(controlPropValue != null);
  if (isControlled && controlPropValue !== internalValue) {
    send();
  }
}
export {
  Listbox,
  ListboxArrow2 as ListboxArrow,
  ListboxButton2 as ListboxButton,
  ListboxGroup,
  ListboxGroupLabel,
  ListboxInput,
  ListboxList,
  ListboxOption,
  ListboxPopover2 as ListboxPopover,
  useListboxContext
};
