"use strict";
/**
  * @reach/machine v0.18.0-pre.5
  *
  * Copyright (c) 2018-2022, React Training LLC
  *
  * This source code is licensed under the MIT license found in the
  * LICENSE.md file in the root directory of this source tree.
  *
  * @license MIT
  */


// src/reach-machine.ts
import * as React from "react";
import {
  assign,
  createMachine,
  interpret,
  InterpreterStatus
} from "@xstate/fsm";
import { isString, useConstant } from "@reach/utils";
var getServiceState = (service) => {
  let currentValue;
  service.subscribe((state) => {
    currentValue = state;
  }).unsubscribe();
  return currentValue;
};
function useMachine(initialMachine, refs, DEBUG) {
  let machineRef = React.useRef(initialMachine);
  let service = useConstant(() => interpret(machineRef.current).start());
  let lastEventType = React.useRef(null);
  let [state, setState] = React.useState(() => getServiceState(service));
  let send = React.useCallback((rawEvent) => {
    let event = isString(rawEvent) ? { type: rawEvent } : rawEvent;
    let refValues = unwrapRefs(refs);
    service.send({
      ...event,
      lastEventType: lastEventType.current,
      refs: refValues
    });
    lastEventType.current = event.type;
    if (true) {
      if (DEBUG) {
        console.group("Event Sent");
        console.log("Event:", event);
        console.groupEnd();
      }
    }
  }, [DEBUG]);
  React.useEffect(() => {
    service.subscribe(function setStateIfChanged(newState) {
      if (newState.changed) {
        setState(newState);
      }
    });
    return () => {
      service.stop();
    };
  }, [service]);
  React.useEffect(() => {
    if (true) {
      if (DEBUG && state.changed) {
        console.group("State Updated");
        console.log("State:", state);
        console.groupEnd();
      }
    }
  }, [DEBUG, state]);
  const memoizedState = React.useMemo(() => ({
    ...state,
    matches: (value) => value === state.value
  }), [state.changed, state.context, state.value]);
  return [memoizedState, send, service];
}
function unwrapRefs(refs) {
  return Object.entries(refs).reduce((value, [name, ref]) => {
    value[name] = ref.current;
    return value;
  }, {});
}
function useCreateMachine(machineDefinition, options) {
  return useConstant(() => createMachine(machineDefinition, options));
}
export {
  InterpreterStatus,
  assign,
  createMachine,
  interpret,
  unwrapRefs,
  useCreateMachine,
  useMachine
};
