"use strict";
/**
  * @reach/popover v0.18.0-pre.5
  *
  * Copyright (c) 2018-2022, React Training LLC
  *
  * This source code is licensed under the MIT license found in the
  * LICENSE.md file in the root directory of this source tree.
  *
  * @license MIT
  */


// src/reach-popover.tsx
import * as React from "react";
import { Portal } from "@reach/portal";
import { useRect } from "@reach/rect";
import { getOwnerDocument, useComposedRefs } from "@reach/utils";
import { tabbable } from "tabbable";
var Popover = React.forwardRef(function Popover2({ unstable_skipInitialPortalRender, ...props }, ref) {
  return /* @__PURE__ */ React.createElement(Portal, {
    unstable_skipInitialRender: unstable_skipInitialPortalRender
  }, /* @__PURE__ */ React.createElement(PopoverImpl, {
    ref,
    ...props
  }));
});
Popover.displayName = "Popover";
var PopoverImpl = React.forwardRef(function PopoverImpl2({
  as: Comp = "div",
  targetRef,
  position = positionDefault,
  unstable_observableRefs = [],
  ...props
}, forwardedRef) {
  const popoverRef = React.useRef(null);
  const popoverRect = useRect(popoverRef, { observe: !props.hidden });
  const targetRect = useRect(targetRef, { observe: true });
  const ref = useComposedRefs(popoverRef, forwardedRef);
  useSimulateTabNavigationForReactTree(targetRef, popoverRef);
  return /* @__PURE__ */ React.createElement(Comp, {
    "data-reach-popover": "",
    ref,
    ...props,
    style: {
      position: "absolute",
      ...getStyles(position, targetRect, popoverRect, ...unstable_observableRefs),
      ...props.style
    }
  });
});
PopoverImpl.displayName = "PopoverImpl";
function getStyles(position, targetRect, popoverRect, ...unstable_observableRefs) {
  return popoverRect ? position(targetRect, popoverRect, ...unstable_observableRefs.map((ref) => ref.current)) : { visibility: "hidden" };
}
function getTopPosition(targetRect, popoverRect, isDirectionUp) {
  return {
    top: isDirectionUp ? `${targetRect.top - popoverRect.height + window.pageYOffset}px` : `${targetRect.top + targetRect.height + window.pageYOffset}px`
  };
}
var positionDefault = (targetRect, popoverRect) => {
  if (!targetRect || !popoverRect) {
    return {};
  }
  const { directionRight, directionUp } = getCollisions(targetRect, popoverRect);
  return {
    left: directionRight ? `${targetRect.right - popoverRect.width + window.pageXOffset}px` : `${targetRect.left + window.pageXOffset}px`,
    ...getTopPosition(targetRect, popoverRect, directionUp)
  };
};
var positionRight = (targetRect, popoverRect) => {
  if (!targetRect || !popoverRect) {
    return {};
  }
  const { directionLeft, directionUp } = getCollisions(targetRect, popoverRect);
  return {
    left: directionLeft ? `${targetRect.left + window.pageXOffset}px` : `${targetRect.right - popoverRect.width + window.pageXOffset}px`,
    ...getTopPosition(targetRect, popoverRect, directionUp)
  };
};
var positionMatchWidth = (targetRect, popoverRect) => {
  if (!targetRect || !popoverRect) {
    return {};
  }
  const { directionUp } = getCollisions(targetRect, popoverRect);
  return {
    width: targetRect.width,
    left: targetRect.left,
    ...getTopPosition(targetRect, popoverRect, directionUp)
  };
};
function getCollisions(targetRect, popoverRect, offsetLeft = 0, offsetBottom = 0) {
  const collisions = {
    top: targetRect.top - popoverRect.height < 0,
    right: window.innerWidth < targetRect.left + popoverRect.width - offsetLeft,
    bottom: window.innerHeight < targetRect.bottom + popoverRect.height - offsetBottom,
    left: targetRect.left + targetRect.width - popoverRect.width < 0
  };
  const directionRight = collisions.right && !collisions.left;
  const directionLeft = collisions.left && !collisions.right;
  const directionUp = collisions.bottom && !collisions.top;
  const directionDown = collisions.top && !collisions.bottom;
  return { directionRight, directionLeft, directionUp, directionDown };
}
function useSimulateTabNavigationForReactTree(triggerRef, popoverRef) {
  const ownerDocument = getOwnerDocument(triggerRef.current);
  function handleKeyDown(event) {
    if (event.key === "Tab" && popoverRef.current && tabbable(popoverRef.current).length === 0) {
      return;
    }
    if (event.key === "Tab" && event.shiftKey) {
      if (shiftTabbedFromElementAfterTrigger(event)) {
        focusLastTabbableInPopover(event);
      } else if (shiftTabbedOutOfPopover(event)) {
        focusTriggerRef(event);
      } else if (shiftTabbedToBrowserChrome(event)) {
        disableTabbablesInPopover();
      }
    } else if (event.key === "Tab") {
      if (tabbedFromTriggerToPopover()) {
        focusFirstPopoverTabbable(event);
      } else if (tabbedOutOfPopover()) {
        focusTabbableAfterTrigger(event);
      } else if (tabbedToBrowserChrome(event)) {
        disableTabbablesInPopover();
      }
    }
  }
  React.useEffect(() => {
    ownerDocument.addEventListener("keydown", handleKeyDown);
    return () => {
      ownerDocument.removeEventListener("keydown", handleKeyDown);
    };
  }, []);
  function getElementAfterTrigger() {
    const elements = tabbable(ownerDocument);
    const targetIndex = elements && triggerRef.current ? elements.indexOf(triggerRef.current) : -1;
    const elementAfterTrigger = elements && elements[targetIndex + 1];
    return popoverRef.current && popoverRef.current.contains(elementAfterTrigger || null) ? false : elementAfterTrigger;
  }
  function tabbedFromTriggerToPopover() {
    return triggerRef.current ? triggerRef.current === ownerDocument.activeElement : false;
  }
  function focusFirstPopoverTabbable(event) {
    const elements = popoverRef.current && tabbable(popoverRef.current);
    if (elements && elements[0]) {
      event.preventDefault();
      elements[0].focus();
    }
  }
  function tabbedOutOfPopover() {
    const inPopover = popoverRef.current ? popoverRef.current.contains(ownerDocument.activeElement || null) : false;
    if (inPopover) {
      const elements = popoverRef.current && tabbable(popoverRef.current);
      return Boolean(elements && elements[elements.length - 1] === ownerDocument.activeElement);
    }
    return false;
  }
  function focusTabbableAfterTrigger(event) {
    const elementAfterTrigger = getElementAfterTrigger();
    if (elementAfterTrigger) {
      event.preventDefault();
      elementAfterTrigger.focus();
    }
  }
  function shiftTabbedFromElementAfterTrigger(event) {
    if (!event.shiftKey)
      return;
    const elementAfterTrigger = getElementAfterTrigger();
    return event.target === elementAfterTrigger;
  }
  function focusLastTabbableInPopover(event) {
    const elements = popoverRef.current && tabbable(popoverRef.current);
    const last = elements && elements[elements.length - 1];
    if (last) {
      event.preventDefault();
      last.focus();
    }
  }
  function shiftTabbedOutOfPopover(event) {
    const elements = popoverRef.current && tabbable(popoverRef.current);
    if (elements) {
      return elements.length === 0 ? false : event.target === elements[0];
    }
    return false;
  }
  function focusTriggerRef(event) {
    event.preventDefault();
    triggerRef.current?.focus();
  }
  function tabbedToBrowserChrome(event) {
    const elements = popoverRef.current ? tabbable(ownerDocument).filter((element) => !popoverRef.current.contains(element)) : null;
    return elements ? event.target === elements[elements.length - 1] : false;
  }
  function shiftTabbedToBrowserChrome(event) {
    return event.target === tabbable(ownerDocument)[0];
  }
  let restoreTabIndexTupl\u00E9s = [];
  function disableTabbablesInPopover() {
    const elements = popoverRef.current && tabbable(popoverRef.current);
    if (elements) {
      elements.forEach((element) => {
        restoreTabIndexTupl\u00E9s.push([element, element.tabIndex]);
        element.tabIndex = -1;
      });
      ownerDocument.addEventListener("focusin", enableTabbablesInPopover);
    }
  }
  function enableTabbablesInPopover() {
    ownerDocument.removeEventListener("focusin", enableTabbablesInPopover);
    restoreTabIndexTupl\u00E9s.forEach(([element, tabIndex]) => {
      element.tabIndex = tabIndex;
    });
  }
}
export {
  Popover,
  getCollisions,
  positionDefault,
  positionMatchWidth,
  positionRight
};
