"use strict";
/**
  * @reach/utils v0.18.0-pre.5
  *
  * Copyright (c) 2018-2022, React Training LLC
  *
  * This source code is licensed under the MIT license found in the
  * LICENSE.md file in the root directory of this source tree.
  *
  * @license MIT
  */


// src/can-use-dom.ts
function canUseDOM() {
  return !!(typeof window !== "undefined" && window.document && window.document.createElement);
}

// src/clone-valid-element.ts
import { cloneElement, isValidElement } from "react";
function cloneValidElement(element, props, ...children) {
  return isValidElement(element) ? cloneElement(element, props, ...children) : element;
}

// src/compose-event-handlers.ts
function composeEventHandlers(theirHandler, ourHandler) {
  return (event) => {
    theirHandler && theirHandler(event);
    if (!event.defaultPrevented) {
      return ourHandler(event);
    }
  };
}

// src/compose-refs.ts
import { useCallback } from "react";

// src/type-check.ts
function isBoolean(value) {
  return typeof value === "boolean";
}
function isFunction(value) {
  return !!(value && {}.toString.call(value) == "[object Function]");
}
function isNumber(value) {
  return typeof value === "number" && !isNaN(value);
}
function isString(value) {
  return typeof value === "string";
}

// src/compose-refs.ts
function assignRef(ref, value) {
  if (ref == null)
    return;
  if (isFunction(ref)) {
    ref(value);
  } else {
    try {
      ref.current = value;
    } catch (error) {
      throw new Error(`Cannot assign value "${value}" to ref "${ref}"`);
    }
  }
}
function useComposedRefs(...refs) {
  return useCallback((node) => {
    for (let ref of refs) {
      assignRef(ref, node);
    }
  }, refs);
}

// src/owner-document.ts
function getOwnerDocument(element) {
  return canUseDOM() ? element ? element.ownerDocument : document : null;
}
function getOwnerWindow(element) {
  let ownerDocument = getOwnerDocument(element);
  return ownerDocument ? ownerDocument.defaultView || window : null;
}

// src/computed-styles.ts
function getComputedStyles(element) {
  let ownerWindow = getOwnerWindow(element);
  if (ownerWindow) {
    return ownerWindow.getComputedStyle(element, null);
  }
  return null;
}
function getComputedStyle(element, styleProp) {
  return getComputedStyles(element)?.getPropertyValue(styleProp) || null;
}

// src/context.tsx
import * as React from "react";
function createNamedContext(name, defaultValue) {
  let Ctx = React.createContext(defaultValue);
  Ctx.displayName = name;
  return Ctx;
}
function createContext2(rootComponentName, defaultContext) {
  let Ctx = React.createContext(defaultContext);
  function Provider(props) {
    let { children, ...context } = props;
    let value = React.useMemo(() => context, Object.values(context));
    return /* @__PURE__ */ React.createElement(Ctx.Provider, {
      value
    }, children);
  }
  function useContext2(callerComponentName) {
    let context = React.useContext(Ctx);
    if (context) {
      return context;
    }
    if (defaultContext) {
      return defaultContext;
    }
    throw Error(`${callerComponentName} must be rendered inside of a ${rootComponentName} component.`);
  }
  Ctx.displayName = `${rootComponentName}Context`;
  Provider.displayName = `${rootComponentName}Provider`;
  return [Provider, useContext2];
}

// src/dev-utils.ts
import { useRef, useEffect } from "react";
function useControlledSwitchWarning(controlledValue, controlledPropName, componentName) {
  if (true) {
    let controlledRef = useRef(controlledValue != null);
    let nameCache = useRef({ componentName, controlledPropName });
    useEffect(() => {
      nameCache.current = { componentName, controlledPropName };
    }, [componentName, controlledPropName]);
    useEffect(() => {
      let { current: wasControlled } = controlledRef;
      let { componentName: componentName2, controlledPropName: controlledPropName2 } = nameCache.current;
      let isControlled = controlledValue != null;
      if (wasControlled !== isControlled) {
        console.error(`A component is changing an ${wasControlled ? "" : "un"}controlled \`${controlledPropName2}\` state of ${componentName2} to be ${wasControlled ? "un" : ""}controlled. This is likely caused by the value changing from undefined to a defined value, which should not happen. Decide between using a controlled or uncontrolled ${componentName2} element for the lifetime of the component.
      More info: https://fb.me/react-controlled-components`);
      }
    }, [controlledValue]);
  }
}

// src/get-document-dimensions.ts
function getDocumentDimensions(element) {
  let ownerDocument = getOwnerDocument(element);
  let ownerWindow = ownerDocument.defaultView || window;
  if (!ownerDocument) {
    return {
      width: 0,
      height: 0
    };
  }
  return {
    width: ownerDocument.documentElement.clientWidth ?? ownerWindow.innerWidth,
    height: ownerDocument.documentElement.clientHeight ?? ownerWindow.innerHeight
  };
}

// src/get-scroll-position.ts
function getScrollPosition(element) {
  let ownerWindow = getOwnerWindow(element);
  if (!ownerWindow) {
    return {
      scrollX: 0,
      scrollY: 0
    };
  }
  return {
    scrollX: ownerWindow.scrollX,
    scrollY: ownerWindow.scrollY
  };
}

// src/get-scrollbar-offset.ts
function getScrollbarOffset() {
  try {
    if (window.innerWidth > document.documentElement.clientWidth) {
      return window.innerWidth - document.documentElement.clientWidth;
    }
  } catch (err) {
  }
  return 0;
}

// src/is-right-click.ts
function isRightClick(nativeEvent) {
  return "which" in nativeEvent ? nativeEvent.which === 3 : "button" in nativeEvent ? nativeEvent.button === 2 : false;
}

// src/make-id.ts
function makeId(...args) {
  return args.filter((val) => val != null).join("--");
}

// src/noop.ts
function noop() {
}

// src/use-constant.ts
import { useRef as useRef2 } from "react";
function useConstant(fn) {
  const ref = useRef2();
  if (!ref.current) {
    ref.current = { v: fn() };
  }
  return ref.current.v;
}

// src/use-controlled-state.ts
import { useState, useRef as useRef3, useCallback as useCallback2 } from "react";
function useControlledState({
  controlledValue,
  defaultValue,
  calledFrom = "A component"
}) {
  let wasControlled = controlledValue !== void 0;
  let isControlledRef = useRef3(wasControlled);
  if (true) {
    if (!isControlledRef.current && wasControlled) {
      console.warn(`${calledFrom} is changing from controlled to uncontrolled. Components should not switch from controlled to uncontrolled (or vice versa). Decide between using a controlled or uncontrolled value for the lifetime of the component.`);
    }
    if (isControlledRef.current && !wasControlled) {
      console.warn(`${calledFrom} is changing from uncontrolled to controlled. Components should not switch from uncontrolled to controlled (or vice versa). Decide between using a controlled or uncontrolled value for the lifetime of the component.`);
    }
  }
  let [valueState, setValue] = useState(isControlledRef.current ? controlledValue : defaultValue);
  let set = useCallback2((n) => {
    if (!isControlledRef.current) {
      setValue(n);
    }
  }, []);
  return [isControlledRef.current ? controlledValue : valueState, set];
}

// src/use-event-listener.ts
import { useRef as useRef4, useEffect as useEffect2 } from "react";
function useEventListener(eventName, listener, element = window) {
  const savedHandler = useRef4(listener);
  useEffect2(() => {
    savedHandler.current = listener;
  }, [listener]);
  useEffect2(() => {
    const isSupported = element && element.addEventListener;
    if (!isSupported) {
      if (true) {
        console.warn("Event listener not supported on the element provided");
      }
      return;
    }
    function eventListener(event) {
      savedHandler.current(event);
    }
    element.addEventListener(eventName, eventListener);
    return () => {
      element.removeEventListener(eventName, eventListener);
    };
  }, [eventName, element]);
}

// src/use-focus-change.ts
import { useRef as useRef5, useEffect as useEffect3 } from "react";
function useFocusChange(handleChange = console.log, when = "focus", ownerDocument = document) {
  let lastActiveElement = useRef5(ownerDocument.activeElement);
  useEffect3(() => {
    lastActiveElement.current = ownerDocument.activeElement;
    function onChange(event) {
      if (lastActiveElement.current !== ownerDocument.activeElement) {
        handleChange(ownerDocument.activeElement, lastActiveElement.current, event);
        lastActiveElement.current = ownerDocument.activeElement;
      }
    }
    ownerDocument.addEventListener(when, onChange, true);
    return () => {
      ownerDocument.removeEventListener(when, onChange);
    };
  }, [when, handleChange, ownerDocument]);
}

// src/use-force-update.ts
import { useState as useState2, useCallback as useCallback3 } from "react";
function useForceUpdate() {
  let [, dispatch] = useState2(/* @__PURE__ */ Object.create(null));
  return useCallback3(() => {
    dispatch(/* @__PURE__ */ Object.create(null));
  }, []);
}

// src/use-isomorphic-layout-effect.ts
import { useEffect as useEffect4, useLayoutEffect } from "react";
var useIsomorphicLayoutEffect = canUseDOM() ? useLayoutEffect : useEffect4;

// src/use-lazy-ref.ts
import { useRef as useRef6 } from "react";
function useLazyRef(fn) {
  let isSet = useRef6(false);
  let ref = useRef6();
  if (!isSet.current) {
    isSet.current = true;
    ref.current = fn();
  }
  return ref;
}

// src/use-previous.ts
import { useRef as useRef7, useEffect as useEffect5 } from "react";
function usePrevious(value) {
  const ref = useRef7(null);
  useEffect5(() => {
    ref.current = value;
  }, [value]);
  return ref.current;
}

// src/use-stable-callback.ts
import { useRef as useRef8, useEffect as useEffect6, useCallback as useCallback4 } from "react";
function createStableCallbackHook(useEffectHook, callback) {
  let callbackRef = useRef8(callback);
  useEffectHook(() => {
    callbackRef.current = callback;
  });
  return useCallback4((...args) => {
    callbackRef.current && callbackRef.current(...args);
  }, []);
}
function useStableCallback(callback) {
  return createStableCallbackHook(useEffect6, callback);
}
function useStableLayoutCallback(callback) {
  return createStableCallbackHook(useIsomorphicLayoutEffect, callback);
}

// src/use-stateful-ref-value.ts
import { useState as useState3, useCallback as useCallback5 } from "react";
function useStatefulRefValue(ref, initialState) {
  let [state, setState] = useState3(initialState);
  let callbackRef = useCallback5((refValue) => {
    ref.current = refValue;
    setState(refValue);
  }, []);
  return [state, callbackRef];
}

// src/use-update-effect.ts
import { useRef as useRef9, useEffect as useEffect7 } from "react";
function useUpdateEffect(effect, deps) {
  const mounted = useRef9(false);
  useEffect7(() => {
    if (mounted.current) {
      effect();
    } else {
      mounted.current = true;
    }
  }, deps);
}
export {
  assignRef,
  canUseDOM,
  cloneValidElement,
  composeEventHandlers,
  createContext2 as createContext,
  createNamedContext,
  getComputedStyle,
  getComputedStyles,
  getDocumentDimensions,
  getOwnerDocument,
  getOwnerWindow,
  getScrollPosition,
  getScrollbarOffset,
  isBoolean,
  isFunction,
  isNumber,
  isRightClick,
  isString,
  makeId,
  noop,
  useComposedRefs,
  useConstant,
  useControlledState,
  useControlledSwitchWarning,
  useEventListener,
  useFocusChange,
  useForceUpdate,
  useIsomorphicLayoutEffect,
  useLazyRef,
  usePrevious,
  useStableCallback,
  useStableLayoutCallback,
  useStatefulRefValue,
  useUpdateEffect
};
