'use strict';

var React = require('react');
var d3 = require('d3');
var DataSeries = require('./DataSeries');
var utils = require('../utils');

var $__0=      require('../common'),Chart=$__0.Chart,XAxis=$__0.XAxis,YAxis=$__0.YAxis;
var $__1=     require('../mixins'),CartesianChartPropsMixin=$__1.CartesianChartPropsMixin,ViewBoxMixin=$__1.ViewBoxMixin;

module.exports = React.createClass({

  mixins: [ CartesianChartPropsMixin, ViewBoxMixin ],

  displayName: 'BarChart',

  propTypes: {
    chartClassName:         React.PropTypes.string,
    data:                   React.PropTypes.array.isRequired,
    hoverAnimation:         React.PropTypes.bool,
    height:                 React.PropTypes.number,
    margins:                React.PropTypes.object,
    rangeRoundBandsPadding: React.PropTypes.number,
    // https://github.com/mbostock/d3/wiki/Stack-Layout#offset
    stackOffset:            React.PropTypes.oneOf(['silhouette', 'expand', 'wigget', 'zero']),
    valuesAccessor:         React.PropTypes.func,
    title:                  React.PropTypes.string,
    width:                  React.PropTypes.number,
    xAxisClassName:         React.PropTypes.string,
    yAxisClassName:         React.PropTypes.string,
    yAxisTickCount:         React.PropTypes.number,
  },

  getDefaultProps:function() {
    return {
      chartClassName:         'rd3-barchart',
      hoverAnimation:         true,
      margins:                {top: 10, right: 20, bottom: 40, left: 45},
      rangeRoundBandsPadding: 0.25,
      stackOffset:            'zero',
      valuesAccessor:         function(d)  {return d.values;},
      xAxisClassName:         'rd3-barchart-xaxis',
      yAxisClassName:         'rd3-barchart-yaxis',
      yAxisTickCount:         4,
    };
  },

  _getStackedValuesMaxY:function(_data) {
    // in stacked bar chart, the maximum height we need for 
    // yScale domain is the sum of y0 + y
    var $__0=    this.props,valuesAccessor=$__0.valuesAccessor;
    return d3.max(_data, function(d) {
      return d3.max(valuesAccessor(d), function(d) {
        // where y0, y is generated by d3.layout.stack()
        return d.y0 + d.y;
      });
    });
  },

  _getLabels:function(firstSeries) {
    // we only need first series to get all the labels
    var $__0=     this.props,valuesAccessor=$__0.valuesAccessor,xAccessor=$__0.xAccessor;
    return valuesAccessor(firstSeries).map(xAccessor);
  },

  _stack:function() {
    var $__0=       this.props,stackOffset=$__0.stackOffset,xAccessor=$__0.xAccessor,yAccessor=$__0.yAccessor,valuesAccessor=$__0.valuesAccessor;
    return d3.layout.stack()
                    .offset(stackOffset)
                    .x(xAccessor)
                    .y(yAccessor)
                    .values(valuesAccessor);
  },

  render:function() {

    var props = this.props;

    var _data = this._stack()(props.data);

    var margins = props.margins;

    var innerHeight = props.height - ( margins.top + margins.bottom );
    var innerWidth = props.width - ( margins.left + margins.right );

    var xScale = d3.scale.ordinal()
      .domain(this._getLabels(_data[0]))
      .rangeRoundBands([0, innerWidth], props.rangeRoundBandsPadding);

    var yScale = d3.scale.linear()
      .range([innerHeight, 0])
      .domain([0, this._getStackedValuesMaxY(_data)]);

    var trans = ("translate(" +  margins.left + "," +  margins.top + ")");

    return (
      React.createElement(Chart, {
        viewBox: props.viewBox, 
        legend: props.legend, 
        data: props.data, 
        margins: props.margins, 
        colors: props.colors, 
        colorAccessor: props.colorAccessor, 
        width: props.width, 
        height: props.height, 
        title: props.title
      }, 
        React.createElement("g", {transform: trans, className: props.chartClassName}, 
          React.createElement(YAxis, {
            yAxisClassName: props.yAxisClassName, 
            yAxisTickValues: props.yAxisTickValues, 
            yAxisLabel: props.yAxisLabel, 
            yAxisLabelOffset: props.yAxisLabelOffset, 
            yScale: yScale, 
            margins: margins, 
            yAxisTickCount: props.yAxisTickCount, 
            tickFormatting: props.yAxisFormatter, 
            width: innerWidth, 
            height: innerHeight, 
            xOrient: props.xOrient, 
            yOrient: props.yOrient, 
            gridHorizontal: props.gridHorizontal, 
            gridHorizontalStroke: props.gridHorizontalStroke, 
            gridHorizontalStrokeWidth: props.gridHorizontalStrokeWidth, 
            gridHorizontalStrokeDash: props.gridHorizontalStrokeDash}
          ), 
          React.createElement(XAxis, {
            xAxisClassName: props.xAxisClassName, 
            xAxisTickValues: props.xAxisTickValues, 
            xAxisLabel: props.xAxisLabel, 
            xAxisLabelOffset: props.xAxisLabelOffset, 
            xScale: xScale, 
            margins: margins, 
            tickFormatting: props.xAxisFormatter, 
            width: innerWidth, 
            height: innerHeight, 
            xOrient: props.xOrient, 
            yOrient: props.yOrient, 
            gridVertical: props.gridVertical, 
            gridVerticalStroke: props.gridVerticalStroke, 
            gridVerticalStrokeWidth: props.gridVerticalStrokeWidth, 
            gridVerticalStrokeDash: props.gridVerticalStrokeDash}
          ), 
          React.createElement(DataSeries, {
            yScale: yScale, 
            xScale: xScale, 
            margins: margins, 
            _data: _data, 
            width: innerWidth, 
            height: innerHeight, 
            colors: props.colors, 
            colorAccessor: props.colorAccessor, 
            hoverAnimation: props.hoverAnimation, 
            valuesAccessor: props.valuesAccessor}
            )
        )
      )
    );
  }

});
