'use strict'

var React  = require('react')

var moment   = require('moment')
var assign   = require('object-assign')
var asConfig = require('./utils/asConfig')

var MonthView  = require('./MonthView')
var YearView   = require('./YearView')
var DecadeView = require('./DecadeView')
var Header     = require('./Header')

var toMoment   = require('./toMoment')

var hasOwn = function(obj, key){
    return Object.prototype.hasOwnProperty.call(obj, key)
}

var Views = {
    month : MonthView,
    year  : YearView,
    decade: DecadeView
}

function emptyFn(){}

var DatePicker = React.createClass({

    displayName: 'DatePicker',

    propTypes: {
        todayText: React.PropTypes.string,
        gotoSelectedText: React.PropTypes.string,

        renderFooter: React.PropTypes.func,
        onChange: React.PropTypes.func,

        date: React.PropTypes.any,
        viewDate: React.PropTypes.any
    },

    getViewOrder: function() {
        return this.props.viewOrder || ['month', 'year', 'decade']
    },

    getDefaultProps: function() {
        var props = assign({}, asConfig(), {
            navOnDateClick: true,
            defaultStyle: {
                boxSizing: 'border-box'
            }
        })

        delete props.viewDate
        delete props.date

        return props
    },

    getInitialState: function() {
        return {
            view       : this.props.defaultView,
            viewDate   : this.props.defaultViewDate,
            defaultDate: this.props.defaultDate
        }
    },

    getViewName: function() {
        var view = this.props.view != null?
                    this.props.view:
                    this.state.view

        return view || 'month'
    },

    addViewIndex: function(amount) {
        var viewName = this.getViewName()

        var order = this.getViewOrder()
        var index = order.indexOf(viewName)

        index += amount

        return index % order.length
    },

    getNextViewName: function() {
        return this.getViewOrder()[this.addViewIndex(1)]
    },

    getPrevViewName: function() {
        return this.getViewOrder()[this.addViewIndex(-1)]
    },

    getView: function() {
        var views = this.props.views || Views
        return views[this.getViewName()] || views.month
    },

    getViewFactory: function() {
        var view = this.getView()

        if (React.createFactory && view && view.prototype && typeof view.prototype.render == 'function'){
            view.__factory = view.__factory || React.createFactory(view)
            view = view.__factory
        }

        return view
    },

    getViewDate: function() {
        var date = hasOwn(this.props, 'viewDate')?
                        this.props.viewDate:
                        this.state.viewDate

        date = date || this.viewMoment || this.getDate() || new Date()

        if (moment.isMoment(date)){
            //in order to strip the locale - the date picker may have had its locale changed
            //between two render calls. If we don't strip this, moment(mom) returns a new moment
            //with the locale of mom, which is not what we want
            date = +date
        }

        date = this.toMoment(date)

        return date
    },

    getDate: function() {
        var date

        if (hasOwn(this.props, 'date')){
            date = this.props.date
        } else {
            date = this.state.defaultDate
        }

        return date? this.toMoment(date): null
    },

    render: function() {

        var props = assign({}, this.props)

        this.toMoment = function(value, dateFormat){
            return toMoment(value, dateFormat || props.dateFormat, { locale: props.locale })
        }

        var view  = this.getViewFactory()

        props.date = this.getDate()

        var dateString = (props.date == null ? '' : props.date.format(this.props.dateFormat))

        props.viewDate   = this.viewMoment = this.getViewDate()
        props.locale     = this.props.locale
        props.localeData = moment.localeData(props.locale)

        props.renderDay   = this.props.renderDay
        props.onRenderDay = this.props.onRenderDay

        // props.onChange  = this.handleChange
        // props.onSelect  = this.handleSelect

        var className = (this.props.className || '') + ' date-picker'

        props.style = this.prepareStyle(props)

        var viewProps = props
        var viewProps = asConfig(props)

        viewProps.dateString = dateString
        viewProps.localeData = props.localeData
        viewProps.onSelect = this.handleSelect
        viewProps.onChange = this.handleChange

        return (
            <div className={className} style={props.style} {...this.props}>
                <div className="dp-inner" style={{width: '100%', height: '100%'}}>
                    {this.renderHeader(view, props)}

                    <div className="dp-body" style={{flex: 1}}>
                        <div className="dp-anim-target">
                        {view(viewProps)}
                        </div>
                    </div>

                    {this.renderFooter(props)}
                </div>
            </div>
        )
    },

    prepareStyle: function(props) {
        var style = assign({}, props.defaultStyle, props.style)

        return style
    },

    renderFooter: function(props) {
        if (this.props.hideFooter){
            return
        }

        if (this.props.today){
            console.warn('Please use "todayText" prop instead of "today"!')
        }
        if (this.props.gotoSelected){
            console.warn('Please use "gotoSelectedText" prop instead of "gotoSelected"!')
        }

        var todayText    = this.props.todayText || 'Today'
        var gotoSelectedText = this.props.gotoSelectedText || 'Go to selected'

        var footerProps = {
            todayText       : todayText,
            gotoSelectedText: gotoSelectedText,
            gotoToday       : this.gotoNow,
            gotoSelected    : this.gotoSelected.bind(this, props),
            date            : props.date,
            viewDate        : props.viewDate
        }

        var result
        if (typeof this.props.footerFactory == 'function'){
            result = this.props.footerFactory(footerProps)
        }

        if (result !== undefined){
            return result
        }

        return (
            <div className="dp-footer">
                <div className="dp-footer-today" onClick={footerProps.gotoToday}>
                    {todayText}
                </div>
                <div className="dp-footer-selected" onClick={footerProps.gotoSelected}>
                    {gotoSelectedText}
                </div>
            </div>
        )
    },

    gotoNow: function() {
        this.gotoDate(+new Date())
    },

    gotoSelected: function(props) {
        this.gotoDate(props.date || +new Date())
    },

    gotoDate: function(value) {

        this.setView('month')

        this.setViewDate(value)
    },

    getViewColspan: function(){
        var map = {
            month : 5,
            year  : 2,
            decade: 2
        }

        return map[this.getViewName()]
    },

    renderHeader: function(view, props) {

        if (this.props.hideHeader){
            return
        }

        props = props || this.props

        var viewDate   = this.getViewDate()
        var headerText = this.getView().getHeaderText(viewDate, props)

        var colspan = this.getViewColspan()
        var prev    = this.props.navPrev
        var next    = this.props.navNext

        return <Header
                prevText={prev}
                nextText={next}
                colspan={colspan}
                onPrev={this.handleNavPrev}
                onNext={this.handleNavNext}
                onChange={this.handleViewChange}
            >
            {headerText}
        </Header>
    },

    handleRenderDay: function (date) {
        return (this.props.renderDay || emptyFn)(date) || []
    },

    handleViewChange: function() {
        this.setView(this.getNextViewName())
    },

    /**
     * Use this method to set the view.
     *
     * @param {String} view 'month'/'year'/'decade'
     *
     * It calls onViewChange, and if the view is uncontrolled, also sets it is state,
     * so the datepicker gets re-rendered view the new view
     *
     */
    setView: function(view) {

        if (typeof this.props.onViewChange == 'function'){
            this.props.onViewChange(view)
        }

        if (this.props.view == null){
            this.setState({
                view: view
            })
        }
    },

    setViewDate: function(moment) {

        moment = this.toMoment(moment)

        var fn = this.props.onViewDateChange

        if (typeof fn == 'function'){

            var text = moment.format(this.props.dateFormat)
            var view = this.getViewName()

            fn(text, moment, view)
        }

        if (!hasOwn(this.props, 'viewDate')){
            this.setState({
                viewDate: moment
            })
        }
    },

    getNext: function() {
        var current = this.getViewDate()
        var toMoment = this.toMoment

        return ({
            month: function() {
                return toMoment(current).add(1, 'month')
            },
            year: function() {
                return toMoment(current).add(1, 'year')
            },
            decade: function() {
                return toMoment(current).add(10, 'year')
            }
        })[this.getViewName()]()
    },

    getPrev: function() {
        var current = this.getViewDate()
        var toMoment = this.toMoment

        return ({
            month: function() {
                return toMoment(current).add(-1, 'month')
            },
            year: function() {
                return toMoment(current).add(-1, 'year')
            },
            decade: function() {
                return toMoment(current).add(-10, 'year')
            }
        })[this.getViewName()]()
    },

    handleNavigation: function(direction, event) {
        var viewMoment = direction == -1?
                            this.getPrev():
                            this.getNext()

        this.setViewDate(viewMoment)

        if (typeof this.props.onNav === 'function'){
            var text = viewMoment.format(this.props.dateFormat)
            var view = this.getViewName()

            this.props.onNav(text, viewMoment, view, direction, event)
        }
    },

    handleNavPrev: function(event) {
        this.handleNavigation(-1, event)
    },

    handleNavNext: function(event) {
        this.handleNavigation(1, event)
    },

    handleChange: function(date, event) {
        date = this.toMoment(date)

        if (this.props.navOnDateClick){
            var viewDate = this.toMoment(this.getViewDate())

            //it's not enough to compare months, since the year can change as well
            //
            //also it's ok to hardcode the format here
            var viewMonth = viewDate.format('YYYY-MM')
            var dateMonth = date.format('YYYY-MM')

            if (dateMonth > viewMonth){
                this.handleNavNext(event)
            } else if (dateMonth < viewMonth){
                this.handleNavPrev(event)
            }
        }

        var text = date.format(this.props.dateFormat)

        if (!hasOwn(this.props, 'date')){
            this.setState({
                defaultDate: text
            })
        }

        ;(this.props.onChange || emptyFn)(text, date, event)
    },

    handleSelect: function(date, event) {
        var viewName = this.getViewName()

        var property = ({
            decade: 'year',
            year  : 'month'
        })[viewName]

        var value      = date.get(property)
        var viewMoment = this.toMoment(this.getViewDate()).set(property, value)
        var view       = this.getPrevViewName()

        this.setViewDate(viewMoment)

        this.setView(view)

        if (typeof this.props.onSelect === 'function'){
            var text = viewMoment.format(this.props.dateFormat)
            this.props.onSelect(text, viewMoment, view, event)
        }
    }

})

DatePicker.views = Views

module.exports = DatePicker
