"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useGetModifiers = useGetModifiers;
const UI_js_1 = require("./UI.js");
const dateMatchModifiers_js_1 = require("./utils/dateMatchModifiers.js");
/**
 * Return a function to get the modifiers for a given day.
 *
 * @private
 */
function useGetModifiers(days, props, dateLib) {
    const { disabled, hidden, modifiers, showOutsideDays, today } = props;
    const { isSameDay, isSameMonth, Date } = dateLib;
    const internal = {
        [UI_js_1.DayFlag.focused]: [],
        [UI_js_1.DayFlag.outside]: [],
        [UI_js_1.DayFlag.disabled]: [],
        [UI_js_1.DayFlag.hidden]: [],
        [UI_js_1.DayFlag.today]: []
    };
    const custom = {};
    const selection = {
        [UI_js_1.SelectionState.range_end]: [],
        [UI_js_1.SelectionState.range_middle]: [],
        [UI_js_1.SelectionState.range_start]: [],
        [UI_js_1.SelectionState.selected]: []
    };
    for (const day of days) {
        const { date, displayMonth } = day;
        const isOutside = Boolean(displayMonth && !isSameMonth(date, displayMonth));
        const isDisabled = Boolean(disabled && (0, dateMatchModifiers_js_1.dateMatchModifiers)(date, disabled, dateLib));
        const isHidden = Boolean(hidden && (0, dateMatchModifiers_js_1.dateMatchModifiers)(date, hidden, dateLib)) ||
            (!showOutsideDays && isOutside);
        const isToday = isSameDay(date, today ?? new Date());
        if (isOutside)
            internal.outside.push(day);
        if (isDisabled)
            internal.disabled.push(day);
        if (isHidden)
            internal.hidden.push(day);
        if (isToday)
            internal.today.push(day);
        // Add custom modifiers
        if (modifiers) {
            Object.keys(modifiers).forEach((name) => {
                const modifierValue = modifiers?.[name];
                const isMatch = modifierValue
                    ? (0, dateMatchModifiers_js_1.dateMatchModifiers)(date, modifierValue, dateLib)
                    : false;
                if (!isMatch)
                    return;
                if (custom[name]) {
                    custom[name].push(day);
                }
                else {
                    custom[name] = [day];
                }
            });
        }
    }
    return (day) => {
        // Initialize all the modifiers to false
        const dayFlags = {
            [UI_js_1.DayFlag.focused]: false,
            [UI_js_1.DayFlag.disabled]: false,
            [UI_js_1.DayFlag.hidden]: false,
            [UI_js_1.DayFlag.outside]: false,
            [UI_js_1.DayFlag.today]: false
        };
        const selectionStates = {
            [UI_js_1.SelectionState.range_end]: false,
            [UI_js_1.SelectionState.range_middle]: false,
            [UI_js_1.SelectionState.range_start]: false,
            [UI_js_1.SelectionState.selected]: false
        };
        const customModifiers = {};
        // Find the modifiers for the given day
        for (const name in internal) {
            const days = internal[name];
            dayFlags[name] = days.some((d) => d === day);
        }
        for (const name in selection) {
            const days = selection[name];
            selectionStates[name] = days.some((d) => d === day);
        }
        for (const name in custom) {
            customModifiers[name] = custom[name].some((d) => d === day);
        }
        return {
            ...selectionStates,
            ...dayFlags,
            // custom modifiers should override all the previous ones
            ...customModifiers
        };
    };
}
//# sourceMappingURL=useGetModifiers.js.map