"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.useAnimation = useAnimation;
const react_1 = require("react");
const UI_js_1 = require("./UI.js");
/** @private */
function useAnimation(rootElRef, enabled, { classNames, months, focused, dateLib }) {
    const previousRootElSnapshotRef = (0, react_1.useRef)(null);
    const previousMonthsRef = (0, react_1.useRef)(months);
    const animatingRef = (0, react_1.useRef)(false);
    (0, react_1.useLayoutEffect)(() => {
        // get previous months before updating the previous months ref
        const previousMonths = previousMonthsRef.current;
        // update previous months ref for next effect trigger
        previousMonthsRef.current = months;
        if (!enabled ||
            !rootElRef.current ||
            // safety check because the ref can be set to anything by consumers
            !(rootElRef.current instanceof HTMLElement)) {
            return;
        }
        // get previous root element snapshot before updating the snapshot ref
        const previousRootElSnapshot = previousRootElSnapshotRef.current;
        // update snapshot for next effect trigger
        const rootElSnapshot = rootElRef.current.cloneNode(true);
        if (rootElSnapshot instanceof HTMLElement) {
            // if this effect is triggered while animating, we need to remove the old month snapshots from the new root snapshot
            const currentMonthElsSnapshot = [
                ...(rootElSnapshot.querySelectorAll(`[data-animated-month]`) ?? [])
            ];
            currentMonthElsSnapshot.forEach((currentMonthElSnapshot) => {
                const previousMonthElSnapshot = currentMonthElSnapshot.querySelector(`[data-animated-month]`);
                if (previousMonthElSnapshot &&
                    currentMonthElSnapshot.contains(previousMonthElSnapshot)) {
                    currentMonthElSnapshot.removeChild(previousMonthElSnapshot);
                }
            });
            previousRootElSnapshotRef.current = rootElSnapshot;
        }
        else {
            previousRootElSnapshotRef.current = null;
        }
        // validation required for the animation to work as expected
        if (months.length === 0 ||
            previousMonths.length === 0 ||
            months.length !== previousMonths.length ||
            // skip animation if a day is focused because it can cause issues to the animation and is better for a11y
            focused ||
            animatingRef.current) {
            return;
        }
        const isSameMonth = dateLib.isSameMonth(months[0].date, previousMonths[0].date);
        if (isSameMonth) {
            return;
        }
        const previousMonthEls = [
            ...(previousRootElSnapshot?.querySelectorAll(`[data-animated-month]`) ??
                [])
        ];
        const currentMonthEls = [
            ...(rootElRef.current.querySelectorAll(`[data-animated-month]`) ?? [])
        ];
        if (currentMonthEls &&
            currentMonthEls.every((el) => el instanceof HTMLElement) &&
            previousMonthEls &&
            previousMonthEls.every((el) => el instanceof HTMLElement)) {
            animatingRef.current = true;
            const cleanUpFunctions = [];
            const isAfterPreviousMonth = dateLib.isAfter(months[0].date, previousMonths[0].date);
            currentMonthEls.forEach((currentMonthEl, index) => {
                const previousMonthEl = previousMonthEls[index];
                if (!previousMonthEl) {
                    return;
                }
                // animate new displayed month
                const captionAnimationClass = isAfterPreviousMonth
                    ? classNames[UI_js_1.Animation.caption_after_enter]
                    : classNames[UI_js_1.Animation.caption_before_enter];
                const weeksAnimationClass = isAfterPreviousMonth
                    ? classNames[UI_js_1.Animation.weeks_after_enter]
                    : classNames[UI_js_1.Animation.weeks_before_enter];
                currentMonthEl.style.position = "relative";
                currentMonthEl.style.overflow = "hidden";
                const captionEl = currentMonthEl.querySelector(`[data-animated-caption]`);
                if (captionEl && captionEl instanceof HTMLElement) {
                    captionEl.classList.add(captionAnimationClass);
                }
                const weeksEl = currentMonthEl.querySelector(`[data-animated-weeks]`);
                if (weeksEl && weeksEl instanceof HTMLElement) {
                    weeksEl.classList.add(weeksAnimationClass);
                }
                // animate new displayed month end
                const cleanUp = () => {
                    animatingRef.current = false;
                    if (captionEl && captionEl instanceof HTMLElement) {
                        captionEl.classList.remove(captionAnimationClass);
                    }
                    if (weeksEl && weeksEl instanceof HTMLElement) {
                        weeksEl.classList.remove(weeksAnimationClass);
                    }
                    currentMonthEl.style.position = "";
                    currentMonthEl.style.overflow = "";
                    if (currentMonthEl.contains(previousMonthEl)) {
                        currentMonthEl.removeChild(previousMonthEl);
                    }
                };
                cleanUpFunctions.push(cleanUp);
                // animate old displayed month
                previousMonthEl.style.pointerEvents = "none";
                previousMonthEl.style.position = "absolute";
                previousMonthEl.style.overflow = "hidden";
                previousMonthEl.setAttribute("aria-hidden", "true");
                // hide the weekdays container of the old month and only the new one
                const previousWeekdaysEl = previousMonthEl.querySelector(`[data-animated-weekdays]`);
                if (previousWeekdaysEl && previousWeekdaysEl instanceof HTMLElement) {
                    previousWeekdaysEl.style.opacity = "0";
                }
                const previousCaptionEl = previousMonthEl.querySelector(`[data-animated-caption]`);
                if (previousCaptionEl && previousCaptionEl instanceof HTMLElement) {
                    previousCaptionEl.classList.add(isAfterPreviousMonth
                        ? classNames[UI_js_1.Animation.caption_before_exit]
                        : classNames[UI_js_1.Animation.caption_after_exit]);
                    previousCaptionEl.addEventListener("animationend", cleanUp);
                }
                const previousWeeksEl = previousMonthEl.querySelector(`[data-animated-weeks]`);
                if (previousWeeksEl && previousWeeksEl instanceof HTMLElement) {
                    previousWeeksEl.classList.add(isAfterPreviousMonth
                        ? classNames[UI_js_1.Animation.weeks_before_exit]
                        : classNames[UI_js_1.Animation.weeks_after_exit]);
                }
                currentMonthEl.insertBefore(previousMonthEl, currentMonthEl.firstChild);
            });
        }
    });
}
//# sourceMappingURL=useAnimation.js.map