'use strict';

var _expect = require('expect.js');

var _expect2 = _interopRequireDefault(_expect);

var _react = require('react');

var _react2 = _interopRequireDefault(_react);

var _jss = require('jss');

var _getDisplayName = require('./getDisplayName');

var _getDisplayName2 = _interopRequireDefault(_getDisplayName);

var _helper = require('../tests/helper');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { 'default': obj }; }

describe('injectSheet', function () {
  var jss = void 0;

  beforeEach(function () {
    jss = (0, _jss.create)({ createGenerateClassName: _helper.createGenerateClassName });
  });

  describe('.injectSheet()', function () {
    var MyComponent = void 0;

    beforeEach(function () {
      MyComponent = injectSheet({
        button: { color: 'red' }
      })();
    });

    it('should attach and detach a sheet', function () {
      render(_react2['default'].createElement(MyComponent, null), node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(1);
      unmountComponentAtNode(node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(0);
    });

    it('should reuse one sheet for many elements and detach sheet', function () {
      render(_react2['default'].createElement(
        'div',
        null,
        _react2['default'].createElement(MyComponent, null),
        _react2['default'].createElement(MyComponent, null),
        _react2['default'].createElement(MyComponent, null)
      ), node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(1);
      unmountComponentAtNode(node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(0);
    });

    it('should reuse one sheet for many elements wrapped into a JssProvider', function () {
      render(_react2['default'].createElement(
        'div',
        null,
        _react2['default'].createElement(
          JssProvider,
          null,
          _react2['default'].createElement(MyComponent, null)
        ),
        _react2['default'].createElement(
          JssProvider,
          null,
          _react2['default'].createElement(MyComponent, null)
        )
      ), node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(1);
      unmountComponentAtNode(node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(0);
    });

    it('should have correct meta attribute', function () {
      render(_react2['default'].createElement(MyComponent, null), node);
      var meta = document.querySelector('style').getAttribute('data-meta');
      (0, _expect2['default'])(meta).to.be('NoRenderer, Unthemed, Static');
    });
  });

  describe('injectSheet() option "inject"', function () {
    var getInjected = function getInjected(options) {
      var injectedProps = void 0;
      var Renderer = function Renderer(props) {
        injectedProps = props;
        return null;
      };
      var MyComponent = injectSheet(function () {
        return {
          button: { color: 'red' }
        };
      }, options)(Renderer);
      render(_react2['default'].createElement(
        ThemeProvider,
        { theme: {} },
        _react2['default'].createElement(MyComponent, null)
      ), node);
      return Object.keys(injectedProps);
    };

    it('should inject all by default', function () {
      (0, _expect2['default'])(getInjected()).to.eql(['theme', 'classes']);
    });

    it('should inject sheet only', function () {
      (0, _expect2['default'])(getInjected({ inject: ['sheet'] })).to.eql(['sheet']);
    });

    it('should inject classes only', function () {
      (0, _expect2['default'])(getInjected({ inject: ['classes'] })).to.eql(['classes']);
    });

    it('should inject theme only', function () {
      (0, _expect2['default'])(getInjected({ inject: ['theme'] })).to.eql(['theme']);
    });

    it('should inject classes and theme', function () {
      (0, _expect2['default'])(getInjected({ inject: ['classes', 'theme'] })).to.eql(['theme', 'classes']);
    });
  });

  describe('.injectSheet() preserving source order', function () {
    var ComponentA = void 0;
    var ComponentB = void 0;
    var ComponentC = void 0;

    beforeEach(function () {
      ComponentA = injectSheet({
        button: { color: 'red' }
      })();
      ComponentB = injectSheet({
        button: { color: 'blue' }
      })();
      ComponentC = injectSheet({
        button: { color: 'green' }
      }, { index: 1234 })();
    });

    it('should provide a default index in ascending order', function () {
      render(_react2['default'].createElement(ComponentA, null), node);
      (0, _expect2['default'])(sheets.registry.length).to.equal(1);
      var indexA = sheets.registry[0].options.index;
      sheets.reset();
      render(_react2['default'].createElement(ComponentB, null), node);
      (0, _expect2['default'])(sheets.registry.length).to.equal(1);
      var indexB = sheets.registry[0].options.index;

      (0, _expect2['default'])(indexA).to.be.lessThan(0);
      (0, _expect2['default'])(indexB).to.be.lessThan(0);
      (0, _expect2['default'])(indexA).to.be.lessThan(indexB);
    });

    it('should not be affected by rendering order', function () {
      render(_react2['default'].createElement(ComponentB, null), node);
      (0, _expect2['default'])(sheets.registry.length).to.equal(1);
      var indexB = sheets.registry[0].options.index;
      sheets.reset();
      render(_react2['default'].createElement(ComponentA, null), node);
      (0, _expect2['default'])(sheets.registry.length).to.equal(1);
      var indexA = sheets.registry[0].options.index;

      (0, _expect2['default'])(indexA).to.be.lessThan(0);
      (0, _expect2['default'])(indexB).to.be.lessThan(0);
      (0, _expect2['default'])(indexA).to.be.lessThan(indexB);
    });

    it('should keep custom index', function () {
      render(_react2['default'].createElement(ComponentC, null), node);
      (0, _expect2['default'])(sheets.registry.length).to.equal(1);
      var indexC = sheets.registry[0].options.index;
      (0, _expect2['default'])(indexC).to.equal(1234);
    });
  });

  describe('.injectSheet() without a component for global styles', function () {
    var MyComponent = void 0;

    beforeEach(function () {
      MyComponent = injectSheet({
        button: { color: 'red' }
      })();
    });

    it('should attach and detach a sheet', function () {
      render(_react2['default'].createElement(MyComponent, null), node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(1);
      unmountComponentAtNode(node);
      (0, _expect2['default'])(document.querySelectorAll('style').length).to.be(0);
    });

    it('should render children', function () {
      var isRendered = true;
      var ChildComponent = function ChildComponent() {
        isRendered = true;
        return null;
      };
      render(_react2['default'].createElement(
        MyComponent,
        null,
        _react2['default'].createElement(ChildComponent, null)
      ), node);
      unmountComponentAtNode(node);
      (0, _expect2['default'])(isRendered).to.be(true);
    });
  });

  describe('access inner component', function () {
    it('should be exposed using "InnerComponent" property', function () {
      var ComponentOuter = injectSheet({
        button: { color: 'red' }
      })();
      (0, _expect2['default'])(ComponentOuter.InnerComponent).to.be.a(Function);
    });
  });

  describe('override sheet prop', function () {
    var MyComponent = void 0;
    var receivedSheet = void 0;
    var mock = {};

    beforeEach(function () {
      var InnerComponent = function InnerComponent(props) {
        receivedSheet = props.sheet;
        return null;
      };
      MyComponent = injectSheet()(InnerComponent);
    });

    it('should be able to override the sheet prop', function () {
      var Parent = function Parent() {
        return _react2['default'].createElement(MyComponent, { sheet: mock });
      };
      render(_react2['default'].createElement(Parent, null), node);
      (0, _expect2['default'])(receivedSheet).to.be(mock);
    });
  });

  describe('classes prop', function () {
    it('should be prefixed by the parent component name', function () {
      var passedClasses = void 0;
      var InnerComponent = function InnerComponent(_ref) {
        var classes = _ref.classes;

        passedClasses = classes;
        return null;
      };
      var MyComponent = injectSheet({
        button: { color: 'red' }
      })(InnerComponent);
      render(_react2['default'].createElement(MyComponent, null), node);
      Object.keys(passedClasses).forEach(function (ruleName) {
        (0, _expect2['default'])(passedClasses[ruleName]).to.match(new RegExp('^' + (0, _getDisplayName2['default'])(InnerComponent) + '-' + ruleName + '[\\s\\S]*$'));
      });
    });

    it('should use defaultProps.classes from InnerComponent', function () {
      var classes = void 0;
      var InnerComponent = function InnerComponent(props) {
        classes = props.classes;
        return null;
      };
      InnerComponent.defaultProps = {
        classes: { 'default': 'default' }
      };
      var MyComponent = injectSheet({}, { jss: jss })(InnerComponent);
      render(_react2['default'].createElement(MyComponent, null), node);
      (0, _expect2['default'])(classes).to.eql({ 'default': 'default' });
    });

    it('should merge the defaultProps.classes from InnerComponent', function () {
      var classes = void 0;
      var InnerComponent = function InnerComponent(props) {
        classes = props.classes;
        return null;
      };
      InnerComponent.defaultProps = {
        classes: { 'default': 'default' }
      };
      var MyComponent = injectSheet({
        a: { color: 'red' }
      }, { jss: jss })(InnerComponent);
      render(_react2['default'].createElement(MyComponent, null), node);
      (0, _expect2['default'])(classes).to.eql({ 'default': 'default', a: 'a-id' });
    });

    it('should merge users classes', function () {
      var classes = void 0;
      var InnerComponent = function InnerComponent(props) {
        classes = props.classes;
        return null;
      };
      InnerComponent.defaultProps = {
        classes: { 'default': 'default' }
      };
      var MyComponent = injectSheet({
        a: { color: 'red' }
      }, { jss: jss })(InnerComponent);
      render(_react2['default'].createElement(MyComponent, { classes: { user: 'user' } }), node);
      (0, _expect2['default'])(classes).to.eql({ 'default': 'default', a: 'a-id', user: 'user' });
    });
  });
}); /* eslint-disable global-require, react/prop-types */