"use strict";

var _require = require('../utils'),
  getDiIdentifier = _require.getDiIdentifier,
  getDiStatements = _require.getDiStatements;

// use simple numeric comparison to have uppercase (components) first
var compareName = function compareName(a, b) {
  return a.name > b.name ? 1 : -1;
};
module.exports = {
  meta: {
    type: 'suggestion',
    docs: {
      description: 'Require injectable dependencies to be sorted',
      category: 'Stylistic Issues',
      recommended: false
    },
    fixable: 'code',
    schema: [],
    messages: {
      unsortedInjectable: 'Expected dependencies to be ordered by name. ' + "'{{name}}' should be before '{{prevName}}'."
    }
  },
  create: function create(context) {
    var diIdentifier;
    var report = function report(node, prevNode, args, sortedArgs) {
      return context.report({
        node: node,
        messageId: 'unsortedInjectable',
        data: {
          name: node.name,
          prevName: prevNode.name
        },
        fix: function fix(fixer) {
          // grab whatever between 1st arg end / 2nd arg start as separator
          var separator = context.getSourceCode().text.slice(args[0].range[1], args[1].range[0]);
          var start = args[0].range[0];
          var end = args[args.length - 1].range[1];
          var sorted = sortedArgs.map(function (n) {
            return n.name;
          }).join(separator);
          // fixes all order issues at once
          // so avoids the need of multiple saves
          return fixer.replaceTextRange([start, end], sorted);
        }
      });
    };
    return {
      ImportDeclaration: function ImportDeclaration(node) {
        if (!diIdentifier) diIdentifier = getDiIdentifier(node);
      },
      BlockStatement: function BlockStatement(node) {
        if (!diIdentifier) return;
        var diStatements = getDiStatements(node, diIdentifier);
        // ignore locations where di was not explicitly set
        if (!diStatements.length) return;
        diStatements.forEach(function (statement) {
          var args = statement.expression.arguments;
          // sort uppercase first, lowercase after
          // so we get components and hooks grouped
          var sortedArgs = args.slice().sort(compareName);
          args.forEach(function (arg, i) {
            var prevArg = args[i - 1];
            if (prevArg && compareName(arg, prevArg) < 0) {
              report(arg, prevArg, args, sortedArgs);
            }
          });
        });
      }
    };
  }
};