/**
 * Copyright (c) 2015-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */

package com.facebook.react.uimanager;

import java.util.Arrays;
import java.util.HashSet;

import com.facebook.csslayout.Spacing;
import com.facebook.react.common.SetBuilder;

/**
 * Keys for props that need to be shared across multiple classes.
 */
public class ViewProps {

  public static final String VIEW_CLASS_NAME = "RCTView";

  // Layout only (only affect positions of children, causes no drawing)
  // !!! Keep in sync with LAYOUT_ONLY_PROPS below
  public static final String ALIGN_ITEMS = "alignItems";
  public static final String ALIGN_SELF = "alignSelf";
  public static final String BOTTOM = "bottom";
  public static final String COLLAPSABLE = "collapsable";
  public static final String FLEX = "flex";
  public static final String FLEX_DIRECTION = "flexDirection";
  public static final String FLEX_WRAP = "flexWrap";
  public static final String HEIGHT = "height";
  public static final String JUSTIFY_CONTENT = "justifyContent";
  public static final String LEFT = "left";
  public static final String[] MARGINS = {
      "margin", "marginVertical", "marginHorizontal", "marginLeft", "marginRight", "marginTop",
      "marginBottom"
  };
  public static final String[] PADDINGS = {
      "padding", "paddingVertical", "paddingHorizontal", "paddingLeft", "paddingRight",
      "paddingTop", "paddingBottom"
  };
  public static final String POSITION = "position";
  public static final String RIGHT = "right";
  public static final String TOP = "top";
  public static final String WIDTH = "width";

  // Props that affect more than just layout
  public static final String ENABLED = "enabled";
  public static final String BACKGROUND_COLOR = "backgroundColor";
  public static final String COLOR = "color";
  public static final String FONT_SIZE = "fontSize";
  public static final String FONT_WEIGHT = "fontWeight";
  public static final String FONT_STYLE = "fontStyle";
  public static final String FONT_FAMILY = "fontFamily";
  public static final String LINE_HEIGHT = "lineHeight";
  public static final String NEEDS_OFFSCREEN_ALPHA_COMPOSITING = "needsOffscreenAlphaCompositing";
  public static final String NUMBER_OF_LINES = "numberOfLines";
  public static final String ON = "on";
  public static final String RESIZE_MODE = "resizeMode";
  public static final String TEXT_ALIGN = "textAlign";
  public static final String BORDER_WIDTH = "borderWidth";
  public static final String BORDER_LEFT_WIDTH = "borderLeftWidth";
  public static final String BORDER_TOP_WIDTH = "borderTopWidth";
  public static final String BORDER_RIGHT_WIDTH = "borderRightWidth";
  public static final String BORDER_BOTTOM_WIDTH = "borderBottomWidth";
  public static final int[] BORDER_SPACING_TYPES = {
      Spacing.ALL, Spacing.LEFT, Spacing.RIGHT, Spacing.TOP, Spacing.BOTTOM
  };
  public static final String[] BORDER_WIDTHS = {
      BORDER_WIDTH, BORDER_LEFT_WIDTH, BORDER_RIGHT_WIDTH, BORDER_TOP_WIDTH, BORDER_BOTTOM_WIDTH,
  };
  public static final int[] PADDING_MARGIN_SPACING_TYPES = {
      Spacing.ALL, Spacing.VERTICAL, Spacing.HORIZONTAL, Spacing.LEFT, Spacing.RIGHT, Spacing.TOP,
      Spacing.BOTTOM
  };

  private static final HashSet<String> LAYOUT_ONLY_PROPS = createLayoutOnlyPropsMap();

  private static HashSet<String> createLayoutOnlyPropsMap() {
    HashSet<String> layoutOnlyProps = SetBuilder.newHashSet();
    layoutOnlyProps.addAll(
        Arrays.asList(
            ALIGN_SELF,
            ALIGN_ITEMS,
            BOTTOM,
            COLLAPSABLE,
            FLEX,
            FLEX_DIRECTION,
            FLEX_WRAP,
            HEIGHT,
            JUSTIFY_CONTENT,
            LEFT,
            POSITION,
            RIGHT,
            TOP,
            WIDTH));
    for (int i = 0; i < MARGINS.length; i++) {
      layoutOnlyProps.add(MARGINS[i]);
    }
    for (int i = 0; i < PADDINGS.length; i++) {
      layoutOnlyProps.add(PADDINGS[i]);
    }
    return layoutOnlyProps;
  }

  public static boolean isLayoutOnly(String prop) {
    return LAYOUT_ONLY_PROPS.contains(prop);
  }
}
