/**
 * Copyright (c) 2016-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */
'use strict';
/*eslint no-console-disallow: "off"*/
/*global React ReactDOM Table stringInterner stackRegistry aggrow preLoadedCapture:true*/

function RefVisitor(refs,id){
this.refs=refs;
this.id=id;
}

RefVisitor.prototype={
moveToEdge:function moveToEdge(name){
var ref=this.refs[this.id];
if(ref&&ref.edges){
var edges=ref.edges;
for(var edgeId in edges){
if(edges[edgeId]===name){
this.id=edgeId;
return this;
}
}
}
this.id=undefined;
return this;
},
moveToFirst:function moveToFirst(callback){
var ref=this.refs[this.id];
if(ref&&ref.edges){
var edges=ref.edges;
for(var edgeId in edges){
this.id=edgeId;
if(callback(edges[edgeId],this)){
return this;
}
}
}
this.id=undefined;
return this;
},
forEachEdge:function forEachEdge(callback){
var ref=this.refs[this.id];
if(ref&&ref.edges){
var edges=ref.edges;
var visitor=new RefVisitor(this.refs,undefined);
for(var edgeId in edges){
visitor.id=edgeId;
callback(edges[edgeId],visitor);
}
}
},
getType:function getType(){
var ref=this.refs[this.id];
if(ref){
return ref.type;
}
return undefined;
},
getRef:function getRef(){
return this.refs[this.id];
},
clone:function clone(){
return new RefVisitor(this.refs,this.id);
},
isDefined:function isDefined(){
return!!this.id;
},
getValue:function getValue(){var _this=this;
var ref=this.refs[this.id];
if(ref){
if(ref.type==='string'){
if(ref.value){
return ref.value;
}else{var _ret=function(){
var rope=[];
_this.forEachEdge(function(name,visitor){
if(name&&name.startsWith('[')&&name.endsWith(']')){
var index=parseInt(name.substring(1,name.length-1),10);
rope[index]=visitor.getValue();
}
});
return{v:rope.join('')};}();if(typeof _ret==="object")return _ret.v;
}
}else if(ref.type==='ScriptExecutable'||
ref.type==='EvalExecutable'||
ref.type==='ProgramExecutable'){
return ref.value.url+':'+ref.value.line+':'+ref.value.col;
}else if(ref.type==='FunctionExecutable'){
return ref.value.name+'@'+ref.value.url+':'+ref.value.line+':'+ref.value.col;
}else if(ref.type==='NativeExecutable'){
return ref.value.function+' '+ref.value.constructor+' '+ref.value.name;
}else if(ref.type==='Function'){
var executable=this.clone().moveToEdge('@Executable');
if(executable.id){
return executable.getRef().type+' '+executable.getValue();
}
}
}
return'#none';
}};


function forEachRef(refs,callback){
var visitor=new RefVisitor(refs,undefined);
for(var id in refs){
visitor.id=id;
callback(visitor);
}
}

function firstRef(refs,callback){
for(var id in refs){
var ref=refs[id];
if(callback(id,ref)){
return new RefVisitor(refs,id);
}
}
return new RefVisitor(refs,undefined);
}

function getInternalInstanceName(visitor){
var type=visitor.clone().moveToEdge('_currentElement').moveToEdge('type');
if(type.getType()==='string'){// element.type is string
return type.getValue();
}else if(type.getType()==='Function'){// element.type is function
var displayName=type.clone().moveToEdge('displayName');
if(displayName.isDefined()){
return displayName.getValue();// element.type.displayName
}
var name=type.clone().moveToEdge('name');
if(name.isDefined()){
return name.getValue();// element.type.name
}
type.moveToEdge('@Executable');
if(type.getType()==='FunctionExecutable'){
return type.getRef().value.name;// element.type symbolicated name
}
}
return'#unknown';
}

function buildReactComponentTree(visitor,registry,strings){
var ref=visitor.getRef();
if(ref.reactTree||ref.reactParent===undefined){
return;// has one or doesn't need one
}
var parentVisitor=ref.reactParent;
if(parentVisitor===null){
ref.reactTree=registry.insert(registry.root,strings.intern(getInternalInstanceName(visitor)));
}else if(parentVisitor){
var parentRef=parentVisitor.getRef();
buildReactComponentTree(parentVisitor,registry,strings);
var relativeName=getInternalInstanceName(visitor);
if(ref.reactKey){
relativeName=ref.reactKey+': '+relativeName;
}
ref.reactTree=registry.insert(parentRef.reactTree,strings.intern(relativeName));
}else{
throw'non react instance parent of react instance';
}
}

function markReactComponentTree(refs,registry,strings){
// annotate all refs that are react internal instances with their parent and name
// ref.reactParent = visitor that points to parent instance,
//   null if we know it's an instance, but don't have a parent yet
// ref.reactKey = if a key is used to distinguish siblings
forEachRef(refs,function(visitor){
var visitorClone=visitor.clone();// visitor will get stomped on next iteration
var ref=visitor.getRef();
visitor.forEachEdge(function(edgeName,edgeVisitor){
var edgeRef=edgeVisitor.getRef();
if(edgeRef){
if(edgeName==='_renderedChildren'){
if(ref.reactParent===undefined){
// ref is react component, even if we don't have a parent yet
ref.reactParent=null;
}
edgeVisitor.forEachEdge(function(childName,childVisitor){
var childRef=childVisitor.getRef();
if(childRef&&childName.startsWith('.')){
childRef.reactParent=visitorClone;
childRef.reactKey=childName;
}
});
}else if(edgeName==='_renderedComponent'){
if(ref.reactParent===undefined){
ref.reactParent=null;
}
edgeRef.reactParent=visitorClone;
}
}
});
});
// build tree of react internal instances (since that's what has the structure)
// fill in ref.reactTree = path registry node
forEachRef(refs,function(visitor){
buildReactComponentTree(visitor,registry,strings);
});
// hook in components by looking at their _reactInternalInstance fields
forEachRef(refs,function(visitor){
var ref=visitor.getRef();
var instanceRef=visitor.moveToEdge('_reactInternalInstance').getRef();
if(instanceRef){
ref.reactTree=instanceRef.reactTree;
}
});
}

function functionUrlFileName(visitor){
var executable=visitor.clone().moveToEdge('@Executable');
var ref=executable.getRef();
if(ref&&ref.value&&ref.value.url){
var url=ref.value.url;
var file=url.substring(url.lastIndexOf('/')+1);
if(file.endsWith('.js')){
file=file.substring(0,file.length-3);
}
return file;
}
return undefined;
}

function markModules(refs){
var modules=firstRef(refs,function(id,ref){return ref.type==='CallbackGlobalObject';});
modules.moveToEdge('require');
modules.moveToFirst(function(name,visitor){return visitor.getType()==='JSActivation';});
modules.moveToEdge('modules');
modules.forEachEdge(function(name,visitor){
var ref=visitor.getRef();
visitor.moveToEdge('exports');
if(visitor.getType()==='Object'){
visitor.moveToFirst(function(memberName,member){return member.getType()==='Function';});
if(visitor.isDefined()){
ref.module=functionUrlFileName(visitor);
}
}else if(visitor.getType()==='Function'){
var displayName=visitor.clone().moveToEdge('displayName');
if(displayName.isDefined()){
ref.module=displayName.getValue();
}
ref.module=functionUrlFileName(visitor);
}
if(ref&&!ref.module){
ref.module='#unknown '+name;
}
});
}

function registerPathToRoot(refs,registry,strings){
markReactComponentTree(refs,registry,strings);
markModules(refs);
var breadth=[];
forEachRef(refs,function(visitor){
var ref=visitor.getRef();
if(ref.type==='CallbackGlobalObject'){
ref.rootPath=registry.insert(registry.root,strings.intern(ref.type));
breadth.push(visitor.clone());
}
});var _loop=function _loop(){

var nextBreadth=[];var _loop2=function _loop2(
i){
var visitor=breadth[i];
var ref=visitor.getRef();
visitor.forEachEdge(function(edgeName,edgeVisitor){
var edgeRef=edgeVisitor.getRef();
if(edgeRef&&edgeRef.rootPath===undefined){
var pathName=edgeRef.type;
if(edgeName){
pathName=edgeName+': '+pathName;
}
edgeRef.rootPath=registry.insert(ref.rootPath,strings.intern(pathName));
nextBreadth.push(edgeVisitor.clone());
// copy module and react tree forward
if(edgeRef.module===undefined){
edgeRef.module=ref.module;
}
if(edgeRef.reactTree===undefined){
edgeRef.reactTree=ref.reactTree;
}
}
});};for(var i=0;i<breadth.length;i++){_loop2(i);
}
breadth=nextBreadth;};while(breadth.length>0){_loop();
}
}

function captureRegistry(){
var strings=stringInterner();
var stacks=stackRegistry(strings);
var data=new Int32Array(0);

var idField=0;
var typeField=1;
var sizeField=2;
var traceField=3;
var pathField=4;
var reactField=5;
var valueField=6;
var moduleField=7;
var numFields=8;

return{
strings:strings,
stacks:stacks,
data:data,
register:function registerCapture(captureId,capture){var _this2=this;
// NB: capture.refs is potentially VERY large, so we try to avoid making
// copies, even of iteration is a bit more annoying.
var rowCount=0;
for(var id in capture.refs){// eslint-disable-line no-unused-vars
rowCount++;
}
for(var _id in capture.markedBlocks){// eslint-disable-line no-unused-vars
rowCount++;
}
console.log(
'increasing row data from '+(this.data.length*4).toString()+'B to '+
(this.data.length*4+rowCount*numFields*4).toString()+'B');

var newData=new Int32Array(this.data.length+rowCount*numFields);
newData.set(data);
var dataOffset=this.data.length;
this.data=null;

registerPathToRoot(capture.refs,this.stacks,this.strings);
var internedCaptureId=this.strings.intern(captureId);
var noneString=this.strings.intern('#none');
var noneStack=this.stacks.insert(this.stacks.root,noneString);
forEachRef(capture.refs,function(visitor){
var ref=visitor.getRef();
var id=visitor.id;
newData[dataOffset+idField]=parseInt(id,16);
newData[dataOffset+typeField]=_this2.strings.intern(ref.type);
newData[dataOffset+sizeField]=ref.size;
newData[dataOffset+traceField]=internedCaptureId;
if(ref.rootPath===undefined){
newData[dataOffset+pathField]=noneStack.id;
}else{
newData[dataOffset+pathField]=ref.rootPath.id;
}
if(ref.reactTree===undefined){
newData[dataOffset+reactField]=noneStack.id;
}else{
newData[dataOffset+reactField]=ref.reactTree.id;
}
newData[dataOffset+valueField]=_this2.strings.intern(visitor.getValue());
if(ref.module){
newData[dataOffset+moduleField]=_this2.strings.intern(ref.module);
}else{
newData[dataOffset+moduleField]=noneString;
}
dataOffset+=numFields;
});
for(var _id2 in capture.markedBlocks){
var block=capture.markedBlocks[_id2];
newData[dataOffset+idField]=parseInt(_id2,16);
newData[dataOffset+typeField]=this.strings.intern('Marked Block Overhead');
newData[dataOffset+sizeField]=block.capacity-block.size;
newData[dataOffset+traceField]=internedCaptureId;
newData[dataOffset+pathField]=noneStack.id;
newData[dataOffset+reactField]=noneStack.id;
newData[dataOffset+valueField]=this.strings.intern(
'capacity: '+block.capacity+
', size: '+block.size+
', granularity: '+block.cellSize);

newData[dataOffset+moduleField]=noneString;
dataOffset+=numFields;
}
this.data=newData;
},
getAggrow:function getAggrow(){
var agStrings=this.strings;
var agStacks=this.stacks.flatten();
var agData=this.data;
var agNumRows=agData.length/numFields;
var ag=new aggrow(agNumRows);

ag.addFieldExpander('Id',
function getId(row){
var id=agData[row*numFields+idField];
if(id<0){
id+=0x100000000;// data is int32, id is uint32
}
return'0x'+id.toString(16);
},
function compareAddress(rowA,rowB){
return agData[rowA*numFields+idField]-agData[rowB*numFields+idField];
});

var typeExpander=ag.addFieldExpander('Type',
function getType(row){return agStrings.get(agData[row*numFields+typeField]);},
function compareType(rowA,rowB){
return agData[rowA*numFields+typeField]-agData[rowB*numFields+typeField];
});

ag.addFieldExpander('Size',
function getSize(row){return agData[row*numFields+sizeField].toString();},
function compareSize(rowA,rowB){
return agData[rowA*numFields+sizeField]-agData[rowB*numFields+sizeField];
});

ag.addFieldExpander('Trace',
function getSize(row){return agStrings.get(agData[row*numFields+traceField]);},
function compareSize(rowA,rowB){
return agData[rowA*numFields+traceField]-agData[rowB*numFields+traceField];
});

var pathExpander=ag.addCalleeStackExpander(
'Path',
agStacks.maxDepth,
function getStack(row){return agStacks.get(agData[row*numFields+pathField]);},
function getFrame(id){return agStrings.get(id);});


var reactExpander=ag.addCalleeStackExpander(
'React Tree',
agStacks.maxDepth,
function getStack(row){return agStacks.get(agData[row*numFields+reactField]);},
function getFrame(id){return agStrings.get(id);});


var valueExpander=ag.addFieldExpander('Value',
function getValue(row){return agStrings.get(agData[row*numFields+valueField]);},
function compareValue(rowA,rowB){
return agData[rowA*numFields+valueField]-agData[rowB*numFields+valueField];
});

var moduleExpander=ag.addFieldExpander('Module',
function getModule(row){return agStrings.get(agData[row*numFields+moduleField]);},
function compareModule(rowA,rowB){
return agData[rowA*numFields+moduleField]-agData[rowB*numFields+moduleField];
});

var sizeAggregator=ag.addAggregator('Size',
function aggregateSize(indices){
var size=0;
for(var i=0;i<indices.length;i++){
var row=indices[i];
size+=agData[row*numFields+sizeField];
}
return size;
},
function formatSize(value){return value.toString();},
function sortSize(a,b){return b-a;});

var countAggregator=ag.addAggregator('Count',
function aggregateCount(indices){
return indices.length;
},
function formatCount(value){return value.toString();},
function sortCount(a,b){return b-a;});

ag.setActiveExpanders([
pathExpander,
reactExpander,
moduleExpander,
typeExpander,
valueExpander]);

ag.setActiveAggregators([sizeAggregator,countAggregator]);
return ag;
}};

}

if(preLoadedCapture){
var r=new captureRegistry();
r.register('trace',preLoadedCapture);
preLoadedCapture=undefined;// let GG clean up the capture
ReactDOM.render(React.createElement(Table,{aggrow:r.getAggrow()}),document.body);
}
//# sourceMappingURL=data:application/json;base64,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
// @generated
