import { assert } from 'chai';
import getForm, { clearGetFormCache } from '../src/utils/get-form';
import { combineReducers } from 'redux';
import { modelReducer, combineForms, createForms, actions } from '../src';

describe('combineForms()', () => {
  beforeEach(() => clearGetFormCache());

  context('standard combined reducer', () => {
    const reducer = combineForms({
      foo: modelReducer('foo', { one: 'two' }),
      bar: modelReducer('bar', { three: 'four' }),
    });

    it('exists as a function', () => {
      assert.isFunction(combineForms);
    });

    it('should return a reducer function', () => {
      assert.isFunction(reducer);
    });

    describe('initial state', () => {
      const initialState = reducer(undefined, { type: null });

      it('should contain the initial state of each reducer', () => {
        assert.containSubset(initialState, {
          foo: { one: 'two' },
          bar: { three: 'four' },
        });
      });
    });

    describe('usage with getForm()', () => {
      const state = reducer(undefined, { type: null });

      it('should be able to retrieve the proper form', () => {
        const fooForm = getForm(state, 'foo');

        assert.equal(fooForm, state.forms.foo);

        const barForm = getForm(state, 'bar');

        assert.equal(barForm, state.forms.bar);
      });
    });
  });

  describe('implicit model reducer creation with initial state', () => {
    const implicitReducer = combineForms({
      foo: { one: 'two' },
      bar: { three: 'four' },
    });

    it('should respond to change actions', () => {
      let state = implicitReducer(undefined, actions.change('foo.one', 'changed'));

      assert.equal(state.foo.one, 'changed');

      state = implicitReducer(state, actions.change('bar.three', 'changed again'));

      assert.equal(state.bar.three, 'changed again');
    });
  });

  describe('setting the "key" option', () => {
    const customKeyReducer = combineForms({
      foo: { bar: 'baz' },
    }, '', { key: 'myForms' });

    const initialState = customKeyReducer(undefined, { type: null });

    it('should have the form reducer state under the custom forms key', () => {
      assert.equal(initialState.myForms.$form.model, '');
    });

    it('should be retrievable with getForm()', () => {
      const fooForm = getForm(initialState, 'foo');

      assert.equal(fooForm, initialState.myForms.foo);
    });
  });

  describe('deep forms', () => {
    const reducer = combineReducers({
      deep: combineForms({
        foo: { bar: 'baz' },
      }, 'deep'),
    });

    it('should be able to find the deep form', () => {
      const state = reducer(undefined, { type: null });

      assert.ok(getForm(state, 'deep.foo'));
    });
  });
});

describe('createForms()', () => {
  beforeEach(() => clearGetFormCache());

  it('exists as a function', () => {
    assert.isFunction(createForms);
  });

  context('standard mapped reducers', () => {
    const forms = createForms({
      foo: modelReducer('foo', { one: 'two' }),
      bar: modelReducer('bar', { three: 'four' }),
    });

    it('should create a map of object keys to reducers', () => {
      assert.isFunction(forms.foo);
      assert.isFunction(forms.bar);
    });

    it('should create the form reducer at .forms', () => {
      assert.isFunction(forms.forms);
    });
  });

  context('reducers mapped from initial state', () => {
    const forms = createForms({
      foo: { one: 'two' },
      bar: { three: 'four' },
    });

    it('should create a map of object keys to reducers', () => {
      assert.isFunction(forms.foo);
      assert.isFunction(forms.bar);
    });

    it('should create the form reducer at .forms', () => {
      assert.isFunction(forms.forms);
    });
  });

  context('setting the "key" option', () => {
    const forms = createForms({
      foo: { one: 'two' },
      bar: { three: 'four' },
    }, '', { key: 'myForms' });

    it('should create the form reducer at .myForms', () => {
      assert.isFunction(forms.myForms);
    });
  });

  context('deep forms', () => {
    const reducers = {
      deep: createForms({
        foo: { one: 'two' },
        bar: { three: 'four' },
      }, 'deep'),
    };

    it('should create a map of object keys to reducers', () => {
      assert.isFunction(reducers.deep.foo);
      assert.isFunction(reducers.deep.bar);
    });

    it('should create the form reducer at .forms', () => {
      assert.isFunction(reducers.deep.forms);
    });
  });
});
