/**
 * Copyright 2013-2015, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule printRelayQuery
 * @typechecks
 * 
 */

'use strict';

var _Object$keys = require('babel-runtime/core-js/object/keys')['default'];

var RelayProfiler = require('./RelayProfiler');
var RelayQuery = require('./RelayQuery');

var invariant = require('fbjs/lib/invariant');

/**
 * @internal
 *
 * `printRelayQuery(query)` returns a string representation of the query. The
 * supplied `node` must be flattened (and not contain fragments).
 */
function printRelayQuery(node) {
  var fragmentMap = {};
  var queryText = null;
  if (node instanceof RelayQuery.Root) {
    queryText = printRoot(node, fragmentMap);
  } else if (node instanceof RelayQuery.Fragment) {
    queryText = printFragment(node, fragmentMap);
  } else if (node instanceof RelayQuery.Field) {
    queryText = printField(node, fragmentMap);
  } else if (node instanceof RelayQuery.Mutation) {
    queryText = printMutation(node, fragmentMap);
  }
  !queryText ? process.env.NODE_ENV !== 'production' ? invariant(false, 'printRelayQuery(): Unsupported node type.') : invariant(false) : undefined;
  // Reassign to preserve Flow type refinement within closure.
  var query = queryText;
  _Object$keys(fragmentMap).forEach(function (fragmentID) {
    var fragmentText = fragmentMap[fragmentID];
    if (fragmentText) {
      query = query + ' ' + fragmentText;
    }
  });
  return query;
}

function printRoot(node, fragmentMap) {
  !!node.getBatchCall() ? process.env.NODE_ENV !== 'production' ? invariant(false, 'printRelayQuery(): Deferred queries are not supported.') : invariant(false) : undefined;

  var rootCall = node.getRootCall();
  var rootArgumentName = node.getRootCallArgument();
  var rootFieldString = rootCall.name;
  if (rootCall.value != null) {
    !rootArgumentName ? process.env.NODE_ENV !== 'production' ? invariant(false, 'printRelayQuery(): Expected an argument name for root field `%s`.', rootCall.name) : invariant(false) : undefined;
    var rootArgString = printArgument(rootArgumentName, rootCall.value, node.getCallType());
    if (rootArgString) {
      rootFieldString += '(' + rootArgString + ')';
    }
  }

  return 'query ' + node.getName() + '{' + rootFieldString + printChildren(node, fragmentMap) + '}';
}

function printMutation(node, fragmentMap) {
  var inputName = node.getCallVariableName();
  var call = '(' + inputName + ':$' + inputName + ')';
  return 'mutation ' + node.getName() + '($' + inputName + ':' + node.getInputType() + ')' + '{' + node.getCall().name + call + printChildren(node, fragmentMap) + '}';
}

function printFragment(node, fragmentMap) {
  return 'fragment ' + node.getDebugName() + ' on ' + node.getType() + printChildren(node, fragmentMap);
}

function printInlineFragment(node, fragmentMap) {
  var fragmentID = node.getFragmentID();
  if (!(fragmentID in fragmentMap)) {
    fragmentMap[fragmentID] = 'fragment ' + fragmentID + ' on ' + node.getType() + printChildren(node, fragmentMap);
  }
  return '...' + fragmentID;
}

function printField(node, fragmentMap) {
  !(node instanceof RelayQuery.Field) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'printRelayQuery(): Query must be flattened before printing.') : invariant(false) : undefined;
  var schemaName = node.getSchemaName();
  var serializationKey = node.getSerializationKey();
  var callsWithValues = node.getCallsWithValues();
  var fieldString = schemaName;
  var argStrings = null;
  if (callsWithValues.length) {
    callsWithValues.forEach(function (_ref) {
      var name = _ref.name;
      var value = _ref.value;

      var argString = printArgument(name, value, node.getCallType(name));
      if (argString) {
        argStrings = argStrings || [];
        argStrings.push(argString);
      }
    });
    if (argStrings) {
      fieldString += '(' + argStrings.join(',') + ')';
    }
  }
  return (serializationKey !== schemaName ? serializationKey + ':' : '') + fieldString + printChildren(node, fragmentMap);
}

function printChildren(node, fragmentMap) {
  var children = node.getChildren().map(function (node) {
    if (node instanceof RelayQuery.Field) {
      return printField(node, fragmentMap);
    } else {
      !(node instanceof RelayQuery.Fragment) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'printRelayQuery(): expected child node to be a `Field` or ' + '`Fragment`, got `%s`.', node.constructor.name) : invariant(false) : undefined;
      return printInlineFragment(node, fragmentMap);
    }
  });
  if (!children.length) {
    return '';
  }
  return '{' + children.join(',') + '}';
}

function printArgument(name, value, type) {
  var stringValue;
  if (value == null) {
    return value;
  }
  if (type === 'enum') {
    !(typeof value === 'string') ? process.env.NODE_ENV !== 'production' ? invariant(false, 'RelayQuery: Expected enum argument `%s` to be a string, got `%s`.', name, value) : invariant(false) : undefined;
    stringValue = value;
  } else if (type === 'object') {
    !(typeof value === 'object' && !Array.isArray(value) && value !== null) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'RelayQuery: Expected object argument `%s` to be an object, got `%s`.', name, value) : invariant(false) : undefined;
    stringValue = stringifyInputObject(name, value);
  } else {
    stringValue = JSON.stringify(value);
  }
  return name + ':' + stringValue;
}

function stringifyInputObject(name, value) {
  !(value != null) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'RelayQuery: Expected input object `%s` to have non-null values.', name) : invariant(false) : undefined;
  if (typeof value !== 'object') {
    return JSON.stringify(value);
  }
  if (Array.isArray(value)) {
    return '[' + value.map(stringifyInputObject.bind(null, name)).join(',') + ']';
  }
  // Reassign to preserve Flow type refinement within closure.
  var objectValue = value; // non-null object
  return '{' + _Object$keys(objectValue).map(function (key) {
    return key + ':' + stringifyInputObject(name, objectValue[key]);
  }).join(',') + '}';
}

module.exports = RelayProfiler.instrument('printRelayQuery', printRelayQuery);