/**
 * Copyright 2013-2015, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule RelayMetricsRecorder
 * @typechecks
 * 
 */

'use strict';

var _classCallCheck = require('babel-runtime/helpers/class-call-check')['default'];

var _extends = require('babel-runtime/helpers/extends')['default'];

var _Object$keys = require('babel-runtime/core-js/object/keys')['default'];

var GraphQLStoreChangeEmitter = require('./GraphQLStoreChangeEmitter');
var GraphQLStoreQueryResolver = require('./GraphQLStoreQueryResolver');
var RelayProfiler = require('./RelayProfiler');

var checkRelayQueryData = require('./checkRelayQueryData');
var diffRelayQuery = require('./diffRelayQuery');
var flattenRelayQuery = require('./flattenRelayQuery');
var invariant = require('fbjs/lib/invariant');
var getRelayQueries = require('./getRelayQueries');
var performanceNow = require('fbjs/lib/performanceNow');
var printRelayQuery = require('./printRelayQuery');
var readRelayQueryData = require('./readRelayQueryData');
var splitDeferredRelayQueries = require('./splitDeferredRelayQueries');
var subtractRelayQuery = require('./subtractRelayQuery');
var writeRelayQueryPayload = require('./writeRelayQueryPayload');
var writeRelayUpdatePayload = require('./writeRelayUpdatePayload');

// Singleton methods:
var INSTRUMENTED_METHODS = [GraphQLStoreChangeEmitter.addListenerForIDs, GraphQLStoreChangeEmitter.broadcastChangeForID, GraphQLStoreChangeEmitter._processSubscribers, GraphQLStoreQueryResolver.prototype.resolve, checkRelayQueryData, diffRelayQuery, flattenRelayQuery, getRelayQueries, printRelayQuery, readRelayQueryData, splitDeferredRelayQueries, subtractRelayQuery, writeRelayQueryPayload, writeRelayUpdatePayload];

// Non-singleton (e.g. `prototype`) methods:
var INSTRUMENTED_AGGREGATE_METHODS = ['RelayContainer.prototype.componentWillMount', 'RelayContainer.prototype.componentWillReceiveProps', 'RelayContainer.prototype.shouldComponentUpdate'];

// Runtime "profiles" registered with `RelayProfiler.profile()`:
var INSTRUMENTED_PROFILES = ['RelayContainer.handleDeferredFailure', 'RelayContainer.handleDeferredSuccess', 'RelayContainer.handleFragmentDataUpdate', 'RelayContainer.update'];

var measurementDefaults = {
  aggregateTime: 0,
  callCount: 0
};

/**
 * Collects timing information from key Relay subsystems. For metrics on all
 * functions, call `RelayProfiler.setEnableProfile(true)` on app initialization.
 *
 * Example:
 *
 * ```
 * var recorder = new RelayMetricsRecorder();
 * recorder.start();
 * // ... do work ...
 * recorder.stop();
 * var metrics = recorder.getMetrics();
 * ```
 *
 * Metrics:
 * - `fetchTime`: the total time spent fetching data from the network.
 * - `recordingTime`: the total time spent recording (between calls to `start()`
 *   and `stop()`).
 * - `totalTime`: the total time spent inside profiled Relay functions.
 * - `measurements`: an object mapping names of profiled functions to profiling
 *   data including:
 *   - `aggregateTime`: total time spent in the method.
 *   - `callCount`: number of times the method was called.
 */

var RelayMetricsRecorder = (function () {
  function RelayMetricsRecorder() {
    _classCallCheck(this, RelayMetricsRecorder);

    this._fetchTime = 0;
    this._isEnabled = false;
    this._measurements = {};
    this._profileStack = [];
    this._recordingStartTime = 0;
    this._recordingTotalTime = 0;
    this._startTimesStack = [];

    this._measure = this._measure.bind(this);
    this._profileFetchRelayQuery = this._profileFetchRelayQuery.bind(this);
    this._profileInstrumentedMethod = this._profileInstrumentedMethod.bind(this);
    this._startMeasurement = this._startMeasurement.bind(this);
    this._stopMeasurement = this._stopMeasurement.bind(this);
  }

  RelayMetricsRecorder.prototype.start = function start() {
    var _this = this;

    !(process.env.NODE_ENV !== 'production') ? process.env.NODE_ENV !== 'production' ? invariant(false, 'RelayMetricsRecorder: Recording metrics requires `__DEV__`.') : invariant(false) : undefined;
    if (this._isEnabled) {
      return;
    }
    this._recordingStartTime = performanceNow();
    this._isEnabled = true;
    this._profileStack = [0];
    this._startTimesStack = [0];

    INSTRUMENTED_METHODS.forEach(function (method) {
      !(method && method.attachHandler) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'RelayMetricsRecorder: Attempted to measure an invalid method.') : invariant(false) : undefined;
      method.attachHandler(_this._measure);
    });
    INSTRUMENTED_AGGREGATE_METHODS.forEach(function (name) {
      RelayProfiler.attachAggregateHandler(name, _this._measure);
    });
    INSTRUMENTED_PROFILES.forEach(function (name) {
      RelayProfiler.attachProfileHandler(name, _this._profileInstrumentedMethod);
    });

    RelayProfiler.attachProfileHandler('fetchRelayQuery', this._profileFetchRelayQuery);
  };

  RelayMetricsRecorder.prototype.stop = function stop() {
    var _this2 = this;

    if (!this._isEnabled) {
      return;
    }
    this._isEnabled = false;
    this._recordingTotalTime += performanceNow() - this._recordingStartTime;

    INSTRUMENTED_METHODS.forEach(function (method) {
      method.detachHandler(_this2._measure);
    });
    INSTRUMENTED_AGGREGATE_METHODS.forEach(function (name) {
      RelayProfiler.detachAggregateHandler(name, _this2._measure);
    });
    INSTRUMENTED_PROFILES.forEach(function (name) {
      RelayProfiler.detachProfileHandler(name, _this2._profileInstrumentedMethod);
    });

    RelayProfiler.detachProfileHandler('fetchRelayQuery', this._profileFetchRelayQuery);
  };

  RelayMetricsRecorder.prototype.getMetrics = function getMetrics() {
    var _measurements = this._measurements;

    var totalTime = 0;
    var sortedMeasurements = {};
    _Object$keys(_measurements).sort(function (a, b) {
      return _measurements[b].aggregateTime - _measurements[a].aggregateTime;
    }).forEach(function (name) {
      totalTime += _measurements[name].aggregateTime;
      sortedMeasurements[name] = _measurements[name];
    });

    return {
      fetchTime: this._fetchTime,
      measurements: sortedMeasurements,
      recordingTime: this._recordingTotalTime,
      totalTime: totalTime
    };
  };

  RelayMetricsRecorder.prototype._measure = function _measure(name, callback) {
    this._startMeasurement(name);
    callback();
    this._stopMeasurement(name);
  };

  RelayMetricsRecorder.prototype._profileFetchRelayQuery = function _profileFetchRelayQuery() {
    var _this3 = this;

    var startTime = performanceNow();
    return function () {
      _this3._fetchTime += performanceNow() - startTime;
    };
  };

  RelayMetricsRecorder.prototype._profileInstrumentedMethod = function _profileInstrumentedMethod(name) {
    var _this4 = this;

    this._startMeasurement(name);
    return function () {
      return _this4._stopMeasurement(name);
    };
  };

  RelayMetricsRecorder.prototype._startMeasurement = function _startMeasurement(name) {
    this._measurements[name] = this._measurements[name] || _extends({}, measurementDefaults);
    this._profileStack.unshift(0);
    this._startTimesStack.unshift(performanceNow());
  };

  RelayMetricsRecorder.prototype._stopMeasurement = function _stopMeasurement(name) {
    var innerTime = this._profileStack.shift();
    var start = this._startTimesStack.shift();
    var totalTime = performanceNow() - start;

    this._measurements[name].aggregateTime += totalTime - innerTime;
    this._measurements[name].callCount++;

    this._profileStack[0] += totalTime;
  };

  return RelayMetricsRecorder;
})();

module.exports = RelayMetricsRecorder;