/**
 * Copyright 2013-2015, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule GraphQLDeferredQueryTracker
 * @typechecks
 * 
 */

'use strict';

var _classCallCheck = require('babel-runtime/helpers/class-call-check')['default'];

var _Object$assign = require('babel-runtime/core-js/object/assign')['default'];

var ErrorUtils = require('fbjs/lib/ErrorUtils');
var Map = require('fbjs/lib/Map');

var forEachObject = require('fbjs/lib/forEachObject');
var forEachRootCallArg = require('./forEachRootCallArg');
var invariant = require('fbjs/lib/invariant');
var isEmpty = require('fbjs/lib/isEmpty');
var resolveImmediate = require('fbjs/lib/resolveImmediate');

/**
 * This module tracks pending queries and maintains information about which
 * deferred data is pending or resolved. It also provides a method to observe
 * when a deferred query for a given node either resolves or fails.
 *
 * @internal
 */

var GraphQLDeferredQueryTracker = (function () {
  function GraphQLDeferredQueryTracker(recordStore) {
    _classCallCheck(this, GraphQLDeferredQueryTracker);

    this.reset();
    this._recordStore = recordStore;
  }

  /**
   * Helper to extract the JSONPath value(s) of a query from a response.
   */

  /**
   * Add a listener for when the given fragment resolves/fails for dataID.
   * Returns a subscription object {remove} where calling remove cancels the
   * subscription.
   */

  GraphQLDeferredQueryTracker.prototype.addListenerForFragment = function addListenerForFragment(dataID, fragmentID, callbacks) {
    var _this = this;

    var subscriber = {
      callbacks: callbacks,
      dataID: dataID,
      fragmentID: fragmentID
    };
    this._subscribers.push(subscriber);
    return {
      remove: function remove() {
        var index = _this._subscribers.indexOf(subscriber);
        !(index >= 0) ? process.env.NODE_ENV !== 'production' ? invariant(false, 'remove() can only be called once') : invariant(false) : undefined;
        _this._subscribers[index] = null;
      }
    };
  };

  /**
   * Record the query as being sent, updating internal tracking to note
   * that the dataID/fragment pairs are pending.
   */

  GraphQLDeferredQueryTracker.prototype.recordQuery = function recordQuery(query) {
    var _this2 = this;

    var parentID = getQueryParentID(query);
    if (parentID) {
      // child query: record parent => [children] list
      var children = this._parentToChildQueryMap.get(parentID) || [];
      children.push(query);
      this._parentToChildQueryMap.set(parentID, children);
    } else {
      var deferredFragmentNames = query.getDeferredFragmentNames();
      if (deferredFragmentNames) {
        // deferred query: record ID => fragment set
        var dataIDs = this._getRootCallToIDMap(query);
        forEachObject(dataIDs, function (dataID, storageKey) {
          if (dataID) {
            var dataIDSet = _this2._dataIDToFragmentNameMap.get(dataID) || {};
            _Object$assign(dataIDSet, deferredFragmentNames); // set union
            _this2._dataIDToFragmentNameMap.set(dataID, dataIDSet);
          } else {
            var rootCallSet = _this2._rootCallToFragmentNameMap.get(storageKey) || {};
            _Object$assign(rootCallSet, deferredFragmentNames);
            _this2._rootCallToFragmentNameMap.set(storageKey, rootCallSet);
          }
        });
      }
    }
  };

  /**
   * Record the query as being resolved with the given data, updating
   * internal tracking and firing subscriptions.
   */

  GraphQLDeferredQueryTracker.prototype.resolveQuery = function resolveQuery(query, response, refParams) {
    var _this3 = this;

    var parentID = getQueryParentID(query);
    this._resolveFragmentsForRootCall(query);
    if (query.isDeferred()) {
      this._resolveDeferredQuery(query, function (dataID, fragmentID) {
        _this3._broadcastChangeForFragment(dataID, fragmentID);
      }, refParams);
      if (parentID) {
        this._resolveDeferredRefQuery(query);
      }
    } else if (response) {
      this._resolveDeferredParentQuery(query, response);
    }
  };

  /**
   * Record that the query has resolved with an error.
   */

  GraphQLDeferredQueryTracker.prototype.rejectQuery = function rejectQuery(query, error) {
    var _this4 = this;

    var parentID = getQueryParentID(query);
    if (query.isDeferred()) {
      this._rejectDeferredFragmentsForRootCall(query);
      this._resolveDeferredQuery(query, function (dataID, fragmentID) {
        _this4._broadcastErrorForFragment(dataID, fragmentID, error);
      });
      if (parentID) {
        this._resolveDeferredRefQuery(query);
      }
    } else {
      this._rejectDeferredParentQuery(query);
    }
  };

  /**
   * Determine if the given query is pending by checking if it is fetching
   * the same dataID/fragments as any pending queries.
   */

  GraphQLDeferredQueryTracker.prototype.isQueryPending = function isQueryPending(dataID, fragmentID) {
    if (this._dataIDToFragmentNameMap.has(dataID)) {
      var dataIDSet = this._dataIDToFragmentNameMap.get(dataID);
      if (dataIDSet.hasOwnProperty(fragmentID)) {
        return true;
      }
    }

    return false;
  };

  /**
   * Clear all query tracking and subscriptions.
   */

  GraphQLDeferredQueryTracker.prototype.reset = function reset() {
    this._dataIDToFragmentNameMap = new Map();
    this._parentToChildQueryMap = new Map();
    this._rootCallToFragmentNameMap = new Map();
    this._subscribers = [];
    this._broadcastItems = null;
  };

  /**
   * Clears all pending dataID => fragmentID associations for this query
   * and calls the callback for each (dataID, fragmentID) pair.
   */

  GraphQLDeferredQueryTracker.prototype._resolveDeferredQuery = function _resolveDeferredQuery(query, callback, refParams) {
    var _this5 = this;

    var deferredFragmentNames = query.getDeferredFragmentNames();
    if (!deferredFragmentNames) {
      return;
    }
    var dataIDs = {};
    var batchCall = query.getBatchCall();
    if (batchCall) {
      // refParams can be undefined if the node is null in the parent query.
      var refIDs = refParams && refParams[batchCall.refParamName];
      if (refIDs != null) {
        refIDs = Array.isArray(refIDs) ? refIDs : [refIDs];
        refIDs.forEach(function (id) {
          return dataIDs[id] = id;
        });
      }
    } else {
      dataIDs = this._getRootCallToIDMap(query);
    }
    forEachObject(dataIDs, function (dataID) {
      if (dataID && _this5._dataIDToFragmentNameMap.has(dataID)) {
        var dataIDSet = _this5._dataIDToFragmentNameMap.get(dataID);
        forEachObject(deferredFragmentNames, function (fragmentID) {
          delete dataIDSet[fragmentID];
          callback(dataID, fragmentID);
        });
        if (!isEmpty(dataIDSet)) {
          _this5._dataIDToFragmentNameMap.set(dataID, dataIDSet);
        } else {
          _this5._dataIDToFragmentNameMap['delete'](dataID);
        }
      }
    });
  };

  /**
   * Clears the deferred query from its parent's list of dependent queries.
   */

  GraphQLDeferredQueryTracker.prototype._resolveDeferredRefQuery = function _resolveDeferredRefQuery(query) {
    var parentID = getQueryParentID(query);
    var children = this._parentToChildQueryMap.get(parentID) || [];
    children = children.filter(function (q) {
      return q !== query;
    });
    if (children.length) {
      this._parentToChildQueryMap.set(parentID, children);
    } else {
      this._parentToChildQueryMap['delete'](parentID);
    }
  };

  /**
   * Resolves the root IDs for any dependent queries of the given query.
   */

  GraphQLDeferredQueryTracker.prototype._resolveDeferredParentQuery = function _resolveDeferredParentQuery(query, response) {
    var _this6 = this;

    // resolve IDs in child queries, add to ID => fragment set
    var children = this._parentToChildQueryMap.get(query.getID()) || [];
    for (var ii = 0; ii < children.length; ii++) {
      var childQuery = children[ii];
      var childFragmentNames = childQuery.getDeferredFragmentNames();
      var childDataIDs = getRefParamFromResponse(response, childQuery);
      forEachObject(childDataIDs, function (dataID) {
        var dataIDSet = _this6._dataIDToFragmentNameMap.get(dataID) || {};
        _Object$assign(dataIDSet, childFragmentNames);
        _this6._dataIDToFragmentNameMap.set(dataID, dataIDSet);
      });
    }
  };

  /**
   * Maps the deferred fragments for a root call with a previously unknown ID to
   * the resolved ID value.
   */

  GraphQLDeferredQueryTracker.prototype._resolveFragmentsForRootCall = function _resolveFragmentsForRootCall(query) {
    var _this7 = this;

    var rootCallMap = this._getRootCallToIDMap(query);
    forEachObject(rootCallMap, function (dataID, storageKey) {
      if (dataID && _this7._rootCallToFragmentNameMap.has(storageKey)) {
        var rootCallSet = _this7._rootCallToFragmentNameMap.get(storageKey) || {};
        var dataIDSet = _this7._dataIDToFragmentNameMap.get(dataID) || {};
        _Object$assign(dataIDSet, rootCallSet);
        _this7._dataIDToFragmentNameMap.set(dataID, dataIDSet);
        _this7._rootCallToFragmentNameMap['delete'](storageKey);
      }
    });
  };

  /**
   * Removes the deferred fragments for a previously unresolved root call ID.
   */

  GraphQLDeferredQueryTracker.prototype._rejectDeferredFragmentsForRootCall = function _rejectDeferredFragmentsForRootCall(query) {
    var _this8 = this;

    var rootCallMap = this._getRootCallToIDMap(query);
    var deferredFragmentNames = query.getDeferredFragmentNames();
    forEachObject(rootCallMap, function (dataID, storageKey) {
      if (_this8._rootCallToFragmentNameMap.has(storageKey)) {
        var rootCallSet = _this8._rootCallToFragmentNameMap.get(storageKey) || {};
        forEachObject(deferredFragmentNames, function (fragmentID) {
          delete rootCallSet[fragmentID];
        });
        if (!isEmpty(rootCallSet)) {
          _this8._rootCallToFragmentNameMap['delete'](storageKey);
        } else {
          _this8._rootCallToFragmentNameMap.set(storageKey, rootCallSet);
        }
      }
    });
  };

  /**
   * Rejects the parent ID, clearing all tracking for both the parent and all
   * its dependent deferred ref queries.
   */

  GraphQLDeferredQueryTracker.prototype._rejectDeferredParentQuery = function _rejectDeferredParentQuery(query) {
    var parentID = query.getID();
    this._parentToChildQueryMap['delete'](parentID);
  };

  /**
   * Notify observers that the given deferred fragment has resolved for node
   * with dataID.
   */

  GraphQLDeferredQueryTracker.prototype._broadcastChangeForFragment = function _broadcastChangeForFragment(dataID, fragmentID) {
    var _this9 = this;

    var broadcastItems = this._broadcastItems;
    if (!broadcastItems) {
      this._broadcastItems = broadcastItems = [];
      resolveImmediate(function () {
        return _this9._processBroadcasts();
      });
    }
    broadcastItems.push({ dataID: dataID, fragmentID: fragmentID, error: null });
  };

  /**
   * Record that an error occurred for this dataID, fragment pair
   * and broadcast an update.
   */

  GraphQLDeferredQueryTracker.prototype._broadcastErrorForFragment = function _broadcastErrorForFragment(dataID, fragmentID, error) {
    var _this10 = this;

    var broadcastItems = this._broadcastItems;
    if (!broadcastItems) {
      this._broadcastItems = broadcastItems = [];
      resolveImmediate(function () {
        return _this10._processBroadcasts();
      });
    }
    broadcastItems.push({ dataID: dataID, fragmentID: fragmentID, error: error });
  };

  /**
   * Process broadcast items from previous event loop.
   */

  GraphQLDeferredQueryTracker.prototype._processBroadcasts = function _processBroadcasts() {
    if (!this._broadcastItems) {
      return;
    }

    for (var ii = 0; ii < this._subscribers.length; ii++) {
      for (var jj = 0; jj < this._broadcastItems.length; jj++) {
        var subscriber = this._subscribers[ii];
        if (subscriber) {
          var _broadcastItems$jj = this._broadcastItems[jj];
          var dataID = _broadcastItems$jj.dataID;
          var error = _broadcastItems$jj.error;
          var fragmentID = _broadcastItems$jj.fragmentID;

          var method;
          var args;
          if (error) {
            method = subscriber.callbacks.onFailure;
            args = [dataID, fragmentID, error];
          } else {
            method = subscriber.callbacks.onSuccess;
            args = [dataID, fragmentID];
          }
          ErrorUtils.applyWithGuard(method, null, args, null, 'GraphQLDeferredQueryTracker');
        }
      }
    }

    this._subscribers = this._subscribers.filter(function (subscriber) {
      return subscriber !== null;
    });
    this._broadcastItems = null;
  };

  GraphQLDeferredQueryTracker.prototype._getRootCallToIDMap = function _getRootCallToIDMap(query) {
    var _this11 = this;

    var mapping = {};
    if (!query.getBatchCall()) {
      (function () {
        var storageKey = query.getStorageKey();
        forEachRootCallArg(query, function (identifyingArgValue) {
          var compositeStorageKey = identifyingArgValue == null ? storageKey : storageKey + ':' + identifyingArgValue;
          mapping[compositeStorageKey] = _this11._recordStore.getDataID(storageKey, identifyingArgValue);
        });
      })();
    }
    return mapping;
  };

  return GraphQLDeferredQueryTracker;
})();

function getRefParamFromResponse(response, query) {
  var batchCall = query.getBatchCall();
  var refTarget = batchCall ? batchCall.sourceQueryPath : null;
  if (!refTarget) {
    return {};
  }
  var values = {};
  var tokens = refTarget.split('.');

  getRefParamFromNode(values, response, tokens, 1); // skip root '$' marker
  return values;
}

/**
 * Recursive helper to extract the ref parameter (represented as tokens)
 * into the values object.
 */
function getRefParamFromNode(values, node, tokens, index) {
  if (index === tokens.length && typeof node === 'string') {
    // base case
    values[node] = node;
    return;
  } else if (
  // mismatched path/response
  index >= tokens.length || !node || typeof node !== 'object' || Array.isArray(node)) {
    return;
  }
  var token = tokens[index];
  if (token === '*') {
    forEachObject(node, function (subNode) {
      getRefParamFromNode(values, subNode, tokens, index + 1);
    });
  } else if (node.hasOwnProperty(token)) {
    getRefParamFromNode(values, node[token], tokens, index + 1);
  }
}

/**
 * Helper to get a query's sourceQueryID
 */
function getQueryParentID(query) {
  var batchCall = query.getBatchCall();
  if (batchCall) {
    return batchCall.sourceQueryID;
  }
  return null;
}

module.exports = GraphQLDeferredQueryTracker;

/**
 * List of all subscriptions of form {callback, dataID, fragmentID}
 */

/**
 * List of all deferred queries that have resolved/failed since the last
 * broadcast.
 */

/**
 * Map of pending dataID => Set<fragmentID>
 * Stores a set as object<string,string> of all pending deferred fragmentIDs
 * for a given dataID. Presence of dataID => fragmentID pair
 * means that the query is pending, absence that it has resolved.
 */

/**
 * Map of pending rootCall => Set<fragmentID>
 * Stores a temporary mapping of fragmentIDs when the correct dataID is
 * unknown. Entries will get moved to dataIDToFragmentNameMap as the dataID
 * for the rootCall is determinble.
 */

/**
 * Map of parent query ID => [child queries]
 */