'use strict';

var _classCallCheck = require('babel-runtime/helpers/class-call-check')['default'];

Object.defineProperty(exports, '__esModule', {
  value: true
});

var Promise = require('fbjs/lib/Promise');

/**
 * Copyright (c) 2013-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule RelayNetworkDebug
 * @typechecks
 * 
 */

'use strict';

var Relay = require('./RelayPublic');

var performanceNow = require('fbjs/lib/performanceNow');

var RelayNetworkDebugger = (function () {
  function RelayNetworkDebugger(networkLayer) {
    var _this = this;

    _classCallCheck(this, RelayNetworkDebugger);

    this._initTime = performanceNow();
    this._queryID = 0;

    Relay.injectNetworkLayer({
      sendQueries: function sendQueries(requests) {
        requests.forEach(function (request) {
          _this.logRequest(createDebuggableFromRequest('Relay Query', request));
        });
        return networkLayer.sendQueries(requests);
      },
      sendMutation: function sendMutation(request) {
        _this.logRequest(createDebuggableFromRequest('Relay Mutation', request));
        return networkLayer.sendMutation(request);
      },
      supports: function supports() {
        return networkLayer.supports.apply(networkLayer, arguments);
      }
    });

    this._fetch = global.fetch;
    global.fetch = function (url, options) {
      for (var _len = arguments.length, args = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
        args[_key - 2] = arguments[_key];
      }

      var name = url.split('/')[2];
      _this.logRequest(createDebuggableFromFetch(name, { url: url, options: options, args: args }, _this._fetch.apply(_this, [url, options].concat(args))));
    };
  }

  RelayNetworkDebugger.prototype.uninstall = function uninstall() {
    global.fetch = this._fetch;
  };

  RelayNetworkDebugger.prototype.logRequest = function logRequest(_ref) {
    var _this2 = this;

    var name = _ref.name;
    var type = _ref.type;
    var promise = _ref.promise;
    var logResult = _ref.logResult;

    var id = this._queryID++;
    var timerName = '[' + id + '] Request Duration';

    console.timeStamp('START: [' + id + '] ' + type + ': ' + name + ' →');
    console.time(timerName);

    var onSettled = function onSettled(error, response) {
      var time = (performanceNow() - _this2._initTime) / 1000;
      console.timeStamp('← END: [' + id + '] ' + type + ': ' + name);
      console.groupCollapsed('%c[' + id + '] ' + type + ': ' + name + ' @ ' + time + 's', 'color:' + (error ? 'red' : 'black') + ';');
      console.timeEnd(timerName);
      logResult(error, response);
      console.groupEnd();
    };

    promise.then(function (response) {
      return onSettled(null, response);
    }, function (error) {
      return onSettled(error, null);
    });
  };

  return RelayNetworkDebugger;
})();

function createDebuggableFromRequest(type, request) {
  return {
    name: request.getDebugName(),
    type: 'Relay Query',
    promise: request.getPromise(),
    logResult: function logResult(error, response) {
      console.debug('%c%s\n', 'font-size:10px; color:#333; font-family:mplus-2m-regular,menlo,' + 'monospaced;', request.getQueryString());
      error && console.error(error);
      response && console.log(response);
    }
  };
}

function createDebuggableFromFetch(name, config, promise) {
  return {
    name: name,
    type: 'Relay Mutation',
    promise: promise,
    logResult: function logResult(error, response) {
      console.debug(config);
      error && console.error(error);
      response && console.warn(response);
      try {
        response && console.debug(JSON.parse(response._bodyText));
      } catch (_) {}
    }
  };
}

var networkDebugger = undefined;

var RelayNetworkDebug = {
  init: function init(networkLayer) {
    networkDebugger && networkDebugger.uninstall();
    networkDebugger = new RelayNetworkDebugger(networkLayer);
  },
  logRequest: function logRequest(request) {
    networkDebugger && networkDebugger.logRequest(request);
  }
};

module.exports = RelayNetworkDebug;