import { useRef, useState, useEffect, createElement, Fragment } from 'react';

var ImageRendererStatus;

(function (ImageRendererStatus) {
  ImageRendererStatus[ImageRendererStatus["LOADING"] = 0] = "LOADING";
  ImageRendererStatus[ImageRendererStatus["LOADED"] = 1] = "LOADED";
  ImageRendererStatus[ImageRendererStatus["ERRORED"] = 2] = "ERRORED";
})(ImageRendererStatus || (ImageRendererStatus = {}));

var isBrowser = typeof window !== 'undefined';

var isLoaded = function isLoaded(status) {
  return status === ImageRendererStatus.LOADED;
};

var isErrored = function isErrored(status) {
  return status === ImageRendererStatus.ERRORED;
};

var ImageRenderer = function ImageRenderer(_ref) {
  var src = _ref.src,
      onLoad = _ref.onLoad,
      onError = _ref.onError,
      loadingView = _ref.loading,
      loadedView = _ref.loaded,
      erroredView = _ref.errored,
      children = _ref.children;
  var image = useRef(undefined);

  var _React$useState = useState(ImageRendererStatus.LOADING),
      status = _React$useState[0],
      setStatus = _React$useState[1];

  useEffect(function () {
    var unload = function unload() {
      if (image.current) {
        image.current.onload = null;
        image.current.onerror = null;
      }
    };

    if (isBrowser) {
      setStatus(ImageRendererStatus.LOADING);
      image.current = isBrowser ? new window.Image() : undefined;

      if (image.current) {
        image.current.onload = function () {
          unload();
          setStatus(ImageRendererStatus.LOADED);
        };

        image.current.onerror = function () {
          unload();
          setStatus(ImageRendererStatus.ERRORED);
        };

        image.current.src = src;
      }
    }

    return function () {
      return unload();
    };
  }, [src]);
  useEffect(function () {
    if (isLoaded(status) && onLoad) {
      onLoad();
    }

    if (isErrored(status) && onError) {
      onError();
    }
  }, [status, onLoad, onError]);

  if (isLoaded(status) && loadedView) {
    // <>{x}</> is a hack to get around typings issue
    // @see https://github.com/DefinitelyTyped/DefinitelyTyped/pull/23422
    // @see https://github.com/microsoft/TypeScript/issues/21699
    return createElement(Fragment, null, loadedView);
  }

  if (isErrored(status) && erroredView) {
    // <>{x}</> is a hack to get around typings issue
    // @see https://github.com/DefinitelyTyped/DefinitelyTyped/pull/23422
    // @see https://github.com/microsoft/TypeScript/issues/21699
    return createElement(Fragment, null, erroredView);
  }

  if (!isLoaded(status) && !isErrored(status) && loadingView) {
    // <>{x}</> is a hack to get around typings issue
    // @see https://github.com/DefinitelyTyped/DefinitelyTyped/pull/23422
    // @see https://github.com/microsoft/TypeScript/issues/21699
    return createElement(Fragment, null, loadingView);
  }

  if (children) {
    // <>{x}</> is a hack to get around typings issue
    // @see https://github.com/DefinitelyTyped/DefinitelyTyped/pull/23422
    // @see https://github.com/microsoft/TypeScript/issues/21699
    return createElement(Fragment, null, children({
      image: isLoaded(status) ? image.current : undefined,
      loaded: isLoaded(status),
      errored: isErrored(status)
    }));
  }

  return null;
};

export default ImageRenderer;
export { ImageRendererStatus };
//# sourceMappingURL=react-render-image.esm.js.map
