const _excluded = ["children", "scope", "isGlobal"];

function _objectWithoutPropertiesLoose(source, excluded) { if (source == null) return {}; var target = {}; var sourceKeys = Object.keys(source); var key, i; for (i = 0; i < sourceKeys.length; i++) { key = sourceKeys[i]; if (excluded.indexOf(key) >= 0) continue; target[key] = source[key]; } return target; }

import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Context } from '../context';
import { StoreRegistry, bindActions, defaultRegistry } from '../store';
import shallowEqual from '../utils/shallow-equal';

const noop = () => () => {};

export default class Container extends Component {
  static getDerivedStateFromProps(nextProps, prevState) {
    const {
      scope
    } = nextProps;
    const hasScopeChanged = scope !== prevState.scope;
    let nextState = null;

    if (hasScopeChanged) {
      const actions = prevState.bindContainerActions(scope);
      nextState = {
        scope,
        scopedActions: actions
      };
    } // We trigger the action here so subscribers get new values ASAP


    prevState.triggerContainerAction(nextProps);
    return nextState;
  }

  constructor(_props, context) {
    super(_props, context);
    this.registry = new StoreRegistry('__local__');
    this.scopedHooks = {};

    this.bindContainerActions = scope => {
      const {
        storeType,
        hooks
      } = this.constructor;
      const {
        api
      } = this.state; // we explicitly pass scope as it might be changed

      const {
        storeState
      } = api.getStore(storeType, scope);
      const actions = bindActions(storeType.actions, storeState, this.getContainerProps);
      this.scopedHooks = bindActions(hooks, storeState, this.getContainerProps, actions); // make sure we also reset actionProps

      this.actionProps = null;
      return actions;
    };

    this.triggerContainerAction = nextProps => {
      const nextActionProps = this.filterActionProps(nextProps);
      const prevActionProps = this.actionProps;
      if (shallowEqual(prevActionProps, nextActionProps)) return; // store restProps on instance so we can diff and use fresh props
      // in actions even before react sets them in this.props

      this.actionProps = nextActionProps;

      if (this.scopedHooks.onInit) {
        this.scopedHooks.onInit();
        this.scopedHooks.onInit = null;
      } else {
        this.scopedHooks.onUpdate();
      }
    };

    this.filterActionProps = props => {
      // eslint-disable-next-line no-unused-vars
      const restProps = _objectWithoutPropertiesLoose(props, _excluded);

      return restProps;
    };

    this.getContainerProps = () => this.actionProps;

    const {
      // These fallbacks are needed only to make enzyme shallow work
      // as it does not fully support provider-less Context enzyme#1553
      globalRegistry = defaultRegistry,
      getStore: _getStore = defaultRegistry.getStore
    } = this.context;
    this.state = {
      api: {
        globalRegistry,
        getStore: (Store, scope) => this.getScopedStore(Store, scope) || _getStore(Store)
      },
      // stored to make them available in getDerivedStateFromProps
      // as js context there is null https://github.com/facebook/react/issues/12612
      bindContainerActions: this.bindContainerActions,
      triggerContainerAction: this.triggerContainerAction,
      scope: _props.scope
    };
    this.state.scopedActions = this.bindContainerActions(_props.scope);
  }

  componentDidUpdate(prevProps) {
    if (this.props.scope !== prevProps.scope) {
      // Trigger a forced update on all subscribers
      // as render might have been blocked
      this.triggerScopeChange(prevProps.scope); // Check if instance has still subscribers, if not delete

      this.deleteScopedStore(prevProps.scope);
    }
  }

  componentWillUnmount() {
    // schedule on next tick as this is called by React before useEffect cleanup
    // so if we run immediately listeners will still be there and run
    Promise.resolve().then(() => {
      this.scopedHooks.onCleanup(); // Check if scope has still subscribers, if not delete

      this.deleteScopedStore();
    });
  }

  getRegistry() {
    const isLocal = !this.props.scope && !this.props.isGlobal;
    return isLocal ? this.registry : this.state.api.globalRegistry;
  }

  getScopedStore(Store, scopeId) {
    if (scopeId === void 0) {
      scopeId = this.props.scope;
    }

    const {
      storeType
    } = this.constructor;

    if (Store !== storeType) {
      return null;
    }

    const {
      storeState
    } = this.getRegistry().getStore(Store, scopeId); // instead of returning global bound actions
    // we return the ones with the countainer props binding

    return {
      storeState,
      actions: this.state.scopedActions
    };
  }

  triggerScopeChange(prevScopeId) {
    const {
      storeType
    } = this.constructor;
    const previous = this.getScopedStore(storeType, prevScopeId); // When called, subscribers that have already re-rendered with the new
    // scope are no longer subscribed to the old one, so we "force update"
    // the remaining.
    // This is sub-optimal as if there are other containers with the same
    // old scope id we will re-render those too, but better than using context
    // as that will re-render all children even if pure/memo

    previous.storeState.notify();
  }

  deleteScopedStore(scopeId) {
    if (scopeId === void 0) {
      scopeId = this.props.scope;
    }

    const {
      storeType
    } = this.constructor;
    const {
      storeState
    } = this.getScopedStore(storeType, scopeId);

    if (scopeId != null && !storeState.listeners().length) {
      this.getRegistry().deleteStore(storeType, scopeId);
    }
  }

  render() {
    const {
      children
    } = this.props;
    return /*#__PURE__*/React.createElement(Context.Provider, {
      value: this.state.api
    }, children);
  }

}
Container.propTypes = {
  children: PropTypes.node,
  scope: PropTypes.string,
  isGlobal: PropTypes.bool
};
Container.storeType = null;
Container.hooks = null;
Container.contextType = Context;
export function createContainer(Store, _temp) {
  var _class;

  let {
    onInit = noop,
    onUpdate = noop,
    onCleanup = noop,
    displayName = ''
  } = _temp === void 0 ? {} : _temp;
  return _class = class extends Container {}, _class.storeType = Store, _class.displayName = displayName || `Container(${Store.key.split('__')[0]})`, _class.hooks = {
    onInit,
    onUpdate,
    onCleanup
  }, _class;
}