import Subscriber from '../Subscriber';
import Subject from '../Subject';
import tryCatch from '../util/tryCatch';
import { errorObject } from '../util/errorObject';
export default function retryWhen(notifier) {
    return this.lift(new RetryWhenOperator(notifier, this));
}
class RetryWhenOperator {
    constructor(notifier, source) {
        this.notifier = notifier;
        this.source = source;
    }
    call(subscriber) {
        return new FirstRetryWhenSubscriber(subscriber, this.notifier, this.source);
    }
}
class FirstRetryWhenSubscriber extends Subscriber {
    constructor(destination, notifier, source) {
        super(null);
        this.destination = destination;
        this.notifier = notifier;
        this.source = source;
        this.lastSubscription = this;
    }
    _next(value) {
        this.destination.next(value);
    }
    error(err) {
        if (!this.isUnsubscribed) {
            super.unsubscribe();
            if (!this.retryNotifications) {
                this.errors = new Subject();
                const notifications = tryCatch(this.notifier).call(this, this.errors);
                if (notifications === errorObject) {
                    this.destination.error(errorObject.e);
                }
                else {
                    this.retryNotifications = notifications;
                    const notificationSubscriber = new RetryNotificationSubscriber(this);
                    this.notificationSubscription = notifications.subscribe(notificationSubscriber);
                }
            }
            this.errors.next(err);
        }
    }
    destinationError(err) {
        this.tearDown();
        this.destination.error(err);
    }
    _complete() {
        this.destinationComplete();
    }
    destinationComplete() {
        this.tearDown();
        this.destination.complete();
    }
    unsubscribe() {
        const lastSubscription = this.lastSubscription;
        if (lastSubscription === this) {
            super.unsubscribe();
        }
        else {
            this.tearDown();
        }
    }
    tearDown() {
        super.unsubscribe();
        this.lastSubscription.unsubscribe();
        const notificationSubscription = this.notificationSubscription;
        if (notificationSubscription) {
            notificationSubscription.unsubscribe();
        }
    }
    resubscribe() {
        this.lastSubscription.unsubscribe();
        const nextSubscriber = new MoreRetryWhenSubscriber(this);
        this.lastSubscription = this.source.subscribe(nextSubscriber);
    }
}
class MoreRetryWhenSubscriber extends Subscriber {
    constructor(parent) {
        super(null);
        this.parent = parent;
    }
    _next(value) {
        this.parent.destination.next(value);
    }
    _error(err) {
        this.parent.errors.next(err);
    }
    _complete() {
        this.parent.destinationComplete();
    }
}
class RetryNotificationSubscriber extends Subscriber {
    constructor(parent) {
        super(null);
        this.parent = parent;
    }
    _next(value) {
        this.parent.resubscribe();
    }
    _error(err) {
        this.parent.destinationError(err);
    }
    _complete() {
        this.parent.destinationComplete();
    }
}
//# sourceMappingURL=retryWhen.js.map